// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
#ifndef PACKAGE_NAMESPACE_CHILDRENHOLDER_H_INCLUDED
#define PACKAGE_NAMESPACE_CHILDRENHOLDER_H_INCLUDED 1

#include <string>
#include <list>
#include <map>
#include <vector>
//#include <deque> // causes crashes in some cases where vector does not :/

namespace PACKAGE_NAMESPACE
{
	/**
           EXPERIMENTAL!

           children_holder is an experimental object for creating pointer
           containers for arbitrary child types. The idea is that many objects
           i create have a heirarchical structure, and i'm looking for a
           generic way to code this pattern.

           The interface should allow any given class, including
           non-modifiable 3rd-party classes, to have any number of
           child types.

           This code does no automatic clean-up of children. Client
           code may free all child pointers by calling
           cleanup_parent(), and subclasses or proxies of this object
           should call that in their dtor.
        */
	template < class PType, class CType > class children_holder
	{
	      public:
                /**
                   The type of this object's parents in
                   parent-child relationships.
                 */
		typedef PType parent_type;

                /**
                   The type of this object's children in
                   parent-child relationships.
                 */
		typedef CType child_type;

                /**
                   The container type used to store the lists of children.
                 */
		typedef std::list < child_type * >list_type;

                /**
                   A shortcut typedef to help ease the implementation code for this class.
                 */
		typedef children_holder < parent_type, child_type > ThisType;

                /**
                   iterator which can be dereferenced to a (child_type *).
                 */
		typedef typename list_type::iterator iterator;
                /**
                   iterator which can be dereferenced to a (const child_type *).
                 */
		typedef typename list_type::const_iterator const_iterator;

		/**
                   Returns the child list for the given parent
                   object. If creationPolicy is non-zero then this
                   function will create the child list if it does not
                   yet exist (in that case, this function will never
                   return NULL except on an out-of-memory error).

                   The caller takes ownership of the returned
                   pointer. All children in the list can be deleted at
                   once by calling cleanup_parent( parent ).

                   Different calls to this function will always return
                   the same list (or the same NULL, depending on
                   creationPolicy ;) until either unmap_parent() or
                   cleanup_parent() are called, in which case further
                   calls to this function may return a different
                   pointer the next time it is called.
                */
		static ThisType::list_type * child_list( const ThisType::parent_type * parent, int creationPolicy = 0 )
		{
			if ( !parent ) return NULL;
			static ThisType::map_type & cmap = parentChildMap();
			typename map_type::const_iterator it = cmap.find( parent );
			if ( cmap.end() != it ) return ( *it ).second;
			if ( 0 == creationPolicy ) return NULL;
			list_type *cl = new list_type();
                        cmap[parent] = cl;
                        return cl;
		}


		/**
                   Removes parent from this object's internal
                   map. This only removes the pointer to the parent's
                   child list and the mapping which binds the parent
                   to the children, but does not delete() anything.

                   Returns true if it unmaps the parent, else
                   false. It will only fail if it doesn't have an
                   entry for parent.


                   This is more of a maintenance detail than anything
                   else.

                */
		static bool unmap_parent( const ThisType::parent_type * parent )
		{
			if ( !parent ) return false;
			static ThisType::map_type & cmap = parentChildMap();
			typename ThisType::map_type::iterator it = cmap.find( parent );
			if ( it == cmap.end() ) return false;
			cmap.erase( parent );
			return true;
		}

		/**
                   Simlar as unmap_parent(), but also deletes the
                   children in the list and then deletes the list.

                   Subclasses or proxies of this class should call
                   this function from their dtor, in order to avoid
                   leaving any dangling pointers in this class'
                   internal data.
                */
		static bool cleanup_parent( const ThisType::parent_type * parent )
		{
			if ( !parent )
				return false;
			static ThisType::map_type & cmap = parentChildMap();
			typename ThisType::map_type::iterator it = cmap.find( parent );
			if ( it == cmap.end() )
			{
				// we were probably just never registed because children() was never called.
				return false;
			}
			typename ThisType::list_type * li = ( *it ).second;
			if ( !unmap_parent( parent ) )
			{
				return false;
			}
			typename ThisType::list_type::iterator vit;
			typename ThisType::child_type * child = 0;
			for ( vit = li->begin(); li->begin() != li->end(); )
			{
                                // note: i explicitely call begin()/end() on each iteration
                                // because i'm not certain of iterator invalidation here :/
				child = ( *vit );
				li->erase( vit );
				delete( child );
				child = 0;
			}
			delete( li );
			return true;
		}


	      private:
		typedef std::map < const ThisType::parent_type *, ThisType::list_type * > map_type;
                /**
                   i hate working with static objects in template
                   classes (for syntax reasons), and parentChildMap()
                   is a helper to avoid that.
                 */
		static map_type & parentChildMap()
		{
			static map_type meyers;
			return meyers;
		}


	};


};				//  namespace PACKAGE_NAMESPACE
#endif // PACKAGE_NAMESPACE_CHILDRENHOLDER_H_INCLUDED
