/*
 * Copyright (c) 2006 Advanced Micro Devices, Inc.
 * Contributed by Ray Bryant <raybry@mpdtxmail.amd.com> 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 */

/* History
 *
 * Feb 10 2006 -- Ray Bryant, raybry@mpdtxmail.amd.com
 *
 * Brought event table up-to-date with the 3.85 (October 2005) version of the
 * "BIOS and Kernel Developer's Guide for the AMD Athlon[tm] 64 and
 * AMD Opteron[tm] Processors," AMD Publication # 26094.
 * 
 */

static pme_amd64_entry_t amd64_pe[]={
/* 0 */{.pme_name    = "DISPATCHED_FPU",
	.pme_code    = 0x00,
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_desc    = "Dispatched FPU Operations",
	.pme_numasks = 6,
	.pme_umasks = {
		{ .pme_uname = "OPS_ADD",
		  .pme_udesc = "Add pipe ops",
		  .pme_ucode = 0x01
		},
		{ .pme_uname = "OPS_MULTIPLY",
		  .pme_udesc = "Multiply pipe ops",
		  .pme_ucode = 0x02
		},
		{ .pme_uname = "OPS_STORE",
		  .pme_udesc = "Store pipe ops",
		  .pme_ucode = 0x04
		},
		{ .pme_uname = "OPS_ADD_PIPE_LOAD_OPS",
		  .pme_udesc = "Add pipe load ops",
		  .pme_ucode = 0x08
		},
		{ .pme_uname = "OPS_MULTIPY_PIPE_LOAD_OPS",
		  .pme_udesc = "Multiply pipe load ops",
		  .pme_ucode = 0x10
		},
		{ .pme_uname = "OPS_STORE_PIPE_LOAD_OPS",
		  .pme_udesc = "Store pipe load ops",
		  .pme_ucode = 0x20
		}
	}
       },
/* 1 */{.pme_name = "CYCLES_NO_FPU_OPS_RETIRED",
	.pme_code = 0x01,
	.pme_desc = "Cycles with no FPU Ops Retired"
	},
/* 2 */{.pme_name = "DISPATCHED_FPU_OPS_FAST_FLAG",
	.pme_code = 0x02,
	.pme_desc = "Dispatched Fast Flag FPU Operations"
	},
/* 3 */{.pme_name    = "SEGMENT_REGISTER_LOADS",
	.pme_code    = 0x20,
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_desc    = "Segment Register Loads",
	.pme_numasks = 8,
	.pme_umasks  = {
		{ .pme_uname = "ES",
		  .pme_udesc = "ES",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "CS",
		  .pme_udesc = "CS",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "SS",
		  .pme_udesc = "SS",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "DS",
		  .pme_udesc = "DS",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "FS",
		  .pme_udesc = "FS",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "GS",
		  .pme_udesc = "GS",
		  .pme_ucode = 0x20,
		},
		{ .pme_uname = "HS",
		  .pme_udesc = "HS",
		  .pme_ucode = 0x40,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "All segments",
		  .pme_ucode = 0x7f,
		},
	},
       },
/* 4  */{.pme_name = "PIPELINE_RESTART_DUE_TO_SELF_MODIFYING_CODE",
	.pme_code = 0x21,
	.pme_desc = "Pipeline Restart Due to Self_Modifying Code"
	},
/* 5  */{.pme_name = "PIPELINE_RESTART_DUE_TO_PROBE_HIT",
	.pme_code = 0x22,
	.pme_desc = "Pipeline Restart Due to Probe Hit"
	},
/* 6  */{.pme_name = "LS_BUFFER_2_FULL_CYCLES",
	.pme_code = 0x23,
	.pme_desc = "Load/Store Buffer 2 Full"
	},
/* 7  */{.pme_name = "LOCKED_OPS",
	.pme_code = 0x24,
	.pme_flags = PFMLIB_AMD64_UMASK_COMBO,
	.pme_desc = "Locked Operations",
	.pme_numasks = 3,
	.pme_umasks = {
		{ .pme_uname = "EXECUTED",
		  .pme_udesc = "The number of locked instructions executed",
		  .pme_ucode = 0x1
		},
		{ .pme_uname = "CYCLES_SPECULATIVE_PHASE",
		  .pme_udesc = " The number of cycles spent in speculative phase",
		  .pme_ucode = 0x2
		},
		{ .pme_uname = "CYCLES_NON_SPECULATIVE_PHASE",
		  .pme_udesc = "The number of cycles spent in non-speculative phase (including cache miss penalty)",
		  .pme_ucode = 0x4
		}
	 }
	},
/* 8  */{.pme_name = "MEMORY_REQUESTS",
	.pme_code = 0x65,
	.pme_desc = "Memory Requests by Type",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 3,
	.pme_umasks  = {
		{ .pme_uname = "NON_CACHEABLE",
		  .pme_udesc = "Requests to non-cacheable (UC) memory",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "WRITE_COMBINING",
		  .pme_udesc = "Requests to write-combining (WC) memory or WC buffer flushes to WB memory",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "STREAMING_STORE",
		  .pme_udesc = "Streaming store (SS) requests",
		  .pme_ucode = 0x80,
		}
	 }
	},
/* 9  */{.pme_name = "DATA_CACHE_ACCESSES",
	.pme_code = 0x40,
	.pme_desc = "Data Cache Accesses"
	},
/* 10 */{.pme_name = "DATA_CACHE_MISSES",
	.pme_code = 0x41,
	.pme_desc = "Data Cache Misses"
	},
/* 11 */{.pme_name = "DATA_CACHE_REFILLS",
	.pme_code = 0x42,
	.pme_desc = "Data Cache Refills from L2 or System",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 6,
	.pme_umasks  = {
		{ .pme_uname = "SYSTEM",
		  .pme_udesc = "Refill from System",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "L2_SHARED",
		  .pme_udesc = "Shared-state line from L2",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "L2_EXCLUSIVE",
		  .pme_udesc = "Exclusive-state line from L2",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "L2_OWNED",
		  .pme_udesc = "Owned-state line from L2",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "L2_MODIFIED",
		  .pme_udesc = "Modified-state line from L2",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "L2_ALL",
		  .pme_udesc = "Shared, Exclusive, Owned, Modified State Refills",
		  .pme_ucode = 0x1e,
		}
	 }
	},
/* 12 */{.pme_name = "DATA_CACHE_REFILLS_FROM_SYSTEM",
	.pme_code = 0x43,
	.pme_desc = "Data Cache Refills from System",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 6,
	.pme_umasks  = {
		{ .pme_uname = "INVALID",
		  .pme_udesc = "Invalid",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "SHARED",
		  .pme_udesc = "Shared",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "EXCLUSIVE",
		  .pme_udesc = "Exclusive",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "OWNED",
		  .pme_udesc = "Owned",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "MODIFIED",
		  .pme_udesc = "Modified",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "Invalid, Shared, Exclusive, Owned, Modified",
		  .pme_ucode = 0x1f,
		}
	 }
	},
/* 13 */{.pme_name = "DATA_CACHE_LINES_EVICTED",
	.pme_code = 0x44,
	.pme_desc = "Data Cache Lines Evicted",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 6,
	.pme_umasks  = {
		{ .pme_uname = "INVALID",
		  .pme_udesc = "Invalid",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "SHARED",
		  .pme_udesc = "Shared",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "EXCLUSIVE",
		  .pme_udesc = "Exclusive",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "OWNED",
		  .pme_udesc = "Owned",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "MODIFIED",
		  .pme_udesc = "Modified",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "Invalid, Shared, Exclusive, Owned, Modified",
		  .pme_ucode = 0x1f,
		}
	 }
	},
/* 14 */{.pme_name = "L1_DTLB_MISS_AND_L2_DLTB_HIT",
	.pme_code = 0x45,
	.pme_desc = "L1 DTLB Miss and L2 DLTB Hit"
	},
/* 15 */{.pme_name = "L1_DTLB_AND_L2_DLTB_MISS",
	.pme_code = 0x46,
	.pme_desc = "L1 DTLB and L2 DLTB Miss"
	},
/* 16 */{.pme_name = "MISALIGNED_ACCESSES",
	.pme_code = 0x47,
	.pme_desc = "Misaligned Accesses"
	},
/* 17 */{.pme_name = "MICROARCHITECTURAL_LATE_CANCEL_OF_AN_ACCESS",
	.pme_code = 0x48,
	.pme_desc = "Microarchitectural Late Cancel of an Access"
	},
/* 18 */{.pme_name = "MICROARCHITECTURAL_EARLY_CANCEL_OF_AN_ACCESS",
	.pme_code = 0x49,
	.pme_desc = "Microarchitectural Early Cancel of an Access"
	},
/* 19 */{.pme_name = "SCRUBBER_SINGLE_BIT_ECC_ERRORS",
	.pme_code = 0x4A,
	.pme_desc = "Single-bit ECC Errors Recorded by Scrubber",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 2,
	.pme_umasks  = {
		{ .pme_uname = "SCRUBBER_ERROR",
		  .pme_udesc = "Scrubber error",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "PIGGYBACK_ERROR",
		  .pme_udesc = "Piggyback scrubber errors",
		  .pme_ucode = 0x02,
		}
	 }
	},
/* 20 */{.pme_name = "PREFETCH_INSTRUCTIONS_DISPATCHED",
	.pme_code = 0x4B,
	.pme_desc = "Prefetch Instructions Dispatched",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 3,
	.pme_umasks  = {
		{ .pme_uname = "LOAD",
		  .pme_udesc = "Load (Prefetch, PrefetchT0/T1/T2)",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "STORE",
		  .pme_udesc = "Store (PrefetchW)",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "NTA",
		  .pme_udesc = "Non-temporal Access NTA (PrefetchNTA)",
		  .pme_ucode = 0x04,
		}
	 }
	},
/* 21 */{.pme_name = "DCACHE_MISS_LOCKED_INSTRUCTIONS",
	.pme_code = 0x024C,
	.pme_desc = "DCACHE Misses by Locked Instructions: Data cache misses by locked instructions"
	},
/* 22 */{.pme_name = "DATA_PREFETCHES",
	.pme_code = 0x67,
	.pme_desc = "Data Prefetcher",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 2,
	.pme_umasks  = {
		{ .pme_uname = "CANCELLED",
		  .pme_udesc = "Cancelled prefetches",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "ATTEMPTED",
		  .pme_udesc = "Prefetch attempts",
		  .pme_ucode = 0x02,
		}
	 }
	},
/* 23 */{.pme_name = "SYSTEM_READ_RESPONSES",
	.pme_code = 0x6C,
	.pme_desc = "System Read Responses by Coherency State",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "EXCLUSIVE",
		  .pme_udesc = "Exclusive",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "MODIFIED",
		  .pme_udesc = "Modified",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "SHARED",
		  .pme_udesc = "Shared",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "Exclusive, Modified, Shared",
		  .pme_ucode = 0x07,
		}
	 }
	},
/* 24 */{.pme_name = "QUADWORD_WRITE_TRANSFERS",
	.pme_code = 0x016D,
	.pme_desc = "Quadwords Written to System: Quadword write transfer"
	},
/* 25 */{.pme_name = "REQUESTS_TO_L2",
	.pme_code = 0x7D,
	.pme_desc = "Requests to L2 Cache",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 6,
	.pme_umasks  = {
		{ .pme_uname = "INSTRUCTIONS",
		  .pme_udesc = "IC fill",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DATA",
		  .pme_udesc = "DC fill",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "TLB_WALK",
		  .pme_udesc = "TLB fill (page table walks)",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "SNOOP",
		  .pme_udesc = "Tag snoop request",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "CANCELLED",
		  .pme_udesc = "Cancelled request",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "All non-cancelled requests",
		  .pme_ucode = 0xf,
		}
	 }
	},
/* 26 */{.pme_name = "L2_CACHE_MISS",
	.pme_code = 0x7E,
	.pme_desc = "L2 Cache Misses",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "INSTRUCTIONS",
		  .pme_udesc = "IC fill",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DATA",
		  .pme_udesc = "DC fill",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "TLB_WALK",
		  .pme_udesc = "TLB page table walk",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "Instructions, Data, TLB walk",
		  .pme_ucode = 0x7,
		}
	 }
	},
/* 27 */{.pme_name = "L2_FILL_WRITEBACK",
	.pme_code = 0x017F,
	.pme_desc = "L2 Fill/Writeback: L2 fills (victims from L1 caches, TLB page table walks and data prefetches)"
	},
/* 28 */{.pme_name = "INSTRUCTION_CACHE_FETCHES",
	.pme_code = 0x80,
	.pme_desc = "Instruction Cache Fetches"
	},
/* 29 */{.pme_name = "INSTRUCTION_CACHE_MISSES",
	.pme_code = 0x81,
	.pme_desc = "Instruction Cache Misses"
	},
/* 30 */{.pme_name = "INSTRUCTION_CACHE_REFILLS_FROM_L2",
	.pme_code = 0x82,
	.pme_desc = "Instruction Cache Refills from L2"
	},
/* 31 */{.pme_name = "INSTRUCTION_CACHE_REFILLS_FROM_SYSTEM",
	.pme_code = 0x83,
	.pme_desc = "Instruction Cache Refills from System"
	},
/* 32 */{.pme_name = "L1_ITLB_MISS_AND_L2_ITLB_HIT",
	.pme_code = 0x84,
	.pme_desc = "L1 ITLB Miss, L2 ITLB Hit"
	},
/* 33 */{.pme_name = "L1_ITLB_MISS_AND_L2_ITLB_MISS",
	.pme_code = 0x85,
	.pme_desc = "L1 ITLB Miss, L2 ITLB Miss"
	},
/* 34 */{.pme_name = "PIPELINE_RESTART_DUE_TO_INSTRUCTION_STREAM_PROBE",
	.pme_code = 0x86,
	.pme_desc = "Pipeline Restart Due to Instruction Stream Probe"
	},
/* 35 */{.pme_name = "INSTRUCTION_FETCH_STALL",
	.pme_code = 0x87,
	.pme_desc = "Instruction Fetch Stall"
	},
/* 36 */{.pme_name = "RETURN_STACK_HITS",
	.pme_code = 0x88,
	.pme_desc = "Return Stack Hits"
	},
/* 37 */{.pme_name = "RETURN_STACK_OVERFLOWS",
	.pme_code = 0x89,
	.pme_desc = "Return Stack Overflows"
	},
/* 38 */{.pme_name = "RETIRED_CLFLUSH_INSTRUCTIONS",
	.pme_code = 0x26,
	.pme_desc = "Retired CLFLUSH Instructions"
	},
/* 39 */{.pme_name = "RETIRED_CPUID_INSTRUCTIONS",
	.pme_code = 0x27,
	.pme_desc = "Retired CPUID Instructions"
	},
/* 40 */{.pme_name = "CPU_CLK_UNHALTED",
	.pme_code = 0x76,
	.pme_desc = "CPU Clocks not Halted"
	},
/* 41 */{.pme_name = "RETIRED_INSTRUCTIONS",
	.pme_code = 0xC0,
	.pme_desc = "Retired Instructions"
	},
/* 42 */{.pme_name = "RETIRED_UOPS",
	.pme_code = 0xC1,
	.pme_desc = "Retired uops"
	},
/* 43 */{.pme_name = "RETIRED_BRANCH_INSTRUCTIONS",
	.pme_code = 0xC2,
	.pme_desc = "Retired Branch Instructions"
	},
/* 44 */{.pme_name = "RETIRED_MISPREDICTED_BRANCH_INSTRUCTIONS",
	.pme_code = 0xC3,
	.pme_desc = "Retired Mispredicted Branch Instructions"
	},
/* 45 */{.pme_name = "RETIRED_TAKEN_BRANCH_INSTRUCTIONS",
	.pme_code = 0xC4,
	.pme_desc = "Retired Taken Branch Instructions"
	},
/* 46 */{.pme_name = "RETIRED_TAKEN_BRANCH_INSTRUCTIONS_MISPREDICTED",
	.pme_code = 0xC5,
	.pme_desc = "Retired Taken Branch Instructions Mispredicted"
	},
/* 47 */{.pme_name = "RETIRED_FAR_CONTROL_TRANSFERS",
	.pme_code = 0xC6,
	.pme_desc = "Retired Far Control Transfers"
	},
/* 48 */{.pme_name = "RETIRED_BRANCH_RESYNCS",
	.pme_code = 0xC7,
	.pme_desc = "Retired Branch Resyncs"
	},
/* 49 */{.pme_name = "RETIRED_NEAR_RETURNS",
	.pme_code = 0xC8,
	.pme_desc = "Retired Near Returns"
	},
/* 50 */{.pme_name = "RETIRED_NEAR_RETURNS_MISPREDICTED",
	.pme_code = 0xC9,
	.pme_desc = "Retired Near Returns Mispredicted"
	},
/* 51 */{.pme_name = "RETIRED_INDIRECT_BRANCHES_MISPREDICTED",
	.pme_code = 0xCA,
	.pme_desc = "Retired Indirect Branches Mispredicted"
	},
/* 52 */{.pme_name = "RETIRED_MMX_AND_FP_INSTRUCTIONS",
	.pme_code = 0xCB,
	.pme_desc = "Retired MMX/FP Instructions",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 5,
	.pme_umasks  = {
		{ .pme_uname = "X87",
		  .pme_udesc = "x87 instructions",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "MMX_AND_3DNOW",
		  .pme_udesc = "MMX(TM) and 3DNow!(TM) instructions",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "PACKED_SSE_AND_SSE2",
		  .pme_udesc = "Packed SSE and SSE2 instructions",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "SCALAR_SSE_AND_SSE2",
		  .pme_udesc = "Scalar SSE and SSE2 instructions",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "X87, MMX(TM), 3DNow!(TM), Scalar and Packed SSE and SSE2 instructions",
		  .pme_ucode = 0x0f,
		}
	 }
	},
/* 53 */{.pme_name = "RETIRED_FASTPATH_DOUBLE_OP_INSTRUCTIONS",
	.pme_code = 0xCC,
	.pme_desc = "Retired Fastpath Double op Instructions",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "POSITION_0",
		  .pme_udesc = "low op in position 0",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "POSITION_1",
		  .pme_udesc = "low op in position 1",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "POSITION_2",
		  .pme_udesc = "low op in position 2",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "low op in position 0, 1, or 2",
		  .pme_ucode = 0x07,
		}
	 }
	},
/* 54 */{.pme_name = "INTERRUPTS_MASKED_CYCLES",
	.pme_code = 0xCD,
	.pme_desc = "Interrupts Masked Cycles"
	},
/* 55 */{.pme_name = "INTERRUPTS_MASKED_CYCLES_WITH_INTERRUPT_PENDING",
	.pme_code = 0xCE,
	.pme_desc = "Interrupts Masked Cycles with Interrupt Pending"
	},
/* 56 */{.pme_name = "INTERRUPTS_TAKEN",
	.pme_code = 0xCF,
	.pme_desc = "Interrupts Taken"
	},
/* 57 */{.pme_name = "DECODER_EMPTY",
	.pme_code = 0xD0,
	.pme_desc = "Decoder Empty"
	},
/* 58 */{.pme_name = "DISPATCH_STALLS",
	.pme_code = 0xD1,
	.pme_desc = "Dispatch Stalls"
	},
/* 59 */{.pme_name = "DISPATCH_STALL_FOR_BRANCH_ABORT",
	.pme_code = 0xD2,
	.pme_desc = "Dispatch Stall for Branch Abort to Retire"
	},
/* 60 */{.pme_name = "DISPATCH_STALL_FOR_SERIALIZATION",
	.pme_code = 0xD3,
	.pme_desc = "Dispatch Stall for Serialization"
	},
/* 61 */{.pme_name = "DISPATCH_STALL_FOR_SEGMENT_LOAD",
	.pme_code = 0xD4,
	.pme_desc = "Dispatch Stall for Segment Load"
	},
/* 62 */{.pme_name = "DISPATCH_STALL_FOR_REORDER_BUFFER_FULL",
	.pme_code = 0xD5,
	.pme_desc = "Dispatch Stall for Reorder Buffer Full"
	},
/* 63 */{.pme_name = "DISPATCH_STALL_FOR_RESERVATION_STATION_FULL",
	.pme_code = 0xD6,
	.pme_desc = "Dispatch Stall for Reservation Station Full"
	},
/* 64 */{.pme_name = "DISPATCH_STALL_FOR_FPU_FULL",
	.pme_code = 0xD7,
	.pme_desc = "Dispatch Stall for FPU Full"
	},
/* 65 */{.pme_name = "DISPATCH_STALL_FOR_LS_FULL",
	.pme_code = 0xD8,
	.pme_desc = "Dispatch Stall for Load/Store Full"
	},
/* 66 */{.pme_name = "DISPATCH_STALL_WAITING_FOR_ALL_QUIET",
	.pme_code = 0xD9,
	.pme_desc = "Dispatch Stall Waiting for All Quiet"
	},
/* 67 */{.pme_name = "DISPATCH_STALL_FOR_FAR_TRANSFER_OR_RSYNC",
	.pme_code = 0xDA,
	.pme_desc = "Dispatch Stall for Far Transfer or Resync to Retire"
	},
/* 68 */{.pme_name = "FPU_EXCEPTIONS",
	.pme_code = 0xDB,
	.pme_desc = "FPU Exceptions",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 5,
	.pme_umasks  = {
		{ .pme_uname = "X87_RECLASS_MICROFAULTS",
		  .pme_udesc = "x87 reclass microfaults",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "SSE_RETYPE_MICROFAULTS",
		  .pme_udesc = "SSE retype microfaults",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "SSE_RECLASS_MICROFAULTS",
		  .pme_udesc = "SSE reclass microfaults",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "SSE_AND_X87_MICROTRAPS",
		  .pme_udesc = "SSE and x87 microtraps",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "All",
		  .pme_ucode = 0xf,
		}
	 }
	},
/* 69 */{.pme_name = "DR0_BREAKPOINT_MATCHES",
	.pme_code = 0xDC,
	.pme_desc = "DR0 Breakpoint Matches"
	},
/* 70 */{.pme_name = "DR1_BREAKPOINT_MATCHES",
	.pme_code = 0xDD,
	.pme_desc = "DR1 Breakpoint Matches"
	},
/* 71 */{.pme_name = "DR2_BREAKPOINT_MATCHES",
	.pme_code = 0xDE,
	.pme_desc = "DR2 Breakpoint Matches"
	},
/* 72 */{.pme_name = "DR3_BREAKPOINT_MATCHES",
	.pme_code = 0xDF,
	.pme_desc = "DR3 Breakpoint Matches"
	},
/* 73 */{.pme_name = "DRAM_ACCESSES_PAGE",
	.pme_code = 0xE0,
	.pme_desc = "DRAM Accesses",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "HIT",
		  .pme_udesc = "Page Hit",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "MISS",
		  .pme_udesc = "Page Miss",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "CONFLICT",
		  .pme_udesc = "Page Conflict",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "Page Hit, Miss, or Conflict",
		  .pme_ucode = 0x07,
		}
	 }
	},
/* 74 */{.pme_name = "MEMORY_CONTROLLER_PAGE_TABLE_OVERFLOWS",
	.pme_code = 0xE1,
	.pme_desc = "Memory Controller Page Table Overflows"
	},
/* 75 */{.pme_name = "MEMORY_CONTROLLER_TURNAROUNDS",
	.pme_code = 0xE3,
	.pme_desc = "Memory Controller Turnarounds",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "CHIP_SELECT",
		  .pme_udesc = "DIMM (chip select) turnaround",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "READ_TO_WRITE",
		  .pme_udesc = "Read to write turnaround",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "WRITE_TO_READ",
		  .pme_udesc = "Write to read turnaround",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "ALL",
		  .pme_udesc = "All Memory Controller Turnarounds",
		  .pme_ucode = 0x07,
		}
	 }
	},
/* 76 */{.pme_name = "MEMORY_CONTROLLER_BYPASS",
	.pme_code = 0xE4,
	.pme_desc = "Memory Controller Bypass Counter Saturation",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "HIGH_PRIORITY",
		  .pme_udesc = "Memory controller high priority bypass",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "LOW_PRIORITY",
		  .pme_udesc = "Memory controller low priority bypass",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "DRAM_INTERFACE",
		  .pme_udesc = "DRAM controller interface bypass",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "DRAM_QUEUE",
		  .pme_udesc = "DRAM controller queue bypass",
		  .pme_ucode = 0x08,
		}
	 }
	},
/* 77 */{.pme_name = "SIZED_BLOCKS",
	.pme_code = 0xE5,
	.pme_desc = "Sized Blocks Sized Read/Write activity",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "32_BYTE_WRITES",
		  .pme_udesc = "32-byte Sized Writes (Revision D and later revisions)",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "64_BYTE_WRITES",
		  .pme_udesc = "64-byte Sized Writes (Revision D and later revisions)",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "32_BYTE_READS",
		  .pme_udesc = "32-byte Sized Reads (Revision D and later revisions)",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "64_BYTE_READS",
		  .pme_udesc = "64-byte Sized Reads (Revision D and later revisions)",
		  .pme_ucode = 0x20,
		}
	 }
	},
/* 78 */{.pme_name = "DRAM_ECC_ERRORS",
	.pme_code = 0x80E8,
	.pme_desc = "ECC Errors: Number of correctable and Uncorrectable DRAM ECC errors (Revision E)"
	},
/* 79 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_LOCAL_MEMORY",
	.pme_code = 0xA2E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Local Memory"
	},
/* 80 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_LOCAL_I/O",
	.pme_code = 0xA1E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Local I/O"
	},
/* 81 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_LOCAL_ANY",
	.pme_code = 0xA3E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Local Any"
	},
/* 82 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_LOCAL_MEMORY",
	.pme_code = 0xAAE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Local Memory"
	},
/* 83 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_LOCAL_I/O",
	.pme_code = 0xA5E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Local I/O"
	},
/* 84 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_LOCAL_ANY",
	.pme_code = 0xAFE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Local Any"
	},
/* 85 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_REMOTE_MEMORY",
	.pme_code = 0x98E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Remote Memory"
	},
/* 86 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_REMOTE_I/O",
	.pme_code = 0x94E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Remote I/O"
	},
/* 87 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_REMOTE_ANY",
	.pme_code = 0x9CE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Remote Any"
	},
/* 88 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_REMOTE_MEMORY",
	.pme_code = 0x92E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Remote Memory"
	},
/* 89 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_REMOTE_I/O",
	.pme_code = 0x91E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Remote I/O"
	},
/* 90 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_REMOTE_ANY",
	.pme_code = 0x93E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Remote Any"
	},
/* 91 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_REMOTE_MEMORY",
	.pme_code = 0x9AE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Remote Memory"
	},
/* 92 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_REMOTE_I/O",
	.pme_code = 0x95E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Remote I/O"
	},
/* 93 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_REMOTE_ANY",
	.pme_code = 0x9FE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Remote Any"
	},
/* 94 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_ANY_MEMORY",
	.pme_code = 0xB8E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Any Memory"
	},
/* 95 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_ANY_I/O",
	.pme_code = 0xB4E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Any I/O"
	},
/* 96 */{.pme_name = "REQUESTS_LOCAL_CPU_TO_ANY_ANY",
	.pme_code = 0xBCE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local CPU to Any Any"
	},
/* 97 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_ANY_MEMORY",
	.pme_code = 0xB2E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Any Memory"
	},
/* 98 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_ANY_I/O",
	.pme_code = 0xB1E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Any I/O"
	},
/* 99 */{.pme_name = "REQUESTS_LOCAL_I/O_TO_ANY_ANY",
	.pme_code = 0xB3E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local I/O to Any Any"
	},
/* 100 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_ANY_MEMORY",
	.pme_code = 0xBAE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Any Memory"
	},
/* 101 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_ANY_I/O",
	.pme_code = 0xB5E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Any I/O"
	},
/* 102 */{.pme_name = "REQUESTS_LOCAL_ANY_TO_ANY_ANY",
	.pme_code = 0xBFE9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Local Any to Any Any"
	},
/* 103 */{.pme_name = "REQUESTS_REMOTE_CPU_TO_LOCAL_I/O",
	.pme_code = 0x64E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Remote CPU to Local I/O"
	},
/* 104 */{.pme_name = "REQUESTS_REMOTE_I/O_TO_LOCAL_I/O",
	.pme_code = 0x61E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Remote I/O to Local I/O"
	},
/* 105 */{.pme_name = "REQUESTS_REMOTE_ANY_TO_LOCAL_I/O",
	.pme_code = 0x65E9,
	.pme_desc = "CPU/IO Requests to Memory/IO (Revision E):  Requests Remote Any to Local I/O"
	},
/* 106 */{.pme_name = "CACHE_BLOCK",
	.pme_code = 0xEA,
	.pme_desc = "Cache Block Commands (Revision E)",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 5,
	.pme_umasks  = {
		{ .pme_uname = "VICTIM_WRITEBACK",
		  .pme_udesc = "Victim Block (Writeback)",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DCACHE_LOAD_MISS",
		  .pme_udesc = "Read Block (Dcache load miss refill)",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "SHARED_ICACHE_REFILL",
		  .pme_udesc = "Read Block Shared (Icache refill)",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "READ_BLOCK_MODIFIED",
		  .pme_udesc = "Read Block Modified (Dcache store miss refill)",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "READ_TO_DIRTY",
		  .pme_udesc = "Change to Dirty (first store to clean block already in cache)",
		  .pme_ucode = 0x20,
		}
	 }
	},
/* 107 */{.pme_name = "SIZED_COMMANDS",
	.pme_code = 0xEB,
	.pme_desc = "Sized Commands",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 7,
	.pme_umasks  = {
		{ .pme_uname = "NON_POSTED_WRITE_BYTE",
		  .pme_udesc = "NonPosted SzWr Byte (1-32 bytes)",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "NON_POSTED_WRITE_DWORD",
		  .pme_udesc = "NonPosted SzWr Dword (1-16 dwords)",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "POSTED_WRITE_BYTE",
		  .pme_udesc = "Posted SzWr Byte (1-32 bytes)",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "POSTED_WRITE_DWORD",
		  .pme_udesc = "Posted SzWr Dword (1-16 dwords)",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "READ_BYTE_4_BYTES",
		  .pme_udesc = "SzRd Byte (4 bytes)",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "READ_DWORD_1_16_DWORDS",
		  .pme_udesc = "SzRd Dword (1-16 dwords)",
		  .pme_ucode = 0x20,
		},
		{ .pme_uname = "READ_MODIFY_WRITE",
		  .pme_udesc = "RdModWr",
		  .pme_ucode = 0x40,
		}
	 }
	},
/* 108 */{.pme_name = "PROBE",
	.pme_code = 0xEC,
	.pme_desc = "Probe Responses and Upstream Requests",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 7,
	.pme_umasks  = {
		{ .pme_uname = "MISS",
		  .pme_udesc = "Probe miss",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "HIT_CLEAN",
		  .pme_udesc = "Probe hit clean",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "HIT_CLEAN_NO_MEMORY_CANCEL",
		  .pme_udesc = "Probe hit dirty without memory cancel (probed by Sized Write or Change2Dirty)",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "HIT_DIRTY_WITH_MEMORY_CANCEL",
		  .pme_udesc = "Probe hit dirty with memory cancel (probed by DMA read or cache refill request)",
		  .pme_ucode = 0x08,
		},
		{ .pme_uname = "UPSTREAM_DISPLAY_REFRESH_READS",
		  .pme_udesc = "Upstream display refresh reads",
		  .pme_ucode = 0x10,
		},
		{ .pme_uname = "UPSTREAM_NON_DISPLAY_REFRESH_READS",
		  .pme_udesc = "Upstream non-display refresh reads",
		  .pme_ucode = 0x20,
		},
		{ .pme_uname = "UPSTREAM_WRITES",
		  .pme_udesc = "Upstream writes (Revision D and later revisions)",
		  .pme_ucode = 0x40,
		},
	 }
	},
/* 109 */{.pme_name = "GART",
	.pme_code = 0xEE,
	.pme_desc = "GART Events",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 3,
	.pme_umasks  = {
		{ .pme_uname = "APERTURE_HIT_FROM_CPU",
		  .pme_udesc = "GART aperture hit on access from CPU",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "APERTURE_HIT_FROM_IO",
		  .pme_udesc = "GART aperture hit on access from I/O",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "MISS",
		  .pme_udesc = "GART miss",
		  .pme_ucode = 0x04,
		}
	 }
	},
/* 110 */{.pme_name = "HYPERTRANSPORT_LINK0",
	.pme_code = 0xF6,
	.pme_desc = "HyperTransport Link 0 Transmit Bandwidth",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "COMMAND_DWORD_SENT",
		  .pme_udesc = "Command Dword sent",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DATA_DWORD_SENT",
		  .pme_udesc = "Data Dword sent",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "BUFFER_RELEASE_DWORD_SENT",
		  .pme_udesc = "Buffer release Dword sent",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "NOP_DWORD_SENT",
		  .pme_udesc = "Nop Dword sent (idle)",
		  .pme_ucode = 0x08,
		}
	 }
	},
/* 111 */{.pme_name = "HYPERTRANSPORT_LINK1",
	.pme_code = 0xF7,
	.pme_desc = "HyperTransport Link 1 Transmit Bandwidth",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "COMMAND_DWORD_SENT",
		  .pme_udesc = "Command Dword sent",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DATA_DWORD_SENT",
		  .pme_udesc = "Data Dword sent",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "BUFFER_RELEASE_DWORD_SENT",
		  .pme_udesc = "Buffer release Dword sent",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "NOP_DWORD_SENT",
		  .pme_udesc = "Nop Dword sent (idle)",
		  .pme_ucode = 0x08,
		}
	 }
	},
/* 112 */{.pme_name = "HYPERTRANSPORT_LINK2",
	.pme_code = 0xF8,
	.pme_desc = "HyperTransport Link 2 Transmit Bandwidth",
	.pme_flags   = PFMLIB_AMD64_UMASK_COMBO,
	.pme_numasks = 4,
	.pme_umasks  = {
		{ .pme_uname = "COMMAND_DWORD_SENT",
		  .pme_udesc = "Command Dword sent",
		  .pme_ucode = 0x01,
		},
		{ .pme_uname = "DATA_DWORD_SENT",
		  .pme_udesc = "Data Dword sent",
		  .pme_ucode = 0x02,
		},
		{ .pme_uname = "BUFFER_RELEASE_DWORD_SENT",
		  .pme_udesc = "Buffer release Dword sent",
		  .pme_ucode = 0x04,
		},
		{ .pme_uname = "NOP_DWORD_SENT",
		  .pme_udesc = "Nop Dword sent (idle)",
		  .pme_ucode = 0x08,
		}
	 }
	},
};
#define PME_AMD64_CPU_CLK_UNHALTED	40
#define PME_AMD64_RETIRED_INSTRUCTIONS	41
#define PME_AMD64_EVENT_COUNT	   (sizeof(amd64_pe)/sizeof(pme_amd64_entry_t))
