#include "osl/game_playing/gameState.h"
#include "osl/game_playing/searchPlayer.tcc"
#include "osl/sennichite.h"
#include "osl/record/csaString.h"
#include "osl/record/record.h"
#include "osl/record/csaRecord.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <fstream>
#include <iostream>

using namespace osl;
using namespace game_playing;

class GameStateTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(GameStateTest);
  CPPUNIT_TEST(testRepetition);
  CPPUNIT_TEST(testCanPopMove);
  CPPUNIT_TEST(testRootHistoryFiles);
  CPPUNIT_TEST(testRootHistoryImmediateUse);
  CPPUNIT_TEST(testGenerateNotLosingMoves);
  CPPUNIT_TEST_SUITE_END();
public:
  void testRepetition();
  void testCanPopMove();
  void testRootHistoryFiles();
  void testRootHistoryImmediateUse();
  void testRootHistoryRecord(const SimpleState& state, const vector<osl::Move>&);
  void testGenerateNotLosingMoves();
};

CPPUNIT_TEST_SUITE_REGISTRATION(GameStateTest);

void GameStateTest::testCanPopMove()
{
  SimpleState initialState(HIRATE);
  GameState state(initialState);
  CPPUNIT_ASSERT(! state.canPopMove());
  const Move m76(Position(7,7),Position(7,6),PAWN,PTYPE_EMPTY,false,BLACK);
  state.pushMove(m76);
  CPPUNIT_ASSERT(state.canPopMove());
  CPPUNIT_ASSERT_EQUAL(initialState, state.getInitialState());
  
  boost::shared_ptr<GameState> state_clone = state.clone();
  CPPUNIT_ASSERT(state_clone->canPopMove());
  const Move m84(Position(8,3),Position(8,4),PAWN,PTYPE_EMPTY,false,WHITE);
  state_clone->pushMove(m84);
  CPPUNIT_ASSERT(state_clone->canPopMove());

  CPPUNIT_ASSERT_EQUAL(m84, state_clone->popMove());
  CPPUNIT_ASSERT_EQUAL(m76, state.popMove());
  CPPUNIT_ASSERT_EQUAL(initialState, state_clone->getInitialState());
}

void GameStateTest::testRepetition()
{
  GameState state((SimpleState(HIRATE)));
  const Move m48(Position(3,9),Position(4,8),SILVER,PTYPE_EMPTY,false,BLACK);
  const Move m39(Position(4,8),Position(3,9),SILVER,PTYPE_EMPTY,false,BLACK);
  const Move m42(Position(3,1),Position(4,2),SILVER,PTYPE_EMPTY,false,WHITE);
  const Move m31(Position(4,2),Position(3,1),SILVER,PTYPE_EMPTY,false,WHITE);

  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m48));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m42));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m39));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m31));
  
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m48));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m42));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m39));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m31));

  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m48));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m42));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), state.pushMove(m39));
  CPPUNIT_ASSERT_EQUAL(Sennichite::DRAW(), state.pushMove(m31));
}


void GameStateTest::testRootHistoryRecord(const SimpleState& sstate, 
					  const vector<osl::Move>& moves)
{
  GameState state(sstate);
  for (unsigned int i=0;i<moves.size();i++)
  {
    const Move move = moves[i];
    state.pushMove(move);
    DualDfpn searcher;
    searcher.writeRootHistory(state.counter(), state.moveHistory(),
			      state.state(), state.state().getTurn());
  }    
}

void GameStateTest::testRootHistoryImmediateUse()
{
  // 直前に取った駒を使っても大丈夫
  SimpleState state(CsaString(
		      "P1 * -KE *  *  *  * -KI *  * \n"
		      "P2 *  *  *  *  *  * -KI *  * \n"
		      "P3-FU *  * -FU *  * -KE * +HI\n"
		      "P4 *  * -FU *  * -FU * -FU * \n"
		      "P5 *  * +KI+FU * -KY *  *  * \n"
		      "P6 *  *  *  *  *  *  * +KE-OU\n"
		      "P7+FU *  *  * +FU * +FU+FU * \n"
		      "P8+KA * +GI+OU *  * -NG-HI-KY\n"
		      "P9+KY+KE *  *  * +KI *  * +KY\n"
		      "P+00KA00GI00GI\n"
		      "P-00FU00FU00FU00FU00FU00FU00FU00FU\n"
		      "-\n").getInitialState());
  vector<Move> moves;
  moves.push_back(Move(Position(1,5),PAWN,WHITE));
  moves.push_back(Move(Position(1,7),SILVER,BLACK));
  moves.push_back(Move(Position(1,6),Position(1,7),KING,SILVER,false,WHITE));
  moves.push_back(Move(Position(1,3),Position(1,5),PROOK,PAWN,true,BLACK));
  moves.push_back(Move(Position(1,6),SILVER,WHITE));

  testRootHistoryRecord(state, moves);

}
void GameStateTest::testRootHistoryFiles()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0; i<(isShortTest ? 3: 300) && (ifs >> file_name); i++){
    if (file_name == "") 
      break;
    if (! isShortTest)
      std::cerr << file_name << "\n";
    file_name = OslConfig::testCsaFile(file_name);
    Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();
    testRootHistoryRecord(SimpleState(HIRATE), moves);
  }
}

void GameStateTest::testGenerateNotLosingMoves()
{
  GameState state(CsaString(
		    "P1 *  *  *  *  *  *  * -KE-OU\n"
		    "P2 *  *  * -GI * -KI-KI *  * \n"
		    "P3 *  *  * -FU *  * -GI-FU+UM\n"
		    "P4-KY-FU *  * -FU+FU-FU * -KY\n"
		    "P5 *  *  *  *  * -FU * +KE * \n"
		    "P6-HI * -FU+FU+FU *  * +GI * \n"
		    "P7+KE+FU * +KI *  * +KE+FU * \n"
		    "P8+FU+OU+KI+KA * -NY *  *  * \n"
		    "P9 *  *  *  *  *  *  *  *  * \n"
		    "P+00FU00FU\n"
		    "P-00HI00GI00KY00FU00FU00FU\n"
		    "+\n").getInitialState());
  MoveVector normal_or_win_or_draw, loss;
  state.generateNotLosingMoves(normal_or_win_or_draw, loss);
  const Move m12fu(Position(1,2),PAWN,BLACK);
  CPPUNIT_ASSERT(! normal_or_win_or_draw.isMember(m12fu));
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; coding:utf-8
// ;;; End:
