(**
  A model for lists. Also declares some convinience classes for handling
  one-column lists and lists of text.

  TODO
  * Create special ResyncMsgs for certain operatation, that allows the
    corresponding ListObject the make optimized refeshes
**)

MODULE VO:Model:List;

(*
    A Model for lists.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       G   := VO:Object,
       O   := VO:Base:Object,
       T   := VO:Text,
       U   := VO:Base:Util,

       str := Strings;

TYPE
  ListModel*     = POINTER TO ListModelDesc;


  (*
    The baseclass for all entries in the listmodel.
  *)

  ListEntry*     = POINTER TO ListEntryDesc;
  ListEntryDesc* = RECORD
                     last-,next- : ListEntry; (* All elements are conected *)
                     list        : ListModel; (* Our own model *)
                     selected-   : BOOLEAN;   (* This entry is selected *)
                     calced      : BOOLEAN;   (* We already calculated the size of objects *)
                     height-     : LONGINT;   (* The overall hight of all elements in this entry *)
                   END;

  (*
    A (potential) baseclass for all simple entries, this are
    entries that only consist of an object.
  *)

  SimplEntry*    = POINTER TO SimplEntryDesc;
  SimplEntryDesc*= RECORD (ListEntryDesc)
                     object- : G.Object;
                   END;

  (*
    A simple listentry for text.
    TextEntry.object points to the created instanz of VOText.Text
    and text points to the unescaped original text.
  *)

  TextEntry*     = POINTER TO TextEntryDesc;
  TextEntryDesc* = RECORD (SimplEntryDesc)
                     text- : U.Text;
                   END;

  (*
    The listmodel itself.

    list and last point to the corresponding entries of the list.
    You can walk trough the list using this pointers if you want
    information about a number of entries. F.e. if you have a
    multiselect list and want to know all selected entries.
  *)

  ListModelDesc* = RECORD (O.ModelDesc)
                     list-,last-,
                     lastSelected-: ListEntry;
                     size-        : LONGINT;
                   END;

  (*
    The corresponding gadget gets notified that this entry has changed its
    state (seleted<->unselected).
  *)

  EntryChgdMsg*     = POINTER TO EntryChgdMsgDesc;
  EntryChgdMsgDesc* = RECORD (O.ResyncMsgDesc)
                        entry- : ListEntry;
                      END;

  PROCEDURE (l : ListModel) Init*;

  BEGIN
    l.Init^;

    l.list:=NIL;
    l.last:=NIL;
    l.lastSelected:=NIL;
    l.size:=0;
  END Init;

  (*
    Append this entry to the list.
  *)

  PROCEDURE (l : ListModel) Append*(entry : ListEntry);

  BEGIN
    IF l.list=NIL THEN
      l.list:=entry;
      entry.last:=NIL;
      entry.next:=NIL;
    ELSE
      l.last.next:=entry;
      entry.last:=l.last;
      entry.next:=NIL;
    END;
    l.last:=entry;
    entry.list:=l;
    entry.calced:=FALSE;
    entry.selected:=FALSE;
    INC(l.size);
    l.Notify(NIL);
  END Append;

  PROCEDURE (l : ListModel) Delete*(entry : ListEntry);

  BEGIN
    IF l.list=entry THEN
      l.list:=l.list.next;
    END;

    IF l.last=entry THEN
      l.last:=l.last.last;
    END;

    IF entry.last#NIL THEN
      entry.last.next:=entry.next;
    END;

    IF entry.next#NIL THEN
      entry.next.last:=entry.last;
    END;

    DEC(l.size);

    l.Notify(NIL);
  END Delete;

  (*
    Delete all entries in the list.
  *)

  PROCEDURE (l : ListModel) Clear*;

  BEGIN
    l.list:=NIL;
    l.last:=NIL;
    l.size:=0;
    l.Notify(NIL);
  END Clear;

  (*
    Get the xth entry. Index starts with 1.
  *)

  PROCEDURE (l : ListModel) Get*(pos : LONGINT):ListEntry;

  VAR
    x     : LONGINT;
    entry : ListEntry;

  BEGIN
    x:=1;
    entry:=l.list;

    WHILE (entry#NIL) & (x<pos) DO
      entry:=entry.next;
      INC(x);
    END;

    RETURN entry;
  END Get;

  (*
    Get the last selected entry. Not that the correspondig atribute
    is not cleared when the last selected entry gets unselected using
    ListEntry.Deselect. The return value is only valid after getting a
    corresponding msg from the gadget that an entry has been selected.

    However for multiselect this entry may not tell the whole truth.

  *)

  PROCEDURE (l : ListModel) GetSelected*():ListEntry;

  BEGIN
    RETURN l.lastSelected;
  END GetSelected;

  (*
    Get the nth object stored in this ListEntry.
    Entries for a multicolumn list may have more than one object
    stored.
  *)

  PROCEDURE (l : ListEntry) GetObject*(pos : LONGINT):G.Object;

  BEGIN
    RETURN NIL;
  END GetObject;

  (*
    Select this entry.
  *)

  PROCEDURE (l : ListEntry) Select*;

  VAR
    changed : EntryChgdMsg;

  BEGIN
    IF ~l.selected THEN
      l.selected:=TRUE;
      l.list.lastSelected:=l;
      NEW(changed);
      changed.entry:=l;
      l.list.Notify(changed);
    END;
  END Select;

  (*
    Deselect this entry.
  *)

  PROCEDURE (l : ListEntry) Deselect*;

  VAR
    changed : EntryChgdMsg;

  BEGIN
    IF l.selected THEN
      l.selected:=FALSE;
      NEW(changed);
      changed.entry:=l;
      l.list.Notify(changed);
    END;
  END Deselect;

  (*
    Toggle state of entry.
  *)

  PROCEDURE (l : ListEntry) Toggle*;

  VAR
    changed : EntryChgdMsg;

  BEGIN
    l.selected:=~l.selected;
    IF l.selected THEN
      l.list.lastSelected:=l;
    END;
    NEW(changed);
    changed.entry:=l;
    l.list.Notify(changed);
  END Toggle;

  (**
    Selects the given entry.
  **)

  PROCEDURE (l : ListModel) Select*(pos : LONGINT);

  VAR
    entry : ListEntry;

  BEGIN
    entry:=l.Get(pos);
    IF entry#NIL THEN
      entry.Select();
    END;
  END Select;

  (**
    Deselects the given entry.
  **)

  PROCEDURE (l : ListModel) Deselect*(pos : LONGINT);

  VAR
    entry : ListEntry;

  BEGIN
    entry:=l.Get(pos);
    IF entry#NIL THEN
      entry.Deselect();
    END;
  END Deselect;

  (**
    Toggles he selection of the given entry.
  **)

  PROCEDURE (l : ListModel) Toggle*(pos : LONGINT);

  VAR
    entry : ListEntry;

  BEGIN
    entry:=l.Get(pos);
    IF entry#NIL THEN
      entry.Toggle();
    END;
  END Toggle;

  (**
    Returns the position in the list, starting with 1.
    If the entry is not in the list (each entry store
    directly a pointer to its corresponding ListModel) it returns
    -1.

    NOTE
    This is an exspensive operation. The mehtods goes through the
    list of elements until it finds the entry. If you want to now
    the position of more than one entry, goes through the list once
    and count yourself.

  **)

  PROCEDURE (l : ListEntry) GetPos*():LONGINT;

  VAR
    entry : ListEntry;
    pos   : LONGINT;

  BEGIN
    pos:=1;
    entry:=l.list.list;
    WHILE entry#NIL DO
      IF entry=l THEN
        RETURN pos;
      END;
      entry:=entry.next;
      INC(pos);
    END;

    RETURN -1;
  END GetPos;

  (*
    Abstract. Calc size of all entries.
    Set ListEntry.calced to TRUE and ListEntry.height to to
    height of the largest object.
  *)

  PROCEDURE (l : ListEntry) CalcSize*;

  BEGIN
    l.calced:=TRUE;
  END CalcSize;

  (*
    Set the one object a SimplEntry can have.
  *)

  PROCEDURE (l : SimplEntry) SetObject*(object : G.Object);

  BEGIN
    l.object:=object;
  END SetObject;

  (*
    Return the one object, a SimplEntry can have.
  *)

  PROCEDURE (l : SimplEntry) GetObject*(pos : LONGINT):G.Object;

  BEGIN
    RETURN l.object;
  END GetObject;

  (*
    CaclSize for SimplEntry.
  *)

  PROCEDURE (l : SimplEntry) CalcSize*;

  BEGIN
    IF ~l.calced THEN
      l.object.CalcSize;
      l.calced:=TRUE;
      l.height:=l.object.height;
    END;
  END CalcSize;

  (**
    Create a new SimplEntry with @var{object} as only child.
  **)

  PROCEDURE NewSimplEntry*(object : G.Object):SimplEntry;

  VAR
    simplEntry : SimplEntry;

  BEGIN
    NEW(simplEntry);
    simplEntry.SetObject(object);
    RETURN simplEntry;
  END NewSimplEntry;

  (**
    Create a special TextEntry with includes one textobject
    generatd from the text @var{text}. Escaping is done by this function.
  **)

  PROCEDURE NewTextEntry*(text : ARRAY OF CHAR):SimplEntry;

  VAR
    entry      : TextEntry;
    labelText  : T.Text;
    help       : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.leftAlligned,{},D.normalFont);
    help:=U.EscapeString(text);
    labelText.SetText(help^);

    NEW(entry);
    entry.SetObject(labelText);
    NEW(entry.text,str.Length(text)+1);
    COPY(text,entry.text^);
    RETURN entry;
  END NewTextEntry;

END VO:Model:List.