(**
  A layout group, that layout menues and the top group object within a window.
**)

MODULE VO:WindowGroup;

(*
    A layout group, that layouts a number of buttons horizontaly.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VO:Base:Display,
       E := VO:Base:Event,
       U := VO:Base:Util,
       Z := VO:Base:Size,

       G := VO:Object;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hSpace*,
                 vSpace* : Z.SizeDesc;
               END;

  WindowGroup*     = POINTER TO WindowGroupDesc;
  WindowGroupDesc* = RECORD(G.ObjectDesc)
                     men,top,
                     tool,status : G.Object;
                     space       : BOOLEAN;
                   END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hSpace.Init;
    p.vSpace.Init;
    p.hSpace.SetSize(Z.unit,1);
    p.vSpace.SetSize(Z.unit,1);
  END Init;

  (**
    WindowGroup inherits horizontalFlex & verticalFlex from top.
  **)

  PROCEDURE (w : WindowGroup) Init*;

  BEGIN
    w.Init^;

    w.SetPrefs(prefs);

    w.men:=NIL;
    w.top:=NIL;
    w.tool:=NIL;
    w.status:=NIL;
    w.space:=TRUE;
  END Init;

  PROCEDURE (w : WindowGroup) Set*(menu, top : G.Object; space : BOOLEAN);

  BEGIN
    w.men:=menu;
    IF w.men#NIL THEN
      w.men.SetParent(w);
    END;

    w.top:=top;
    IF w.top#NIL THEN
      w.top.SetParent(w);
    END;
  END Set;

  PROCEDURE (w : WindowGroup) SetToolbar*(toolbar : G.Object);

  BEGIN
    w.tool:=toolbar;
    IF w.tool#NIL THEN
      w.tool.SetParent(w);
    END;
  END SetToolbar;

  PROCEDURE (w : WindowGroup) SetStatusLine*(status : G.Object);

  BEGIN
    w.status:=status;
    IF w.status#NIL THEN
      w.status.SetParent(w);
    END;
  END SetStatusLine;

  PROCEDURE (w : WindowGroup) CalcSize*;

  BEGIN
    w.width:=0;
    w.height:=0;
    w.minWidth:=0;
    w.minHeight:=0;

    IF w.men#NIL THEN
      w.men.CalcSize;
      w.width:=w.men.oWidth;
      w.height:=w.men.oHeight;
      w.minWidth:=w.men.oMinWidth;
      w.minHeight:=w.men.oMinHeight;
    END;

    IF w.tool#NIL THEN
      w.tool.CalcSize;
      INC(w.height,w.tool.oHeight);
      INC(w.minHeight,w.tool.oMinHeight);
      w.width:=U.MaxLong(w.width,w.tool.oWidth);
      w.minWidth:=U.MaxLong(w.minWidth,w.tool.oMinWidth);
    END;

    IF w.status#NIL THEN
      w.status.CalcSize;
      INC(w.height,w.status.oHeight);
      INC(w.minHeight,w.status.oMinHeight);
      w.width:=U.MaxLong(w.width,w.status.oWidth);
      w.minWidth:=U.MaxLong(w.minWidth,w.status.oMinWidth);
    END;

    IF w.top#NIL THEN
      w.top.CalcSize;
      INC(w.height,w.top.oHeight);
      INC(w.minHeight,w.top.oMinHeight);

      IF w.space THEN
        INC(w.height,2*w.prefs(Prefs).vSpace.GetSize());
        INC(w.minHeight,2*w.prefs(Prefs).vSpace.GetSize());
        w.width:=U.MaxLong(w.width,
                           w.top.oWidth+2*w.prefs(Prefs).hSpace.GetSize());
        w.minWidth:=U.MaxLong(w.minWidth,
                              w.top.oMinWidth+2*w.prefs(Prefs).hSpace.GetSize());
      ELSE
        w.width:=U.MaxLong(w.width,w.top.oWidth);
        w.minWidth:=U.MaxLong(w.minWidth,w.top.oMinWidth);
      END;

      IF (G.horizontalFlex IN w.top.flags) THEN
        w.SetFlags({G.horizontalFlex});
      END;
      IF G.verticalFlex IN w.top.flags THEN
        w.SetFlags({G.verticalFlex});
      END;
    END;
    w.CalcSize^;
  END CalcSize;


  PROCEDURE (w : WindowGroup) HandleMouseEvent*(event : E.MouseEvent;
                                                VAR grab : G.Object):BOOLEAN;

  BEGIN
    IF w.men#NIL THEN
      IF w.men.HandleMouseEvent(event,grab) THEN
        RETURN TRUE;
      END;
    END;
    IF w.tool#NIL THEN
      IF w.tool.HandleMouseEvent(event,grab) THEN
        RETURN TRUE;
      END;
    END;
    IF w.top#NIL THEN
      IF w.top.HandleMouseEvent(event,grab) THEN
        RETURN TRUE;
      END;
    END;
    IF w.status#NIL THEN
      IF w.status.HandleMouseEvent(event,grab) THEN
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleMouseEvent;


  PROCEDURE (w : WindowGroup) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    help  : G.Object;

  BEGIN
    IF w.men#NIL THEN
      help:=w.men.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.tool#NIL THEN
      help:=w.tool.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.top#NIL THEN
      help:=w.top.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.status#NIL THEN
      help:=w.status.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    RETURN w.GetPosObject^(x,y,type); (** Hoo, not that good *)
  END GetPosObject;

  PROCEDURE (w : WindowGroup) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    help  : G.Object;

  BEGIN
    IF w.men#NIL THEN
      help:=w.men.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.tool#NIL THEN
      help:=w.tool.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.top#NIL THEN
      help:=w.top.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.status#NIL THEN
      help:=w.status.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    RETURN NIL;
  END GetDnDObject;

  PROCEDURE (w : WindowGroup) Layout*;

  VAR
    yPos,height : LONGINT;

  BEGIN
    yPos:=w.y;
    height:=w.height;

    IF w.men#NIL THEN
      w.men.Resize(w.width,-1);
      w.men.Move(w.x,yPos);
      INC(yPos,w.men.oHeight);
      DEC(height,w.men.oHeight);
    END;

    IF w.tool#NIL THEN
      w.tool.Resize(w.width,-1);
      w.tool.Move(w.x,yPos);
      INC(yPos,w.tool.oHeight);
      DEC(height,w.tool.oHeight);
    END;

    IF w.status#NIL THEN
      w.status.Resize(w.width,-1);
      w.status.Move(w.x,w.height-w.status.oHeight);
      DEC(height,w.status.oHeight);
    END;

    IF w.top#NIL THEN
      IF w.space THEN
        w.top.Resize(w.width-2*w.prefs(Prefs).vSpace.GetSize(),
                     height-2*w.prefs(Prefs).vSpace.GetSize());
      ELSE
        w.top.Resize(w.width,height);
      END;

      w.top.Move(w.x+(w.width-w.top.oWidth) DIV 2,
                 yPos+(height-w.top.oHeight) DIV 2);
    END;

    w.Layout^;
  END Layout;

  PROCEDURE (w : WindowGroup) Draw*(x,y,width,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    w.Draw^(x,y,width,h);

    IF ~w.Intersect(x,y,width,h) THEN
      RETURN;
    END;

    draw:=w.GetDrawInfo();

    draw.InstallClip(x,y,width,h);

    IF w.men#NIL THEN
      draw.SubRegion(w.men.oX,w.men.oY,w.men.oWidth,w.men.oHeight);
    END;
    IF w.tool#NIL THEN
      draw.SubRegion(w.tool.oX,w.tool.oY,w.tool.oWidth,w.tool.oHeight);
    END;
    IF w.status#NIL THEN
      draw.SubRegion(w.status.oX,w.status.oY,w.status.oWidth,w.status.oHeight);
    END;
    IF w.top#NIL THEN
      draw.SubRegion(w.top.oX,w.top.oY,w.top.oWidth,w.top.oHeight);
    END;

    w.DrawBackground(w.x,w.y,w.width,w.height);

    draw.FreeLastClip;

    IF w.men#NIL THEN
      w.men.Draw(x,y,width,h);
    END;
    IF w.tool#NIL THEN
      w.tool.Draw(x,y,width,h);
    END;
    IF w.status#NIL THEN
      w.status.Draw(x,y,width,h);
    END;
    IF w.top#NIL THEN
      w.top.Draw(x,y,width,h);
    END;
  END Draw;

  PROCEDURE (w : WindowGroup) Hide*;

  BEGIN
    IF w.visible THEN
      IF w.men#NIL THEN
        w.men.Hide;
      END;
      IF w.tool#NIL THEN
        w.tool.Hide;
      END;
      IF w.top#NIL THEN
        w.top.Hide;
      END;
      IF w.status#NIL THEN
        w.status.Hide;
      END;
      w.DrawHide;
      w.Hide^;
    END;
  END Hide;

  PROCEDURE CreateWindowGroup*():WindowGroup;

  VAR
    windowGroup : WindowGroup;

  BEGIN
    NEW(windowGroup);
    windowGroup.Init;

    RETURN windowGroup;
  END CreateWindowGroup;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:WindowGroup.