(**
   A model for all basic datatypes.
**)

MODULE VO:Model:Value;

(*
    A model for all basic datatypes.
    Copyright (C) 1999  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT O   := VO:Base:Object,
       U   := VO:Base:Util,

       co  := IntStr,
       str := Strings;

CONST
  (*
     datatypes of value stored in model

     TODO: support more datatypes
  *)

  unknown*           =  0;
  char*              =  1;
  text*              =  2;
  real*              =  3;
  longreal*          =  4;
  shortint*          =  5;
  integer*           =  6;
  longint*           =  7;
  boolean*           =  8;


(*
    CASE v.type OF
      unknown:
    | char:
    | text:
    | real:
    | longreal:
    | shortint:
    | integer:
    | longint:
    | boolean:
    END;
*)

TYPE
  Value*          = POINTER TO ValueDesc;
  ValueDesc*      = RECORD
                      string   : U.Text;
                      longint  : LONGINT;
                      real     : REAL;
                      longreal : LONGREAL;
                      char     : CHAR;
                      boolean  : BOOLEAN;
                      isNull   : BOOLEAN;
                      last     : Value;
                    END;

  ValueModel*     = POINTER TO ValueModelDesc;
  ValueModelDesc* = RECORD (O.ModelDesc)
                      type-    : LONGINT;
                      value    : ValueDesc;
                    END;

  BoolModel*      = POINTER TO BoolModelDesc;
  BoolModelDesc*  = RECORD (ValueModelDesc)
                    END;

  IntModel*       = POINTER TO IntModelDesc;
  IntModelDesc*   = RECORD (ValueModelDesc)
                    END;

  StringModel*    = POINTER TO StringModelDesc;
  StringModelDesc*= RECORD (ValueModelDesc)
                    END;

  (* --- Value object --- *)

  PROCEDURE (VAR v : ValueDesc) Equal*(value : Value; type : LONGINT):BOOLEAN;

  BEGIN
    CASE type OF
      unknown:
      RETURN TRUE;
    | char:
      RETURN value.char=v.char;
    | text:
      RETURN value.string^=v.string^;
    | real:
      RETURN value.real=v.real;
    | longreal:
      RETURN value.longreal=v.longreal;
    | shortint,
      integer,
      longint:
      RETURN value.longint=v.longint;
    | boolean:
      RETURN value.boolean=v.boolean;
    END;
  END Equal;

  (**
    Initialize object to type unknown.
  **)

  PROCEDURE (v : ValueModel) Init*;

  BEGIN
    v.Init^;

    v.type:=unknown;
    v.value.isNull:=TRUE;
    v.value.last:=NIL;
  END Init;

  (**
    Convert data from type "from" to destination type "to", if possible.
  **)

  PROCEDURE (v : ValueModel) Convert(from, to : LONGINT):BOOLEAN;


  VAR
    ret : co.ConvResults;

  BEGIN
    IF v.value.isNull THEN
      RETURN FALSE;
    END;

    CASE from OF
      unknown:
      CASE to OF
        unknown:
        RETURN TRUE;
      | char:
        v.value.char:=0X;
        RETURN TRUE;
      | text:
        NEW(v.value.string,1);
        v.value.string[0]:=0X;
        RETURN TRUE;
      | real:
        v.value.real:=0.0;
        RETURN TRUE;
      | longreal:
        v.value.longreal:=0.0;
        RETURN TRUE;
      | shortint,
        integer,
        longint:
        v.value.longint:=0;
        RETURN TRUE;
      | boolean:
        v.value.boolean:=v.value.char#0X;
        RETURN TRUE;
      END;


    | char:
      CASE to OF
        unknown:
      | char:
        RETURN TRUE;
      | text:
        NEW(v.value.string,2);
        v.value.string[0]:=v.value.char;
        v.value.string[1]:=0X;
        RETURN TRUE;
      | real:
      | longreal:
      | shortint:
      | integer:
      | longint:
      | boolean:
        v.value.boolean:=v.value.char#0X;
        RETURN TRUE;
      END;

    | text:
      CASE to OF
        unknown:
      | char:
        v.value.char:=v.value.string[0];
        RETURN TRUE;
      | text:
        RETURN TRUE;
      | real:
      | longreal:
      | shortint,
        integer,
        longint:
        co.StrToInt(v.value.string^,v.value.longint,ret);
        v.value.isNull:=ret#co.strAllRight;
        RETURN ret=co.strAllRight;
      | boolean:
        IF (v.value.string^="true") OR (v.value.string^="TRUE") THEN
          v.value.boolean:=TRUE;
        ELSIF (v.value.string^="false") OR (v.value.string^="FALSE") THEN
          v.value.boolean:=FALSE;
        ELSE
          v.value.boolean:=v.value.string[0]#0X;
        END;
        RETURN TRUE;
      END;

    | real:
      CASE to OF
        unknown:
      | char:
      | text:
      | real:
        RETURN TRUE;
      | longreal:
        v.value.longreal:=LONG(v.value.real);
        RETURN TRUE;
      | shortint:
      | integer:
      | longint:
      | boolean:
        v.value.boolean:=v.value.real#0.0;
        RETURN TRUE;
      END;

    | longreal:
      CASE to OF
        unknown:
      | char:
      | text:
      | real:
        v.value.real:=SHORT(v.value.longreal);
        RETURN TRUE;
      | longreal:
        RETURN TRUE;
      | shortint:
      | integer:
      | longint:
      | boolean:
        v.value.boolean:=v.value.longreal#0.0;
        RETURN TRUE;
      END;

    | shortint,
      integer,
      longint:
      CASE to OF
        unknown:
      | char:
      | text:
        NEW(v.value.string,20);
        co.IntToStr(v.value.longint,v.value.string^);
        RETURN TRUE;
      | real:
        v.value.real:=v.value.longint;
        RETURN TRUE;
      | longreal:
        v.value.longreal:=v.value.longint;
        RETURN TRUE;
      | shortint,
        integer,
        longint:
        RETURN TRUE;
      | boolean:
        v.value.boolean:=v.value.longint#0;
        RETURN TRUE;
      END;

    | boolean:
      CASE to OF
        unknown:
      | char:
      | text:
        IF v.value.boolean THEN
          NEW(v.value.string,5);
          COPY("true",v.value.string^);
        ELSE
          NEW(v.value.string,6);
          COPY("false",v.value.string^);
        END;
      | real:
        IF v.value.boolean THEN
          v.value.real:=1.0;
        ELSE
          v.value.real:=0.0;
        END;
        RETURN TRUE;
      | longreal:
        IF v.value.boolean THEN
          v.value.longreal:=1.0;
        ELSE
          v.value.longreal:=0.0;
        END;
        RETURN TRUE;
      | shortint,
        integer,
        longint:
        IF v.value.boolean THEN
          v.value.longint:=1;
        ELSE
          v.value.longint:=0;
        END;
        RETURN TRUE;
      | boolean:
        RETURN TRUE;
      END;
    END;

    RETURN FALSE;
  END Convert;

  (**
    (Re)set the datatype type of the model.
    All backup levels are lost.
  **)

  PROCEDURE (v : ValueModel) SetType*(type : LONGINT);

  VAR
    oldType : LONGINT;

  BEGIN
    v.value.last:=NIL;

    oldType:=v.type;
    v.type:=type;
    IF ~v.Convert(oldType,type) & ~v.value.isNull THEN
      v.value.isNull:=TRUE;
      v.Notify(NIL);
    END;
  END SetType;

  PROCEDURE (v : ValueModel) Backup*;

  VAR
    value : Value;

  BEGIN
    NEW(value);
    value^:=v.value;

    value.last:=v.value.last;
    v.value.last:=value;
  END Backup;

  PROCEDURE (v : ValueModel) Restore*;

  VAR
    value   : Value;
    changed : BOOLEAN;

  BEGIN
    value:=v.value.last;

    changed:=~v.value.Equal(value,v.type);

    v.value:=value^;
    v.value.last:=value.last;

    IF changed THEN
      v.Notify(NIL);
    END;
  END Restore;

  PROCEDURE (v : ValueModel) DropBackup*;

  BEGIN
    v.value.last:=v.value.last.last;
  END DropBackup;


  PROCEDURE (v : ValueModel) SetLongint*(value : LONGINT);

  BEGIN
    IF v.type=unknown THEN
      v.type:=longint;
      v.value.isNull:=FALSE;
      v.value.longint:=value;
      v.Notify(NIL);
    ELSIF v.type=longint THEN
      IF v.value.isNull OR (v.value.longint#value) THEN
        v.value.isNull:=FALSE;
        v.value.longint:=value;
        v.Notify(NIL);
      END;
    ELSE
      v.value.longint:=value;
      v.value.isNull:=~v.Convert(longint,v.type);
      v.Notify(NIL);
    END;
  END SetLongint;

  PROCEDURE (v : ValueModel) SetReal*(value : REAL);

  BEGIN
    IF v.type=unknown THEN
      v.type:=real;
      v.value.real:=value;
      v.value.isNull:=FALSE;
      v.Notify(NIL);
    ELSIF v.type=real THEN
      IF v.value.isNull OR (v.value.real#value) THEN
        v.value.isNull:=FALSE;
        v.value.real:=value;
        v.Notify(NIL);
      END;
    ELSE
      v.value.real:=value;
      v.value.isNull:=~v.Convert(real,v.type);
      v.Notify(NIL);
    END;
  END SetReal;

  PROCEDURE (v : ValueModel) SetLongreal*(value : LONGREAL);

  BEGIN
    IF v.type=unknown THEN
      v.type:=longreal;
      v.value.longreal:=value;
      v.value.isNull:=FALSE;
      v.Notify(NIL);
    ELSIF v.type=longreal THEN
      IF v.value.isNull OR (v.value.longreal#value) THEN
        v.value.isNull:=FALSE;
        v.value.longreal:=value;
        v.Notify(NIL);
      END;
    ELSE
      v.value.longreal:=value;
      v.value.isNull:=~v.Convert(longreal,v.type);
      v.Notify(NIL);
    END;
  END SetLongreal;

  PROCEDURE (v : ValueModel) SetString*(value : ARRAY OF CHAR);

  VAR
    old : U.Text;

  BEGIN
    old:=v.value.string;
    NEW(v.value.string,str.Length(value)+1);
    COPY(value,v.value.string^);
    IF v.type=unknown THEN
      v.type:=text;
      v.value.isNull:=FALSE;
      v.Notify(NIL);
    ELSIF v.type=text THEN
      IF v.value.isNull OR (old=NIL) OR (old^#value) THEN
        v.value.isNull:=FALSE;
        v.Notify(NIL);
      END;
    ELSE
      v.value.isNull:=~v.Convert(text,v.type);
      v.Notify(NIL);
   END;
  END SetString;

  PROCEDURE (v : ValueModel) SetText*(value : U.Text);

  VAR
    old : U.Text;

  BEGIN
    old:=v.value.string;
    v.value.string:=value;
    IF v.type=unknown THEN
      v.type:=text;
      v.value.isNull:=FALSE;
      v.Notify(NIL);
    ELSIF v.type=text THEN
      IF v.value.isNull OR (old=NIL) OR (old^#value^) THEN
        v.value.isNull:=FALSE;
        v.Notify(NIL);
      END;
    ELSE
      v.value.isNull:=~v.Convert(text,v.type);
      v.Notify(NIL);
    END;
  END SetText;

  PROCEDURE (v : ValueModel) SetBoolean*(value : BOOLEAN);

  BEGIN
    IF v.type=unknown THEN
      v.type:=boolean;
      v.value.boolean:=value;
      v.value.isNull:=FALSE;
      v.Notify(NIL);
    ELSIF v.type=boolean THEN
      IF v.value.isNull OR (v.value.boolean#value) THEN
        v.value.isNull:=FALSE;
        v.value.boolean:=value;
        v.Notify(NIL);
      END;
    ELSE
      v.value.boolean:=value;
      v.value.isNull:=~v.Convert(boolean,v.type);
      v.Notify(NIL);
    END;
  END SetBoolean;

  PROCEDURE (v : ValueModel) SetNull*;

  BEGIN
    IF ~v.value.isNull THEN
      v.value.isNull:=TRUE;
      v.Notify(NIL);
    END;
  END SetNull;

  PROCEDURE (v : ValueModel) GetShortint*():SHORTINT;

  BEGIN
    IF v.Convert(v.type,shortint) THEN
      RETURN SHORT(SHORT(v.value.longint));
    ELSE
      ASSERT(FALSE);
      RETURN 0;
    END;
  END GetShortint;

  PROCEDURE (v : ValueModel) GetInteger*():INTEGER;

  BEGIN
    IF v.Convert(v.type,integer) THEN
      RETURN SHORT(v.value.longint);
    ELSE
      ASSERT(FALSE);
      RETURN 0;
    END;
  END GetInteger;

  PROCEDURE (v : ValueModel) GetLongint*():LONGINT;

  BEGIN
    IF v.Convert(v.type,longint) THEN
      RETURN v.value.longint;
    ELSE
      ASSERT(FALSE);
      RETURN 0;
    END;
  END GetLongint;

  PROCEDURE (v : ValueModel) GetReal*():REAL;

  BEGIN
    IF v.Convert(v.type,real) THEN
      RETURN v.value.real;
    ELSE
      ASSERT(FALSE);
      RETURN 0.0;
    END;
  END GetReal;

  PROCEDURE (v : ValueModel) GetLongreal*():LONGREAL;

  BEGIN
    IF v.Convert(v.type,longreal) THEN
      RETURN v.value.longreal;
    ELSE
      ASSERT(FALSE);
      RETURN 0.0;
    END;
  END GetLongreal;

  PROCEDURE (v : ValueModel) GetText*():U.Text;

  BEGIN
    IF v.Convert(v.type,text) THEN
      RETURN v.value.string;
    ELSE
      ASSERT(FALSE);
      RETURN NIL;
    END;
  END GetText;

  PROCEDURE (v : ValueModel) GetTextCopy*():U.Text;

  VAR
    value : U.Text;

  BEGIN
    IF v.Convert(v.type,text) THEN
      NEW(value,str.Length(v.value.string^)+1);
      COPY(v.value.string^,value^);
      RETURN value;
    ELSE
      ASSERT(FALSE);
      RETURN NIL;
    END;
  END GetTextCopy;

  PROCEDURE (v : ValueModel) GetTextLength*():LONGINT;

  BEGIN
    IF v.Convert(v.type,text) THEN
      RETURN str.Length(v.value.string^);
    ELSE
      ASSERT(FALSE);
      RETURN 0;
    END;
  END GetTextLength;

  PROCEDURE (v : ValueModel) GetTextLEN*():LONGINT;

  BEGIN
    IF v.Convert(v.type,text) THEN
      RETURN LEN(v.value.string^);
    ELSE
      ASSERT(FALSE);
      RETURN 0;
    END;
  END GetTextLEN;

  PROCEDURE (v : ValueModel) GetBoolean*():BOOLEAN;

  BEGIN
    IF v.Convert(v.type,boolean) THEN
      RETURN v.value.boolean;
    ELSE
      ASSERT(FALSE);
      RETURN FALSE;
    END;
  END GetBoolean;

  PROCEDURE (v : ValueModel) IsNull*():BOOLEAN;

  BEGIN
    RETURN (v.type=unknown) OR v.value.isNull;
  END IsNull;

  (*
    boolean representation
   *)

  PROCEDURE (v : ValueModel) ToggleBoolean*;

  BEGIN
    IF ~v.value.isNull THEN
      IF v.Convert(v.type,boolean) THEN
        v.value.boolean:=~v.value.boolean;
        IF ~v.Convert(boolean,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END ToggleBoolean;

  (*
    integer representation
   *)

  PROCEDURE (v : ValueModel) Inc*;

  BEGIN
    IF ~v.value.isNull THEN
      IF v.Convert(v.type,longint) THEN
        INC(v.value.longint);
        IF ~v.Convert(longint,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Inc;

  PROCEDURE (v : ValueModel) Dec*;

  BEGIN
    IF ~v.value.isNull THEN
      IF v.Convert(v.type,longint) THEN
        DEC(v.value.longint);
        IF ~v.Convert(longint,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Dec;

  PROCEDURE (v : ValueModel) Add*(value : LONGINT);

  BEGIN
    IF ~v.value.isNull & (value#0) THEN
      IF v.Convert(v.type,longint) THEN
        INC(v.value.longint,value);
        IF ~v.Convert(longint,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Add;

  PROCEDURE (v : ValueModel) Sub*(value : LONGINT);

  BEGIN
    IF ~v.value.isNull & (value#0) THEN
      IF v.Convert(v.type,longint) THEN
        DEC(v.value.longint,value);
        IF ~v.Convert(longint,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Sub;

  (*
    string representation
   *)

  PROCEDURE (v : ValueModel) Delete*(startPos, count : LONGINT);

  BEGIN
    IF ~v.value.isNull (*& startPos<s.length*) THEN
      IF v.Convert(v.type,text) THEN

        str.Delete(v.value.string^,SHORT(startPos),SHORT(count));

        IF ~v.Convert(text,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Delete;

  PROCEDURE (v : ValueModel) Insert*(source : ARRAY OF CHAR; startPos : LONGINT);

  VAR
    help : U.Text;

  BEGIN
    IF ~v.value.isNull (*& startPos<=s.length*) THEN
      IF v.Convert(v.type,text) THEN

        NEW(help,str.Length(v.value.string^)+str.Length(source)+1);
        COPY(v.value.string^,help^);
        str.Insert(source,SHORT(startPos),help^);
        v.value.string:=help;

        IF ~v.Convert(text,v.type) THEN
          v.value.isNull:=TRUE;
        END;
        v.Notify(NIL);
      ELSE
        v.value.isNull:=TRUE;
        v.Notify(NIL);
      END;
    END;
  END Insert;

   PROCEDURE (b : BoolModel) Init*;

   BEGIN
     b.Init^;

     b.SetType(boolean);
   END Init;

   PROCEDURE (i : IntModel) Init*;

   BEGIN
     i.Init^;

     i.SetType(longint);
   END Init;

   PROCEDURE (s : StringModel) Init*;

   BEGIN
     s.Init^;

     s.SetType(text);
   END Init;

END VO:Model:Value.