(**
   a group object that puts a free defineable frame around its
   single child object.
**)

MODULE VO:FrameGroup;

(*
    Makes an frame around an object.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VO:Base:Display,
       E := VO:Base:Event,
       F := VO:Base:Frame,
       U := VO:Base:Util,
       Z := VO:Base:Size,

       G := VO:Object,
       T := VO:Text;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hSpace*,
                 vSpace* : Z.SizeDesc;
                 mode*   : SET; (* Mode for text of text label *)
               END;


  FrameGroup*     = POINTER TO FrameGroupDesc;
  FrameGroupDesc* = RECORD (G.ObjectDesc)
                      frame      : F.Frame;
                      list,
                      text       : G.Object;
                      horizSpace,
                      vertSpace  : LONGINT;
                      space      : BOOLEAN;
                    END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hSpace.Init;
    p.vSpace.Init;
    p.hSpace.SetSize(Z.unit,2);
    p.vSpace.SetSize(Z.unit,2);

    p.mode:={T.smart};
  END Init;

  PROCEDURE (f : FrameGroup) Init*;

  BEGIN
    f.Init^;

    f.SetPrefs(prefs);

    f.space:=TRUE;

    f.text:=NIL;

    NEW(f.frame);
    f.frame.Init;
    f.frame.SetFrame(F.group3D);
  END Init;

  PROCEDURE (f : FrameGroup) SetObject*(object : G.Object);

  BEGIN
    f.list:=object;
    object.SetParent(f);
  END SetObject;

  PROCEDURE (f : FrameGroup) SetFrame*(type : LONGINT);

  BEGIN
    f.frame.SetFrame(type);

    IF f.visible THEN
      f.Redraw;
    END;
  END SetFrame;

  PROCEDURE (f : FrameGroup) GetFrame*():LONGINT;

  BEGIN
    RETURN f.frame.type;
  END GetFrame;

  (**
  **)

  PROCEDURE (f : FrameGroup) SetSpace*(space : BOOLEAN);

  BEGIN
    f.space:=space;
  END SetSpace;

  PROCEDURE (f : FrameGroup) SetLabel*(text : G.Object);

  BEGIN
    f.text:=text;
    f.text.SetParent(f);
  END SetLabel;

  PROCEDURE (f : FrameGroup) SetTextLabel*(string : ARRAY OF CHAR);

  VAR
    text : T.Text;

  BEGIN
    text:=T.MakeLeftText(string);

    f.SetLabel(text);
  END SetTextLabel;

  PROCEDURE (f : FrameGroup) CalcSize*;

  BEGIN
    IF f.space THEN
      f.horizSpace:=f.prefs(Prefs).hSpace.GetSize();
      f.vertSpace:=f.prefs(Prefs).hSpace.GetSize();
    ELSE
      f.horizSpace:=0;
      f.vertSpace:=0;
    END;

    IF f.text#NIL THEN
      f.text.CalcSize;
      f.frame.SetGap(f.text.oWidth,f.text.oHeight);
    END;

    f.width:=f.frame.leftBorder+f.frame.rightBorder+f.horizSpace;
    f.height:=f.frame.topBorder+f.frame.bottomBorder+f.vertSpace;

    f.minWidth:=f.width;
    f.minHeight:=f.height;

    IF f.list#NIL THEN
      f.list.CalcSize;
      INC(f.width,f.list.oWidth);
      INC(f.height,f.list.oHeight);
      INC(f.minWidth,f.list.oMinWidth);
      INC(f.minHeight,f.list.oMinHeight);
    END;

    IF f.text#NIL THEN
      f.minWidth:=U.MaxLong(f.minWidth,f.frame.minWidth);
      f.minHeight:=U.MaxLong(f.minHeight,f.frame.minHeight);
      f.width:=U.MaxLong(f.width,f.frame.minWidth);
      f.height:=U.MaxLong(f.height,f.frame.minHeight);
    ELSE

    END;

    f.CalcSize^;
  END CalcSize;

  PROCEDURE (m : FrameGroup) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    return : G.Object;

  BEGIN
    return:=NIL;
    IF m.list#NIL THEN
      return:=m.list.GetPosObject(x,y,type);
    END;

    IF return=NIL THEN
      RETURN m.GetPosObject^(x,y,type);
    ELSE
      RETURN return;
    END;
  END GetPosObject;

  PROCEDURE (m : FrameGroup) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    return : G.Object;

  BEGIN
    return:=NIL;
    IF m.list#NIL THEN
      return:=m.list.GetDnDObject(x,y,drag);
    END;

    IF return=NIL THEN
      RETURN m.GetDnDObject^(x,y,drag);
    ELSE
      RETURN return;
    END;
  END GetDnDObject;

  PROCEDURE (m : FrameGroup) HandleMouseEvent*(event : E.MouseEvent;
                                               VAR grab : G.Object):BOOLEAN;

  BEGIN
    IF ~m.visible OR m.disabled THEN
      RETURN FALSE;
    END;

    IF m.list#NIL THEN
      RETURN m.list.HandleMouseEvent(event,grab);
    ELSE
      RETURN FALSE;
    END;
  END HandleMouseEvent;

  PROCEDURE (f : FrameGroup) Layout*;

  BEGIN
    IF f.list#NIL THEN
      f.list.Resize(f.width-f.frame.leftBorder-f.frame.rightBorder-f.horizSpace,
                    f.height-f.frame.topBorder-f.frame.bottomBorder-f.vertSpace);
      f.list.Move(f.x+(f.width-f.list.oWidth) DIV 2,
                  f.y+f.frame.topBorder+(f.height-f.frame.topBorder-f.frame.bottomBorder-f.list.oHeight) DIV 2);
    END;

    f.Layout^;
  END Layout;

  PROCEDURE (f : FrameGroup) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    f.Draw^(x,y,w,h);

    IF ~f.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=f.GetDrawInfo();

    draw.InstallClip(x,y,w,h);
    IF f.list#NIL THEN
      draw.SubRegion(f.list.oX,f.list.oY,f.list.oWidth,f.list.oHeight);
    END;
    IF f.text#NIL THEN
      draw.SubRegion(f.text.oX,f.text.oY,f.text.oWidth,f.text.oHeight);
    END;
    f.DrawBackground(f.x,f.y,f.width,f.height);
    draw.FreeLastClip;

    f.frame.Draw(f.GetDrawInfo(),f.x,f.y,f.width,f.height);

    IF f.text#NIL THEN
      f.text.Move(f.frame.gx,f.frame.gy);
    END;

    IF f.list#NIL THEN
      f.list.Draw(x,y,w,h);
    END;

    IF f.text#NIL THEN
      f.text.Draw(x,y,w,h);
    END;
  END Draw;

  PROCEDURE (f : FrameGroup) Hide*;

  BEGIN
    IF f.visible THEN
      f.frame.Hide;
      IF f.list#NIL THEN
        f.list.Hide;
      END;
      IF f.text#NIL THEN
        f.text.Hide;
      END;
      f.DrawHide;
      f.Hide^;
    END;
  END Hide;

  PROCEDURE CreateFrameGroup*():FrameGroup;

  VAR
    frameGroup : FrameGroup;

  BEGIN
    NEW(frameGroup);
    frameGroup.Init;

    RETURN frameGroup;
  END CreateFrameGroup;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:FrameGroup.