MODULE VO:Edit;



IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       V   := VO:Model:Value,

       G   := VO:Object,
       TE  := VO:Text,

       FL  := VO:EditFile,
       R   := VO:EditRun,
       T   := VO:EditText,

              Ascii,
              Err,
       co  := IntStr,
       str := Strings;

CONST
  enteredMsg * = 0;
  escapeMsg  * = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
               END;

  Line      = RECORD
                line  : R.LineRun;
                drawn : BOOLEAN;
              END;

  Lines     = POINTER TO ARRAY OF Line;


  Edit*     = POINTER TO EditDesc;
  EditDesc* = RECORD (G.ScrollableGadgetDesc)
                font        : D.Font;
                model       : T.Text;

                status      : TE.Text;

                oldModel    : T.Text;

                top         : V.ValueModel;

                hSize-,
                vSize-      : LONGINT;

                fontWidth   : LONGINT;

                lines       : Lines;
                linesBak    : Lines;

                startLine   : R.LineRun;
                endLine     : R.LineRun;

                startPos-   : LONGINT; (* textposition of first visible line *)
                endPos-     : LONGINT; (* startsPos + LEN(e.lines^) -> last possible line displayed *)

                locked,              (* do not update *)
                fast,
                readWrite   : BOOLEAN; (* do a fast update *)
              END;

VAR
  prefs : Prefs;

  (* -------- lines handling ---------- *)

  PROCEDURE (e : Edit) SetLine(line : R.LineRun; pos : LONGINT);

  BEGIN
    e.lines[pos-1].drawn:=FALSE;
    e.lines[pos-1].line:=line;
  END SetLine;

  PROCEDURE (e : Edit) IsSameLine(line : R.LineRun; pos : LONGINT):BOOLEAN;

  BEGIN
    RETURN e.lines[pos-1].line=line;
  END IsSameLine;

  PROCEDURE (e : Edit) ToggleBackup;

  VAR
    help : Lines;

  BEGIN
    help:=e.linesBak;
    e.linesBak:=e.lines;
    e.lines:=help;
  END ToggleBackup;

  PROCEDURE (e : Edit) InitText;

  VAR
    x    : LONGINT;
    line : R.LineRun;

  BEGIN
    e.startLine:=e.model.first;
    e.startPos:=1;

    line:=e.startLine;
    x:=1;
    WHILE(x<=e.vSize) & (line#NIL) DO
      e.SetLine(line,x);
      IF line#NIL THEN
        e.endLine:=line;
      END;
      line:=line.NextLine();
      INC(x);
    END;

    WHILE x<=e.vSize DO
      e.SetLine(NIL,x);
      INC(x);
    END;
    e.endPos:=e.startPos+LEN(e.lines^)-1;
  END InitText;

  (* -------- Displaying ---------- *)

  PROCEDURE (e : Edit) SetTitle;

  VAR
    buffer : ARRAY 1024 OF CHAR;
    help   : ARRAY 10 OF CHAR;

  BEGIN
    IF e.visible & (e.model#NIL) & (e.status#NIL) THEN

      buffer:="";
      co.IntToStr(e.model.cursor.x,help);
      COPY(help,buffer);
      str.Append(",",buffer);
      co.IntToStr(e.model.cursor.y,help);
      str.Append(help,buffer);

      str.Append(" (",buffer);
      co.IntToStr(e.model.lines,help);
      str.Append(help,buffer);
      str.Append(")",buffer);

      str.Append(" ",buffer);
      str.Append(e.model.name^,buffer);
      IF (e.model.fileName#NIL) & (e.model.fileName^#"") THEN
        str.Append("/",buffer);
        str.Append(e.model.fileName^,buffer);
      END;

      e.vAdjustment.Set(e.startPos,e.endPos-e.startPos+1,e.model.lines);



      e.status.SetText(buffer);

(*      e.draw.vWindow.SetTitle(buffer);*)

(*      d.window.SetScroller(d.startPos,d.endPos-d.startPos+1,d.text.lines);*)
    END;
  END SetTitle;

  (**
    Mark the given lines dirty. Coordinats are display relative.
  **)

  PROCEDURE (e : Edit) SetDirty(a,b : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=a-1 TO b-1 DO
      e.lines[x].drawn:=FALSE;
    END;
  END SetDirty;

  (**
    mark the given line (text relative y coordinate) dirty.
  **)

  PROCEDURE (e : Edit) SetDirtyLine(y : LONGINT);

  BEGIN
    IF (y>=e.startPos) & (y<=e.endPos) THEN
      e.lines[y-e.startPos].drawn:=FALSE;
    END;
  END SetDirtyLine;

  (**
    Mark the given range of lines (text relative y coordinate) dirty.
  **)

  PROCEDURE (e : Edit) SetDirtyLines(y1,y2 : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    IF (y1>e.endPos) OR (y2<e.startPos) THEN
      RETURN;
    END;

    IF y1<e.startPos THEN
      y1:=e.startPos;
    END;
    IF y2>e.endPos THEN
      y2:=e.endPos;
    END;

    FOR x:=y1 TO y2 DO
      e.lines[x-e.startPos].drawn:=FALSE;
    END;
  END SetDirtyLines;

  (**
    Mark everything as dirty.
  **)

  PROCEDURE (e : Edit) SetDirtyAll;

  BEGIN
    e.SetDirty(1,e.vSize);
  END SetDirtyAll;

  (**
    Fill line backup array.
  **)

  PROCEDURE (e : Edit) MakeLinesBackup;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO LEN(e.lines^)-1 DO
      e.linesBak[x]:=e.lines[x];
    END;
  END MakeLinesBackup;

  (* ---------- Printing --------------- *)

  (**
    Print an empty line at the given display position.
  **)

  PROCEDURE (e : Edit) PrintBlankLine(draw : D.DrawInfo; pos : LONGINT);

  BEGIN
    draw.PushForeground(e.model.parser.background);
    draw.FillRectangle(e.x,e.y+(pos-1)*e.font.height,e.width,e.font.height);
    draw.PopForeground;
  END PrintBlankLine;

  (**
    Print the given line at the given display position.
  **)

  PROCEDURE (e : Edit) PrintLine(draw : D.DrawInfo; line : R.LineRun; pos,y : LONGINT);

  VAR
    from,to,
    a,p      : LONGINT;

  BEGIN
    e.model.parser.ParseLine(y,e.hSize,line);

    from:=0;
    to:=e.hSize-1;

    (* Draw areas of equal display parameter *)
    p:=from;
    WHILE p<=to DO
      a:=p;
      WHILE (a+1<=to) & ~(e.model.parser.info[a+1].start) DO
        INC(a);
      END;
      draw.PushForeground(e.model.parser.info[p].aPen);
      draw.PushBackground(e.model.parser.info[p].bPen);

      str.Extract(e.model.parser.buffer^,SHORT(p),SHORT(a-p+1),e.model.parser.tmp^);
      draw.DrawFillString(e.x+p*e.fontWidth,
                          e.y+pos*e.font.height-e.font.descent,
                          e.model.parser.tmp^,a-p+1);

      draw.PopBackground;
      draw.PopForeground;

      p:=a+1;
    END;

    IF p*e.fontWidth<e.width THEN
      draw.PushForeground(e.model.parser.background);
      draw.FillRectangle(e.x+p*e.fontWidth,e.y+(pos-1)*e.font.height,
                         e.width-p*e.fontWidth,e.font.height);
      draw.PopForeground;
    END;
  END PrintLine;

  (**
    Show or hide the given mark at the given display position.

    We must not evaluate the position within mark itself but use the
    coordinates explicitly handed.
  **)

  PROCEDURE (e : Edit) DrawMark(draw : D.DrawInfo;
                                mark : R.Mark; x,y : LONGINT; show : BOOLEAN);

  VAR
    m      : R.Mark;
    pos,
    a,b    : LONGINT;

  BEGIN
    IF ~e.locked & e.visible THEN

      IF (y>=e.startPos) & (y<=e.endPos) & (x<=e.hSize) THEN

        pos:=y-e.startPos; (* display position *)
        m:=e.lines[pos].line.GetMarkAt(x); (* pointer to possible mark at position *)
        IF (show=FALSE) & (m#NIL) & (m.type=mark.type) THEN
          (* There is a mark with the same color at the position. *)
          RETURN;
        END;

        e.model.parser.ParseLine(y,e.hSize,e.lines[pos].line);

        a:=0;
        WHILE (a<e.hSize) & (e.model.parser.pos[a]<x) DO
          INC(a);
        END;
        b:=a+1;
        WHILE (b<e.hSize) & (e.model.parser.pos[b]<=x) DO
          INC(b);
        END;
        DEC(b);

        IF a<e.hSize THEN
          str.Extract(e.model.parser.buffer^,SHORT(a),SHORT(b-a+1),e.model.parser.tmp^);

          draw.PushForeground(e.model.parser.info[a].aPen);
          draw.PushBackground(e.model.parser.info[a].bPen);
          draw.PushFont(D.fixedFont,{});
          draw.DrawFillString(e.x+a*e.fontWidth,
                              e.y+(pos+1)*e.font.height-e.font.descent,
                              e.model.parser.tmp^,b-a+1);
          draw.PopFont;
          draw.PopForeground;
          draw.PopBackground;
        END;
      END;
    ELSE
      e.lines[y].drawn:=FALSE;
    END;
  END DrawMark;

  (**
    Refresh each to be refreshed line.
  **)

  PROCEDURE (e : Edit) Update;

  VAR
    x,lines : LONGINT;
    draw    : D.DrawInfo;

  BEGIN
    IF ~e.locked & e.visible THEN

      draw:=e.GetDrawInfo();

      draw.PushFont(D.fixedFont,{});

      lines:=LEN(e.lines^);
      FOR x:=0 TO lines-1 DO
        IF ~e.lines[x].drawn THEN
          IF e.lines[x].line=NIL THEN
            e.PrintBlankLine(draw,x+1);
          ELSE
            e.PrintLine(draw,e.lines[x].line,x+1,e.startPos+x);
          END;
          e.lines[x].drawn:=TRUE;
        END;
      END;

      IF lines*e.font.height<e.height THEN
        draw.PushForeground(e.model.parser.background);
        draw.FillRectangle(e.x,e.y+lines*e.font.height,
                           e.width,e.height-lines*e.font.height);
        draw.PopForeground;
      END;

      IF ~e.fast THEN
        e.SetTitle;
      END;

      draw.PopFont;
    END;
  END Update;

  (**
    Move the given range of lines (display relative) the given amount up
    or down on the display.

    NOTE
    Only the lines array itself will be update. All other elements must
    be updated manually.
  **)

  PROCEDURE (e : Edit) MoveRegion(draw : D.DrawInfo;
                                  start, end : LONGINT; count : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    IF count=0 THEN
      RETURN;
    END;

    IF ~e.locked & e.visible THEN
      draw.CopyArea(e.x,                                (* sX     *)
                    e.y+(start-1)*e.font.height,        (* sY     *)
                    e.width,                            (* width  *)
                    (end-start+1)*e.font.height,        (* height *)
                    e.x,                                (* dX     *)
                    e.y+(start-1+count)*e.font.height); (* dY     *)
    END;


    e.MakeLinesBackup;

    IF count<0 THEN
      FOR x:=start-1 TO end-1 DO
        e.linesBak[x].drawn:=FALSE;
        e.linesBak[x].line:=NIL;
        e.linesBak[x+count]:=e.lines[x];
      END;
    ELSE
      FOR x:=end-1 TO start-1 BY -1 DO
        e.linesBak[x].drawn:=TRUE;
        e.linesBak[x].line:=NIL;
        e.linesBak[x+count]:=e.lines[x];
      END;
    END;

    IF e.locked OR ~e.visible THEN
      FOR x:=start-1 TO end-1 DO
        e.linesBak[x+count].drawn:=FALSE;
      END;
    END;
    e.ToggleBackup;
  END MoveRegion;

  (**
    Update the display, when the given lines are deleted.
  **)

  PROCEDURE (e : Edit) UpdateDeleteLines(from, count : LONGINT);

  VAR
    x    : LONGINT;
    line : R.LineRun;

  BEGIN
    IF (from>=e.startPos) & (from<=e.endPos) THEN
      IF e.startLine.last#NIL THEN
        e.startLine:=e.startLine.last.next(R.LineRun);
      END;

      line:=e.startLine;
      x:=1;
      WHILE(x<=e.vSize) & (line#NIL) DO
        IF ~e.IsSameLine(line,x) THEN
          e.SetLine(line,x);
        END;
        IF line#NIL THEN
          e.endLine:=line;
        END;
        line:=line.NextLine();
        INC(x);
      END;
      WHILE x<=e.vSize DO
        IF ~e.IsSameLine(NIL,x) THEN
          e.SetLine(NIL,x);
        END;
        INC(x);
      END;
      e.endPos:=e.startPos+LEN(e.lines^)-1;
    END;

    IF ~e.locked & e.visible THEN
      e.Update;
    END;
  END UpdateDeleteLines;

  (**
    Update the display, when the given lines are inserted.
  **)

  PROCEDURE (e : Edit) UpdateInsertLines(from, count : LONGINT);

  VAR
    x    : LONGINT;
    line : R.LineRun;

  BEGIN
    IF (from>=e.startPos) & (from<=e.endPos) THEN
      IF e.startLine.last#NIL THEN
        e.startLine:=e.startLine.last.next(R.LineRun);
      END;

      line:=e.startLine;
      x:=1;
      WHILE(x<=e.vSize) & (line#NIL) DO
        IF ~e.IsSameLine(line,x) THEN
          e.SetLine(line,x);
        END;
        IF line#NIL THEN
          e.endLine:=line;
        END;
        line:=line.NextLine();
        INC(x);
      END;
      WHILE x<=e.vSize DO
        IF ~e.IsSameLine(NIL,x) THEN
          e.SetLine(NIL,x);
        END;
        INC(x);
      END;
      e.endPos:=e.startPos+LEN(e.lines^)-1;
    END;

    IF ~e.locked & e.visible THEN
      e.Update;
    END;
  END UpdateInsertLines;

  (* -------- positioning --------- *)

  PROCEDURE (e : Edit) Page*(lines : LONGINT);

  VAR
    next,
    help  : R.LineRun;
    count : LONGINT;
    x     : LONGINT;
    draw  : D.DrawInfo;

  BEGIN
    draw:=e.GetDrawInfo();

    (* Moving visible area *)

    count:=0;
    IF lines>0 THEN (* paging down *)
      next:=e.endLine.NextLine();
      WHILE (next#NIL) & (count<lines) DO
        e.startLine:=e.startLine.NextLine();
        INC(e.startPos);
        e.endLine:=next;
        next:=e.endLine.NextLine();
        INC(count);
      END;
    ELSE (* paging up *)
      next:=e.startLine.LastLine();
      WHILE (next#NIL) & (-count<-lines) DO
        e.startLine:=next;
        DEC(e.startPos);
        e.endLine:=e.endLine.LastLine();
        next:=e.startLine.LastLine();
        DEC(count);
      END;
    END;

    e.endPos:=e.startPos+LEN(e.lines^)-1;

    (* Updating display. Count=0 will be droped silently *)

    IF (ABS(count)>e.vSize) THEN
      help:=e.startLine;
      FOR x:=1 TO LEN(e.lines^) DO
        e.SetLine(help,x);
        help:=help.NextLine();
      END;
    ELSIF count>0 THEN
      e.MoveRegion(draw,count+1,e.vSize,-count);
      help:=e.endLine;
      FOR x:=e.vSize TO e.vSize-count+1 BY -1 DO
        e.SetLine(help,x);
        help:=help.LastLine();
      END;
    ELSIF count<0 THEN
      e.MoveRegion(draw,1,e.vSize+count,-count);
      help:=e.startLine;
      FOR x:=1 TO -count DO
        e.SetLine(help,x);
        help:=help.NextLine();
      END;
    END;

    IF count#0 THEN
      e.Update;
    END;
  END Page;

  PROCEDURE (e : Edit) ScrollTo*(lines : LONGINT);

  BEGIN
    ASSERT((lines>=1) & (lines<=e.model.lines));

    (* Convert absolute to relative coordinates *)
    lines:=lines-e.startPos;

    e.Page(lines);
  END ScrollTo;

  (**
    Returns the display coordinate of the given textline.

    If the line is not currently displayed, GetDisplayPos returns -1.
  **)

  PROCEDURE (e : Edit) TextToDisplayPos(y : LONGINT):LONGINT;

  BEGIN
    IF (y>=e.startPos) & (y<=e.endPos) THEN
      RETURN y-e.startPos+1;
    ELSE
      RETURN -1;
    END;
  END TextToDisplayPos;

  (**
    Returns the text coordinate of the given display line.
  **)

  PROCEDURE (e : Edit) DisplayToTextPos(y : LONGINT):LONGINT;

  BEGIN
    RETURN e.startPos+y-1;
  END DisplayToTextPos;

  (**
    Redisplay text in that way, that the given point can be seen
  **)

  PROCEDURE (e : Edit) MakeVisible*(x,y : LONGINT);

  VAR
    firstNext,
    lastNext,
    help       : R.LineRun;
    lines      : LONGINT;
    h          : LONGINT;
    draw       : D.DrawInfo;

  BEGIN
    (* Check if coordinate is in visible area *)
    IF (y>=e.startPos) & (y<=e.endPos) THEN
      RETURN;
    END;

    draw:=e.GetDrawInfo();

    lines:=0;
    IF y>e.endPos THEN (* Go forward *)
      firstNext:=e.startLine.NextLine();
      lastNext:=e.endLine.NextLine();
      WHILE (lastNext#NIL) & (e.endPos<y) DO
        e.startLine:=firstNext;
        INC(e.startPos);
        e.endLine:=lastNext;
        INC(e.endPos);
        INC(lines);
        firstNext:=e.startLine.NextLine();
        lastNext:=e.endLine.NextLine();
      END;
    ELSE               (* Go back *)
      firstNext:=e.startLine.LastLine();
      lastNext:=e.endLine.LastLine();
      WHILE (firstNext#NIL) & (e.startPos>y) DO
        e.startLine:=firstNext;
        DEC(e.startPos);
        e.endLine:=lastNext;
        DEC(e.endPos);
        DEC(lines);
        firstNext:=e.startLine.LastLine();
        lastNext:=e.endLine.LastLine();
      END;
    END;

    IF ABS(lines)<e.vSize THEN
      (*
        We have move less than the size of the display.
        Thus we can copy parts of the display content
        and only manually draw newly visible lines.
      *)
      IF lines>0 THEN
        e.MoveRegion(draw,lines+1,e.vSize,-lines);
        help:=e.endLine;
        FOR h:=e.vSize TO e.vSize-lines+1 BY -1 DO
          e.SetLine(help,h);
          help:=help.LastLine();
        END;
      ELSE
        e.MoveRegion(draw,1,e.vSize+lines,-lines);
        help:=e.startLine;
        FOR h:=1 TO -lines DO
          e.SetLine(help,h);
          help:=help.NextLine();
        END;
      END;
    ELSE
      (* We must completly redraw the display *)
      IF lines>0 THEN
        h:=e.vSize;
        firstNext:=e.startLine.NextLine();
        lastNext:=e.endLine.NextLine();
        WHILE (lastNext#NIL) & (h>(e.vSize DIV 2)) DO
          e.startLine:=firstNext;
          INC(e.startPos);
          e.endLine:=lastNext;
          INC(e.endPos);
          DEC(h);
          firstNext:=e.startLine.NextLine();
          lastNext:=e.endLine.NextLine();
        END;
      ELSE
        h:=1;
        firstNext:=e.startLine.LastLine();
        lastNext:=e.endLine.LastLine();
        WHILE (firstNext#NIL) & (h<(e.vSize DIV 2)) DO
          e.startLine:=firstNext;
          DEC(e.startPos);
          e.endLine:=lastNext;
          DEC(e.endPos);
          INC(h);
          firstNext:=e.startLine.LastLine();
          lastNext:=e.endLine.LastLine();
        END;
      END;

      help:=e.startLine;
      FOR h:=1 TO LEN(e.lines^) DO
        e.SetLine(help,h);
        IF help#NIL THEN
          help:=help.NextLine();
        END;
      END;
    END;

    e.Update;
  END MakeVisible;

  (**
    Make the given makr visible in the text.
  **)

  PROCEDURE (e : Edit) JumpToMark*(mark : R.Mark);

  BEGIN
    e.MakeVisible(mark.x,mark.y);
  END JumpToMark;

  PROCEDURE (e : Edit) CoordsToTextPos(cX,cY : LONGINT; VAR x,y : LONGINT):BOOLEAN;

  BEGIN
    y:=e.startPos+((cY-e.y) DIV e.font.height);

    IF (y<1) OR (y<e.startPos) OR (y>e.model.lines) OR (y>e.endPos) THEN
      RETURN FALSE;
    END;

    (*
      Since the drawing of the line is not 1:1 similar to the internal
      array of chars (a character can be wider than one char), we let the
      parser fill the draw bufffer and related structs for this line and
      look there for the X coordinate of that screen position.
    *)
    e.model.parser.ParseLine(y,e.hSize+1,e.lines[y-e.startPos].line);
    x:=e.model.parser.pos[(cX-e.x) DIV e.fontWidth];

    RETURN TRUE;
  END CoordsToTextPos;

 (* ---------- VisualOberon ----------- *)

  PROCEDURE (e : Edit) Init*;

  BEGIN
    e.Init^;

    e.SetFlags({G.canFocus});
    e.RemoveFlags({G.stdFocus});

    e.status:=NIL;

    e.model:=NIL;
    e.oldModel:=NIL;

    NEW(e.hAdjustment);
    e.hAdjustment.Init;
    e.AttachModel(e.hAdjustment.GetTopModel());

    NEW(e.vAdjustment);
    e.vAdjustment.Init;
    e.AttachModel(e.vAdjustment.GetTopModel());

    e.top:=e.vAdjustment.GetTopModel();

    e.hSize:=0;
    e.vSize:=0;

    NEW(e.lines,e.vSize);
    NEW(e.linesBak,e.vSize);

    e.locked:=FALSE;
    e.fast:=FALSE;

    e.readWrite:=TRUE;

    e.SetObjectFrame(F.none);
  END Init;

  PROCEDURE (e : Edit) SetModel*(model : O.Model);

  BEGIN
    IF e.model#NIL THEN
      e.UnattachModel(e.model);
      e.oldModel:=e.model;
    ELSE
      e.oldModel:=NIL;
    END;
    IF (model#NIL) & (model IS T.Text) THEN
      e.model:=model(T.Text);
      e.AttachModel(model);
      e.InitText;
      e.Update;
    ELSE
      e.model:=NIL;
    END;
  END SetModel;

  PROCEDURE (e : Edit) SetReadWrite*(readWrite : BOOLEAN);

  BEGIN
    e.readWrite:=readWrite;
  END SetReadWrite;

  PROCEDURE (e : Edit) SetStatusObject*(text : TE.Text);

  BEGIN
    e.status:=text;
  END SetStatusObject;

  PROCEDURE (e : Edit) CalcSize*;

  BEGIN
    e.font:=D.display.GetFont(D.fixedFont);
    e.fontWidth:=e.font.TextWidth("m",1,{});

    e.width:=5*e.fontWidth;
    e.height:=2*e.font.height;


    e.minWidth:=e.width;
    e.minHeight:=e.height;

    e.CalcSize^;
  END CalcSize;

  PROCEDURE (e : Edit) HandleMouseEvent*(event : E.MouseEvent;
                                         VAR grab : G.Object):BOOLEAN;

  VAR
    x,y : LONGINT;

  BEGIN
    IF ~e.visible OR e.disabled OR (e.model=NIL) THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) THEN
        IF e.PointIsIn(event.x,event.y) THEN
          IF (event.qualifier={}) & (event.button=E.button1) THEN
            IF e.CoordsToTextPos(event.x,event.y,x,y) THEN
              IF ~e.model.MoveCursor(x,y) THEN END;
            END;
          ELSIF (event.button=E.button4) THEN
            IF e.startPos>1 THEN
              e.ScrollTo(e.startPos-1);
            END;
          ELSIF (event.button=E.button5) THEN
            IF e.endPos<e.model.lines THEN
              e.ScrollTo(e.startPos+1);
            END;
          END;

          grab:=e;

          RETURN TRUE;
        END;
      ELSIF (event.type=E.mouseUp) & (grab#NIL) THEN
        grab:=NIL;
        RETURN TRUE;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (e : Edit) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    count,
    x,y    : LONGINT;
    buffer : ARRAY 256 OF CHAR;
    keysym : LONGINT;
    text   : U.Text;

    line   : R.LineRun;

  BEGIN
    IF e.model=NIL THEN
      RETURN FALSE;
    END;

    (* We are only interested in key down messages *)
    IF event.type#E.keyDown THEN
      RETURN FALSE;
    END;

    keysym:=event.GetKey();

    CASE keysym OF
      E.left:
        (*e.model.ClearBlock;*)
        IF e.model.MoveCursor(e.model.cursor.x-1,e.model.cursor.y) THEN END;
    | E.right:
        (*e.model.ClearBlock;*)
        IF e.model.MoveCursor(e.model.cursor.x+1,e.model.cursor.y) THEN END;
    | E.up:
        (*e.model.ClearBlock;*)
        IF event.qualifier={} THEN
          IF ~e.model.MoveCursor(e.model.cursor.x,e.model.cursor.y-1) THEN END;
        ELSIF event.qualifier=E.shiftMask THEN
          e.Page(-(e.vSize-1));
        ELSIF event.qualifier=E.controlMask THEN
          e.ScrollTo(e.startPos-1);
        END;
    | E.down:
        (*e.model.ClearBlock;*)
        IF event.qualifier={} THEN
          IF ~e.model.MoveCursor(e.model.cursor.x,e.model.cursor.y+1) THEN END;
        ELSIF event.qualifier=E.shiftMask THEN
          e.Page(e.vSize-1);
        ELSIF event.qualifier=E.controlMask THEN
          e.ScrollTo(e.startPos+1);
        END;
    | E.pageUp:
        (*e.model.ClearBlock;*)
        IF event.qualifier={} THEN
          IF e.startPos=1 THEN
            IF e.model.MoveCursor(e.model.cursor.x,1) THEN END;
          ELSE
            y:=e.TextToDisplayPos(e.model.cursor.y);
            e.Page(-(e.vSize-1));
            IF e.model.MoveCursor(e.model.cursor.x,e.DisplayToTextPos(y)) THEN END;
          END;
        ELSIF event.qualifier=E.controlMask THEN
          IF e.model.MoveCursor(1,1) THEN END;
        END;
    | E.pageDown:
        (*e.model.ClearBlock;*)
        IF event.qualifier={} THEN
          IF e.endPos=e.model.lines THEN
            IF e.model.MoveCursor(e.model.cursor.x,e.model.lines) THEN END;
          ELSE
            y:=e.TextToDisplayPos(e.model.cursor.y);
            e.Page(e.vSize-1);
            y:=e.DisplayToTextPos(y);
            IF y>e.model.lines THEN
              y:=e.model.lines;
            END;
            IF e.model.MoveCursor(e.model.cursor.x,y) THEN END;
          END;
        ELSIF event.qualifier=E.controlMask THEN
          IF e.model.MoveCursor(1,e.model.lines) THEN END;
        END;
    | E.home:
        (*e.model.ClearBlock;*)
        IF (event.qualifier={}) OR (event.qualifier=E.shiftMask) THEN
          IF e.model.MoveCursor(1,e.model.cursor.y) THEN END;
        ELSIF event.qualifier=E.controlMask THEN
          IF e.model.MoveCursor(1,e.startPos) THEN END;
        END;
    | E.end:
        (*e.model.ClearBlock;*)
        IF (event.qualifier={}) OR (event.qualifier=E.shiftMask) THEN
          line:=e.model.cursor.line;
          IF e.model.MoveCursor(line.Length()+1,e.model.cursor.y) THEN END;
        ELSIF event.qualifier=E.controlMask THEN
          IF e.model.MoveCursor(1,e.endPos) THEN END;
        END;
    | E.insert:
        IF event.qualifier=E.shiftMask THEN
          (* block paste *)
          IF e.readWrite THEN
            IF e.model.IsBlockSet()=2 THEN
              IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
                e.model.ClearBlock;
              END;
            END;
            IF e.model.InsertBuffer(e.model.cursor,T.buffer) THEN END;
          END;
        ELSIF event.qualifier=E.controlMask THEN
          (* block copy *)
          IF ~e.model.CopyAreaToBuffer(e.model.markA,e.model.markB,T.buffer) THEN
          END;
        END;
    | E.delete:
        IF e.model.IsBlockSet()=2 THEN
          IF e.readWrite THEN
            IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
              e.model.ClearBlock;
            END;
          END;
        ELSE
          IF e.readWrite THEN
            IF e.model.DeleteCharAtCursor() THEN END;
          END;
        END;
    | E.backspace:
        IF e.readWrite THEN
          IF e.model.IsBlockSet()=2 THEN
            IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
              e.model.ClearBlock;
            END;
          ELSIF e.model.MoveCursor(e.model.cursor.x-1,e.model.cursor.y) THEN
            IF e.model.DeleteCharAtCursor() THEN END;
          ELSIF ~e.model.cursor.line.IsFirstLine() THEN
            line:=e.model.cursor.line.LastLine();
            IF e.model.MoveCursor(line.Length()+1,e.model.cursor.y-1) THEN
              IF e.model.DeleteCharAtCursor() THEN END;
            END;
          ELSE
            IF e.model.DeleteCharAtCursor() THEN END;
          END;
        END;
    | E.return:
        IF e.readWrite THEN
          IF e.model.IsBlockSet()=2 THEN
            IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
              e.model.ClearBlock;
            END;
          END;
          e.model.SplitLine(e.model.cursor);
          e.MakeVisible(e.model.cursor.x,e.model.cursor.y);
        END;
    | E.f1:
        IF e.model.IsBlockSet()=2 THEN
          e.model.ClearBlock;
          e.model.SetBlockStart(e.model.cursor.x,e.model.cursor.y);
        ELSIF e.model.IsBlockSet()=1 THEN
          e.model.SetBlockEnd(e.model.cursor.x,e.model.cursor.y);
        ELSE
          e.model.SetBlockStart(e.model.cursor.x,e.model.cursor.y);
        END;
    | E.f2:
        IF event.qualifier=E.shiftMask THEN
          text:=e.model.GetText();
          Err.String(text^);
          Err.Ln;
        ELSE
          IF FL.SaveToFile(e.model,"./save.tmp")#NIL THEN
          END;
        END;
    | E.f12:
        IF event.qualifier={} THEN
          IF e.model.SetMark(e.model.cursor.x,e.model.cursor.y,"Bookmark",R.mark)#NIL THEN END;
        END;
    | E.tab:
        IF event.qualifier=E.controlMask THEN
          IF e.model=T.buffer THEN
            e.SetModel(e.oldModel);
          ELSE
            e.SetModel(T.buffer);
          END;
        ELSE
          IF e.readWrite THEN
            IF e.model.IsBlockSet()=2 THEN
              IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
                e.model.ClearBlock;
              END;
            END;
            e.model.InsertCharAtCursor(Ascii.ht);
          END;
        END;
    | E.escape:
        e.model.ClearBlock;
    ELSE
      IF e.readWrite THEN
        count:=event.GetText(buffer);
        IF count>0 THEN
          FOR x:=0 TO count-1 DO
            IF e.model.IsBlockSet()=2 THEN
              IF e.model.DeleteArea(e.model.markA,e.model.markB) THEN
                e.model.ClearBlock;
              END;
            END;
            e.model.InsertCharAtCursor(buffer[x]);
          END;
        ELSE
          event.GetName(keysym,buffer);
        END;
      END;
    END;
    RETURN FALSE;

  END HandleKeyEvent;

  PROCEDURE (e : Edit) HandleResize;

  VAR
    lines : LONGINT;
    x     : LONGINT;
    line  : R.LineRun;

  BEGIN
    lines:=e.height DIV e.font.height;
    IF lines#e.vSize THEN
      NEW(e.lines,lines);
      NEW(e.linesBak,lines);
      e.vSize:=lines;

      e.endLine:=e.startLine;
      x:=1;
      WHILE(x<e.vSize) & ~e.endLine.IsLastLine() DO
        e.endLine:=e.endLine.NextLine();
        INC(x);
      END;

      (* We want the last line of the text beeing the last line in display, if possible *)
      x:=e.vSize-x;
      WHILE (x>0) & (e.startLine.LastLine()#NIL) DO
        e.startLine:=e.startLine.LastLine();
        DEC(e.startPos);
        DEC(x);
      END;

      line:=e.startLine;
      FOR x:=1 TO LEN(e.lines^) DO
        e.SetLine(line,x);
        IF line#NIL THEN
          line:=line.NextLine();
        END;
      END;
    END;
    e.endPos:=e.startPos+LEN(e.lines^)-1;
    e.hSize:=e.width DIV e.fontWidth;

    e.vAdjustment.Set(e.startPos,e.endPos-e.startPos+1,e.model.lines);
  END HandleResize;

  PROCEDURE (e : Edit) DrawEmpty;

  VAR
    draw : D.DrawInfo;

  BEGIN
    draw:=e.GetDrawInfo();

    draw.PushForeground(e.model.parser.background);
    draw.FillRectangle(e.x,e.y,e.width,e.height);
    draw.PopForeground;
  END DrawEmpty;

  PROCEDURE (e : Edit) Draw*(x,y,w,h : LONGINT);

  BEGIN
    e.Draw^(x,y,w,h);

    IF ~e.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    e.HandleResize;

    IF e.model=NIL THEN
      e.DrawEmpty;
    ELSE
      e.SetDirtyAll;
      e.Update;
    END;

    IF e.model#NIL THEN
      e.Update;
    END;
  END Draw;

  PROCEDURE (e : Edit) Hide*;

  BEGIN
    IF e.visible THEN
      e.DrawHide;
      e.Hide^;
    END;
  END Hide;

  PROCEDURE (e : Edit) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF msg=NIL THEN
      IF model=e.top THEN
        IF ~e.top.IsNull() & (e.top.GetLongint()#e.startPos) THEN
          e.ScrollTo(e.top.GetLongint());
        END;
      ELSIF model=e.model THEN
        e.SetDirtyAll; (* all visible lines are dirty *)
        e.Update;
      END;
    ELSE
      WITH
        msg : T.RedrawMsg DO
          IF msg.from=msg.to THEN
            e.SetDirtyLine(msg.from);
          ELSE
            e.SetDirtyLines(msg.from,msg.to);
          END;
          e.Update;
      | msg : T.MarkMsg DO
          e.DrawMark(e.GetDrawInfo(),msg.mark,msg.oldX,msg.oldY,FALSE);
          e.DrawMark(e.GetDrawInfo(),msg.mark,msg.newX,msg.newY,TRUE);
          e.JumpToMark(e.model.cursor);
          IF ~e.fast THEN
            e.SetTitle;
          END;
      | msg : T.DeleteLinesMsg DO
          e.UpdateDeleteLines(msg.from,msg.count);
      | msg : T.InsertLinesMsg DO
          e.UpdateInsertLines(msg.from,msg.count);
      ELSE
      END;
    END;
  END Resync;

  PROCEDURE CreateEdit*():Edit;

  VAR
    edit : Edit;

  BEGIN
    NEW(edit);
    edit.Init;

    RETURN edit;
  END CreateEdit;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Edit.