(**
  Implements some classes for drawing object backgrounds in various ways.
**)

MODULE VO:Base:Background;

(*
    Implements some classes for drawing object backgrounds in various ways.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This file is part of VisualOberon.

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VO:Base:Display,

       PP := VO:Prefs:Parser,

       G  := VO:Object,
       I  := VO:Image;

TYPE
  PlainBackground*       = POINTER TO PlainBackgroundDesc;
  PlainBackgroundDesc*   = RECORD (G.BackgroundDesc)
                             color : D.Color;
                           END;

  ImageBackground*       = POINTER TO ImageBackgroundDesc;
  ImageBackgroundDesc*   = RECORD (G.BackgroundDesc)
                             image : I.Image;
                           END;

  TileBackground*        = POINTER TO TileBackgroundDesc;
  TileBackgroundDesc*    = RECORD (G.BackgroundDesc)
                             image : I.Image;
                           END;

  InheritBackground*     = POINTER TO InheritBackgroundDesc;
  InheritBackgroundDesc* = RECORD (G.BackgroundDesc)
                           END;

  PROCEDURE (p : PlainBackground) Init*(color : D.Color);

  BEGIN
    p.color:=color;
  END Init;

  PROCEDURE (p : PlainBackground) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);

  BEGIN
    draw.PushForeground(p.color);
    draw.FillRectangle(x,y,w,h);
    draw.PopForeground;
  END Draw;

  PROCEDURE (p : PlainBackground) Copy*():G.Background;

  VAR
    copy : PlainBackground;

  BEGIN
    NEW(copy);
    copy^:=p^;

    RETURN copy;
  END Copy;

  PROCEDURE (i : ImageBackground) Init*(filename : ARRAY OF CHAR);

  BEGIN
    NEW(i.image);
    i.image.Init;
    i.image.SetFlags({G.horizontalFlex,G.verticalFlex});
    i.image.SetFilename(filename);
  END Init;

  PROCEDURE (i : ImageBackground) CalcSize*;

  BEGIN
    i.image.CalcSize;
  END CalcSize;

  PROCEDURE (i : ImageBackground) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);

  BEGIN
    i.image.Resize(i.source.oWidth,i.source.oHeight);
    i.image.DrawPart(x-i.source.x,y-i.source.y,w,h,x,y,draw);
  END Draw;

  PROCEDURE (b : ImageBackground) Copy*():G.Background;

  VAR
    copy : ImageBackground;

  BEGIN
    NEW(copy);
    copy^:=b^;

    RETURN copy;
  END Copy;

  PROCEDURE (i : ImageBackground) Free*;

  BEGIN
    IF i.image#NIL THEN
      i.image.Free;
      i.image:=NIL;
    END;
  END Free;

  PROCEDURE (t : TileBackground) Init*(filename : ARRAY OF CHAR);

  BEGIN
    NEW(t.image);
    t.image.Init;
    t.image.SetFilename(filename);
  END Init;

  PROCEDURE (t : TileBackground) CalcSize*;

  BEGIN
    t.image.CalcSize;
  END CalcSize;

  PROCEDURE (t : TileBackground) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);

  BEGIN
    t.image.DrawTiled(x,y,w,h,0,0,draw);
  END Draw;

  PROCEDURE (t : TileBackground) Copy*():G.Background;

  VAR
    copy : TileBackground;

  BEGIN
    NEW(copy);
    copy^:=t^;

    RETURN copy;
  END Copy;

  PROCEDURE (t : TileBackground) Free*;

  BEGIN
    IF t.image#NIL THEN
      t.image.Free;
      t.image:=NIL;
    END;
  END Free;

  PROCEDURE (i : InheritBackground) Init*();

  BEGIN
  END Init;

  PROCEDURE (i : InheritBackground) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);

  BEGIN
    IF i.source.parent#NIL THEN
      i.source.parent.DrawBackground(x,y,w,h);
    ELSE
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(x,y,w,h);
      draw.PopForeground;
    END;
  END Draw;

  PROCEDURE (i : InheritBackground) Copy*():G.Background;

  VAR
    copy : InheritBackground;

  BEGIN
    NEW(copy);
    copy^:=i^;

    RETURN copy;
  END Copy;

  PROCEDURE LoadBackground*(top : PP.Item):G.Background;

  VAR
    type,
    data : ARRAY 256 OF CHAR;

    i    : ImageBackground;
    ih   : InheritBackground;
    t    : TileBackground;
    p    : PlainBackground;
    pos  : LONGINT;

  BEGIN
    top:=top.GetEntry("background");
    IF top=NIL THEN
      RETURN NIL;
    END;

    IF top.GetStringEntry("type",type) THEN
      IF type="image" THEN
        IF top.GetStringEntry("name",data) THEN
          NEW(i);
          i.Init(data);
          RETURN i;
        END;
      ELSIF type="inherit" THEN
        NEW(ih);
        ih.Init;
        RETURN ih;
      ELSIF type="imageTile" THEN
        IF top.GetStringEntry("name",data) THEN
          NEW(t);
          t.Init(data);
          RETURN t;
        END;
      ELSIF type="plain" THEN
        IF top.GetStringEntry("color",data) THEN
          pos:=D.GetColorEntry(data);
          IF pos>=0 THEN
            NEW(p);
            p.Init(pos);
            RETURN p;
          END;
        END;
      END;
    END;
    RETURN NIL;
  END LoadBackground;

  PROCEDURE SaveBackground*(top : PP.BlockItem; background : G.Background);

  VAR
    block : PP.BlockItem;
    value : PP.ValueItem;

  BEGIN
    IF background#NIL THEN
      NEW(block);
      block.Init;
      block.SetName("background");
      top.AddItem(block);
      WITH
        background : ImageBackground DO
          NEW(value);
          value.Init;
          value.SetValue("type","image");
          block.AddItem(value);
          NEW(value);
          value.Init;
          value.SetValue("name",background.image.filename^);
          block.AddItem(value);
      | background : TileBackground DO
          NEW(value);
          value.Init;
          value.SetValue("type","imageTile");
          block.AddItem(value);
          NEW(value);
          value.Init;
          value.SetValue("name",background.image.filename^);
          block.AddItem(value);
      | background : PlainBackground DO
          NEW(value);
          value.Init;
          value.SetValue("type","plain");
          block.AddItem(value);
          NEW(value);
          value.Init;
          value.SetValue("color",D.colorNames[background.color]);
          block.AddItem(value);
      | background : InheritBackground DO
          NEW(value);
          value.Init;
          value.SetValue("type","inherit");
          block.AddItem(value);
      END;
    END;
  END SaveBackground;

END VO:Base:Background.