/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: IPAddress.cc,v 1.3 2001/12/19 12:46:40 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/IPAddress.hh>
#include <fwbuilder/Interface.hh>

#include <stdio.h>

#ifndef __MINGW32__
# include <netinet/in.h>
#endif

using namespace std;
using namespace libfwbuilder;

IPAddress::IPAddress(const unsigned char *data, size_t len) throw(FWException)
{
    if(len!=4)
        throw FWException("Invalid IP address length. Must be 4.");
    if(!data)
        throw FWException("NULL IP address data..");
    for(int i=0;i<4;i++)
        octets[i]=data[i];
}

IPAddress::IPAddress(const struct in_addr *na) throw(FWException)
{
    /*
      struct in_addr {
      unsigned long int s_addr;
      }
      
      Note that on the i80x86 the host byte order is Least  Sig
      nificant  Byte  first,  whereas the network byte order, as
      used on the Internet, is Most Significant Byte first.
    */

    unsigned long x = ntohl(na->s_addr);
    for(int i=3;i>=0;i--)
    {
        octets[i]=x&0377;
        x>>=8;
    }
}

IPAddress::IPAddress()
{
    octets[0]=octets[1]=octets[2]=octets[3]=0;
}

IPAddress::operator ulong() const
{
    unsigned long x=0l;
    for(int i=3;i>=0;i--)
    {
        x<<=8;
        x|=octets[i]&0377;
    }
    return x;
}

bool libfwbuilder::operator<(const IPAddress &a, const IPAddress &b)
{
    for(int i=0;i<4;i++)
        if(a.octets[i]!=b.octets[i])
            return a.octets[i]<b.octets[i];
    return false;
}

bool libfwbuilder::operator==(const IPAddress &a, const IPAddress &b)
{
    for(int i=0;i<4;i++)
        if(a.octets[i]!=b.octets[i])
            return false;
    return true;
}

bool libfwbuilder::operator==(const Netmask &a, const Netmask &b)
{
    for(int i=0;i<4;i++)
        if(a.octets[i]!=b.octets[i])
            return false;
    return true;
}


IPAddress::IPAddress(const string &s) throw(FWException, FWNotSupportedException)
{
    *this=s;
}

IPAddress& IPAddress::operator=(const string &s) throw(FWException, FWNotSupportedException)
{
    //TODO: do actual parsing according to RFC1884
    if(s.find(":")!=string::npos && s.find_first_not_of(":0123456789ABCDEFabcdef")==string::npos)
        throw FWNotSupportedException("IPv6 addresses are not supported");
    
    if(sscanf(s.c_str(), "%3u.%3u.%3u.%3u", &octets[0], &octets[1], &octets[2], &octets[3])!=4)
        throw FWException(string("Invalid IP address: '")+s+"'");
    validate();
    return *this;
}

void IPAddress::validate() throw(FWException)
{
    if(!octets[0] && !octets[1] && !octets[2] && !octets[3])
        return; //0.0.0.0 magic IP address.

    if((octets[0]<0 || octets[0]>255) ||
       (octets[1]<0 || octets[1]>255) ||
       (octets[2]<0 || octets[2]>255) ||
       (octets[3]<0 || octets[3]>255)) 
    {
        throw FWException(string("Invalid IP address: '")+string(*this)+"'");
    }
}

IPAddress::IPAddress(const IPAddress &o)
{
    *this=o;
}

IPAddress& IPAddress::operator=(const IPAddress &o)
{
    octets[0]=o.octets[0];
    octets[1]=o.octets[1];
    octets[2]=o.octets[2];
    octets[3]=o.octets[3];
    return *this;
}

string IPAddress::toString() const
{
    char buf[32];
    sprintf(buf, "%u.%u.%u.%u", octets[0], octets[1], octets[2], octets[3]);
    return buf;
}

void Netmask::validate() throw(FWException)
{
    unsigned long nm = octets[3] | (octets[2]<<8) | (octets[1]<<16) | (octets[0]<<24);
    
    if(nm) 
    {
        nm = (~nm)+1;
        
        // at this point nm must consist 
        // of exactly one '1' in binary form 
        
        while(!(nm&1))
            nm>>=1;

        if(nm!=1)
            throw FWException(string("Invalid netmask: ")+string(*this));
    }
}

Netmask& Netmask::operator=(const string &s) throw(FWException)
{
    if(sscanf(s.c_str(), "%3u.%3u.%3u.%3u", &octets[0], &octets[1], &octets[2], &octets[3])!=4)
        throw FWException(string("Invalid netmask: ")+s);
    validate();
    return *this;
}

Netmask::Netmask(const IPAddress &a)
{
    octets[0]=255;
    octets[1]=octets[2]=octets[3]=0;
    
    if(a[0]>127) octets[1]=255;
    if(a[0]>191) octets[2]=255;
}

Netmask::Netmask(const unsigned char *data, size_t len) throw(FWException):IPAddress(data, len) 
{
}

Netmask::Netmask()
{
    octets[0]=octets[1]=octets[2]=octets[3]=255;
}

Netmask::Netmask(const string &s) throw(FWException)
{
    *this=s;
}

IPNetwork::IPNetwork(const IPAddress &a, const Netmask &n, int _bcast_bits)
{    
    bcast_bits = _bcast_bits;
    netmask = n;
    for(unsigned i=0;i<4;i++)
        address.octets[i]=a[i]&netmask[i];
}

const IPAddress& IPNetwork::getAddress() const
{
    return address;
}

const Netmask&   IPNetwork::getNetmask() const
{
    return netmask;
}

IPAddress IPNetwork::getBroadcastAddress () const
{
    ulong a;
    if(bcast_bits)
        a=(ulong)address | ~((ulong)netmask);     
    else 
        a=(ulong)address & (ulong)netmask;
    struct in_addr na;
    na.s_addr=htonl(a);
    return IPAddress(&na);
}

bool IPNetwork::belongs(const IPAddress &o) const
{
    for(unsigned i=0;i<4;i++) 
        if ( ( o[i] & netmask[i] ) != address[i] )
            return false;

    return true;
}

bool IPNetwork::isBroadcast() const
{
    return (address[0]==address[1] && 
            address[1]==address[2] && 
            address[2]==address[3] && 
            (address[0]==255 || address[0]==0)
    );
}

bool IPNetwork::isMulticast() const
{
    return (address[0]>=224 && 
            address[0]<=239 && 
            address[1]==0   && 
            address[2]==0   && 
            address[3]==0
    );
}

IPNetwork& IPNetwork::operator=(const IPNetwork &o)
{
    address=o.address;
    netmask=o.netmask;
    return *this;
}

bool libfwbuilder::operator==(const IPNetwork &a, const IPNetwork &b)
{
    return a.getNetmask()==b.getNetmask() && a.getAddress()==b.getAddress();
}

bool libfwbuilder::operator<(const IPNetwork &a, const IPNetwork &b)
{
    return a.getAddress()<b.getAddress();
}


IPRoute::IPRoute(const IPRoute &o)
{
    nm     = o.nm     ;
    dst    = o.dst    ;
    gw     = o.gw     ;
    intf   = o.intf?new Interface(o.intf):NULL ;
    direct = o.direct ;
}

IPRoute::~IPRoute()
{
    delete intf;
}

IPRoute::IPRoute(const IPAddress &_dst, const Netmask &_nm, const IPAddress &_gw, const Interface *_intf,  bool _direct)
{
    nm     = _nm     ;
    dst    = _dst    ;
    gw     = _gw     ;
    intf   = _intf?new Interface(_intf):NULL ;
    direct = _direct ;
}


    
