/** -*- C++ -*-
    @file predicate/predicate.h
    @author Peter Rockai <me@mornfall.net>
    @author Enrico Zini

    Facilities used to filter packages.
    Example use:
    \code
    // Construct predicate statically
    using namespace aptFront::predicate;
    Predicate<Package> p = nameMatch("lib") and descriptionMatch("shared");

    // Construct predicate dynamically
    predicate::Predicate<Package> p = predicate::True();
    if (!name_f.empty())
        p = p and predicate::nameMatch<Package>(name_f);
    if (!desc_f.empty())
        p = p and predicate::descriptionMatch<Package>(desc_f);

    // Create a filtered range that holds the filtered packages
    utils::Range<Package> matches = filteredRange(
        utils::Range<Package>(range(packages.packagesBegin(),
            packages.packagesEnd())), p);

    // Output the contents of the range to a consumer
    matches.output(consumer);
    \endcode

    \par Adding new search functions
    (to be written)

    \par Adding new generators
    (to be written)
*/

#include <vector>
#include <string>
#include <ept/forward.h>
#include <wibble/range.h>
#include <wibble/amorph.h>
#include <execinfo.h>

#ifndef EPT_PREDICATE_H
#define EPT_PREDICATE_H

namespace ept {
namespace predicate {

// swig would choke without the _
template< typename _ > struct Compound;

template< typename T >
struct PredicateInterface {
    virtual bool test( const T & ) = 0;
    virtual Compound< T > compound() const = 0;
    virtual ~PredicateInterface() {}
};

template< typename T, typename Self >
struct PredicateMixin : wibble::mixin::Comparable< Self > {
    typedef Self PredicateImplementation;
    typedef T argument_type;
    typedef bool result_type;
    const Self &self() const { return *static_cast< const Self * >( this ); }
    Self &self() { return *static_cast< const Self * >( this ); }
    bool operator<=( const Self &o ) const { return this <= &o; }
    bool operator()( const T &t ) { return self().test( t ); }
    Compound< T > compound() const;
    PredicateMixin() {
        /* std::cerr << "~PredicateMixin(), backtrace:" << std::endl;
		const int trace_size = 20;
		void *addrs[trace_size];
		size_t size = backtrace (addrs, trace_size);
		char **strings = backtrace_symbols (addrs, size);
		for (size_t i = 0; i < size; i++)
            std::cerr << "   " << strings[i] << std::endl;
            free( strings ); */
    }
};

template< typename T >
struct CompoundInterface {
    virtual Predicate< T > first() const = 0;
    virtual Predicate< T > second() const = 0;
    virtual Predicate< T > substitute( Predicate< T >, Predicate< T > ) const = 0;
    virtual ~CompoundInterface() {}
};

#ifndef SWIG
template< typename T, template< typename, typename, typename > class Self,
          typename P1, typename P2 >
struct CompoundMixin: PredicateMixin< T, struct Self< T, P1, P2 > > {
    typedef Self< T, P1, P2 > CompoundImplementation;

    CompoundMixin( const P1 &p1, const P2 &p2 )
        : m_p1( p1 ), m_p2( p2 )
    {}

    bool operator<=( const Self< T, P1, P2 > &o ) const {
        if ( first() < o.first() ) return true;
        if ( first() > o.first() ) return false;
        return second() <= o.second();
    }

    Predicate< T > substitute( Predicate< T > f, Predicate< T > s ) const {
        return Self< T, Predicate< T >, Predicate< T > >( f, s );
    }

    Compound< T > compound() const { return this->self(); }
    P1 first() const { return m_p1; }
    P2 second() const { return m_p2; }
protected:
    P1 m_p1;
    P2 m_p2;
};
#endif

template< typename T, typename W >
struct CompoundMorph : wibble::Morph< CompoundMorph< T, W >, W, CompoundInterface< T > >
{
    CompoundMorph( const W &w ) : wibble::Morph< CompoundMorph, W, CompoundInterface< T > >( w ) {}
    virtual Predicate< T > first() const { return this->wrapped().first(); }
    virtual Predicate< T > second() const { return this->wrapped().second(); }
    Predicate< T > substitute( Predicate< T > f, Predicate< T > s ) const {
        return this->wrapped().substitute( f, s );
    };
};

template< typename T, typename W >
struct PredicateMorph : wibble::Morph< PredicateMorph< T, W >, W, PredicateInterface< T > >
{
    PredicateMorph( const W &w ) :
        wibble::Morph< PredicateMorph, W, PredicateInterface< T > >( w ) {}
    virtual bool test( const T &t ) { return this->wrapped()( t ); }
    Compound< T > compound() const { return this->wrapped().compound(); }
};

template< typename T >
struct Predicate : wibble::Amorph< Predicate< T >, PredicateInterface< T > >,
    PredicateMixin< T, Predicate< T > >
{
    typedef wibble::Amorph< Predicate< T >, PredicateInterface< T > > Super;

    template< typename C >
    Predicate( const C &i,
               typename wibble::IsType< int, typename C::PredicateImplementation >::T fake = 0 )
        : Super( PredicateMorph< T, C >( i ) ) { (void)fake; }
    Predicate() {}

    bool operator<=( const Predicate &o ) const { return leq( o ); }
    bool test( const T &t ) { return this->implementation()->test( t ); }
    Compound< T > compound() const { return this->implementation()->compound(); }
};

template< typename T >
struct Compound : wibble::Amorph< Compound< T >, CompoundInterface< T > >
{
    typedef wibble::Amorph< Compound< T >, CompoundInterface< T > > Super;

    template< typename C >
    Compound( const C &i,
               typename wibble::IsType< int, typename C::CompoundImplementation >::T fake = 0 )
        : Super( CompoundMorph< T, C >( i ) ) { (void)fake; }
    Compound() {}

    bool operator<=( const Compound &o ) const { return leq( o ); }
    Predicate< T > first() const { return this->implementation()->first(); }
    Predicate< T > second() const { return this->implementation()->second(); }
    Predicate< T > substitute( Predicate< T > f, Predicate< T > s ) const {
        return this->implementation()->substitute( f, s );
    };
};

template< typename T >
struct Void : PredicateMixin< T, Void< T > > {
    bool operator<=( const Void & ) const { return true; }
    bool operator()( const T & ) { throw 0; } // fixme?
};

template< typename T >
struct True : PredicateMixin< T, True< T > > {
    bool operator<=( const True & ) const { return true; }
    bool operator()( const T & ) { return true; }
};

template<typename Pred>
Predicate< typename Pred::argument_type > predicate( Pred p ) {
    return p;
}

template< typename T, typename S >
Compound< T > PredicateMixin< T, S >::compound() const {
    return Compound< T >();
}

}
}

// vim:set ts=4 sw=4:
#endif
