/***************************************************************************
                           czlib.cpp  -  description
                             -------------------
    begin                : Thu Oct 12 2004
    copyright            : (C) 2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "czlib.h"

/** */
CZLib::CZLib()
{
	m_bInit = FALSE;
}

/** */
CZLib::~CZLib()
{
	if ( m_bInit )
		inflateEnd(&m_ZStream);
}

/** */
bool CZLib::InflateZBlock( char * in, int * inlen, char * out, int * outlen )
{
	bool res = FALSE;

	int i;
	
	if ( m_bInit == FALSE )
	{
		m_bInit = TRUE;

		m_ZStream.next_in  = 0;
		m_ZStream.avail_in = 0;
		m_ZStream.opaque   = 0;
		m_ZStream.zalloc   = 0;
		m_ZStream.zfree    = 0;

		inflateInit(&m_ZStream);
	}
	
	m_ZStream.next_in   = (Bytef*)in;
	m_ZStream.avail_in  = *inlen;
	m_ZStream.next_out  = (Bytef*)out;
	m_ZStream.avail_out = *outlen;
	m_ZStream.total_out = 0;
	
	i = inflate(&m_ZStream,Z_SYNC_FLUSH);

	if ( (i == Z_OK) || (i == Z_STREAM_END) )
	{
		*outlen = m_ZStream.total_out;
		*inlen  = *inlen - m_ZStream.avail_in;
		
		if ( i == Z_STREAM_END )
		{
			inflateEnd(&m_ZStream);
			m_bInit = FALSE;
		}
		
		res = TRUE;
	}
	else
	{
		inflateEnd(&m_ZStream);
		m_bInit = FALSE;
		*outlen = 0;
	}
/*
	printf("INFLATE: %d %d %d %d\n",
		i,
		m_ZStream.avail_in,
		m_ZStream.avail_out,
		m_ZStream.total_out );
*/
	return res;
}

#define MAX_BUFFER_LEN	536870912	// 512MiB max memory

/** */
unsigned long CZLib::OneShotInflate( const char * in, const unsigned long inlen, CByteArray * out )
{
	int res = 0;
	unsigned long outlen = inlen;
	char * buffer = 0;
	
	while (TRUE)
	{
		outlen *= 2;
		
		if ( buffer )
		{
			free(buffer);
		}
		
		if ( outlen < MAX_BUFFER_LEN )
		{
			buffer = (char*) malloc(outlen);
		}
		
		if (!buffer)
		{
			printf("CZLib::OneShotInflate: malloc failed\n");
			outlen = 0;
			break;
		}
		
		res = uncompress( (Bytef*)buffer, &outlen, (const Bytef*)in, inlen );
		
		if ( res == Z_OK )
		{
			out->Append( (const unsigned char*)buffer, outlen );
			break;
		}
		else if ( res == Z_DATA_ERROR ) // corrupt source data
		{
			//printf("CZLib::OneShotInflate: corrupt data\n");
			outlen = 0;
			break;
		}
		else if ( res == Z_MEM_ERROR ) // not enoungh memory
		{
			//printf("CZLib::OneShotInflate: zlib out of memory\n");
			outlen = 0;
			break;
		}
		else if ( res == Z_BUF_ERROR ) // buffer too small
		{
			// nothing, try again
		}
	}
	
	if ( buffer )
	{
		free(buffer);
	}
	
	return outlen;
}

/** */
CDeflater::CDeflater()
{
	m_bInit = FALSE;
}

/** */
CDeflater::~CDeflater()
{
	if ( m_bInit )
	{
		int ret = deflateEnd(&m_ZStream);
		
		if ( ret == Z_DATA_ERROR )
		{
			printf("CDeflater::~CDeflater: warning: data discarded!\n");
		}
	}
}

/** */
bool CDeflater::DeflateBlock( char * in, int * inlen, char * out, int * outlen, bool more )
{
	bool res = FALSE;
	
	int ret, flush;
	
	// if first block
	if ( m_bInit == FALSE )
	{
		m_bInit = TRUE;
		
		m_ZStream.next_in   = 0;
		m_ZStream.avail_in  = 0;
		m_ZStream.next_out  = 0;
		m_ZStream.avail_out = 0;
		m_ZStream.opaque    = 0;
		m_ZStream.zalloc    = Z_NULL;
		m_ZStream.zfree     = Z_NULL;
		
		deflateInit( &m_ZStream, Z_DEFAULT_COMPRESSION );
	}
	
	m_ZStream.next_in   = (Bytef*)in;
	m_ZStream.avail_in  = *inlen;
	m_ZStream.next_out  = (Bytef*)out;
	m_ZStream.avail_out = *outlen;
	m_ZStream.total_out = 0;
	
	if ( more == TRUE )
	{
		flush = Z_NO_FLUSH;
	}
	else
	{
		flush = Z_FINISH;
	}
	
	ret = deflate( &m_ZStream, flush );
	
	if ( more == TRUE )
	{
		if ( ret == Z_OK )
		{
			*outlen = m_ZStream.total_out;
			*inlen = *inlen - m_ZStream.avail_in;
			res = TRUE;
		}
		else
		{
			deflateEnd(&m_ZStream);
			m_bInit = FALSE;
			*outlen = 0;
		}
	}
	else
	{
		if ( res == Z_OK )
		{
			*outlen = m_ZStream.total_out;
			*inlen = *inlen - m_ZStream.avail_in;
			res = TRUE;
		}
		if ( ret == Z_STREAM_END )
		{
			*outlen = m_ZStream.total_out;
			*inlen = *inlen - m_ZStream.avail_in;
			res = FALSE;
			
			deflateEnd(&m_ZStream);
			m_bInit = FALSE;
		}
		else
		{
			*outlen = 0;
			
			deflateEnd(&m_ZStream);
			m_bInit = FALSE;
		}
	}
	
	//printf( "DEFLATE: %d %d %d %d\n", ret, m_ZStream.avail_in, m_ZStream.avail_out, m_ZStream.total_out );
	
	return res;
}

/** */
unsigned long CDeflater::OneShotDeflate( const char * in, const unsigned long inlen, CByteArray * out )
{
	int res = 0;
	unsigned long outlen = inlen;
	char * buffer = 0;
	
	while (TRUE)
	{
		outlen *= 2;
		
		if ( buffer )
		{
			free(buffer);
		}
		
		if ( outlen < MAX_BUFFER_LEN )
		{
			buffer = (char*) malloc(outlen);
		}
		
		if ( !buffer )
		{
			printf("CDeflater::OneShotDeflate: malloc failed\n");
			outlen = 0;
			break;
		}
		
		res = compress( (Bytef*)buffer, &outlen, (const Bytef*)in, inlen );
		
		if ( res == Z_OK )
		{
			out->Append( (const unsigned char*)buffer, outlen );
			break;
		}
		else if ( res == Z_MEM_ERROR ) // not enoungh memory
		{
			printf("CDeflater::OneShotDeflate: zlib out of memory\n");
			outlen = 0;
			break;
		}
		else if ( res == Z_BUF_ERROR ) // buffer too small
		{
			// nothing, try again
		}
	}
	
	if ( buffer )
	{
		free(buffer);
	}
	
	return outlen;
}
