//---------------------------------------------------------------------
//  Algorithmic Conjurings @ http://www.coyotegulch.com
//
//  fixed.h (libcoyotl)
//---------------------------------------------------------------------
//
//  Copyright 1990-2004 Scott Robert Ladd
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the
//      Free Software Foundation, Inc.
//      59 Temple Place - Suite 330
//      Boston, MA 02111-1307, USA.
//
//-----------------------------------------------------------------------
//
//  For more information on this software package, please visit
//  Scott's web site, Coyote Gulch Productions, at:
//
//      http://www.coyotegulch.com
//  
//-----------------------------------------------------------------------

#if !defined(LIBCOYOTL_FIXED_H)
#define LIBCOYOTL_FIXED_H

// Std. C Library
#include <cstdlib>
#include <stdint.h>

// Std. C++ Library
#include <string>

namespace libcoyotl
{
    // this class assumes a 32-bit IEEE-754/IEC-60559 floating-point type
    template < int32_t DP = 16 >
    class fixed
    {
        public:
            // constructors
            fixed();

            // copy constructor
            fixed(const fixed<DP> & source);

            // assignment constructors
            fixed(const float & source);
            fixed(const int32_t & source);

            // assignment
            void operator = (const fixed<DP> & source);
            void operator = (const int32_t & source);

            // conversion to float
            operator float();

            // get minimum and maximum possible values
            static fixed get_max_value();
            static fixed get_min_value();

            // get the raw value
            int32_t get_value();

            // unary negation
            fixed operator - ();

            // binary operators
            fixed operator + (const fixed<DP> & f);
            fixed operator - (const fixed<DP> & f);
            fixed operator * (const fixed<DP> & f);
            fixed operator / (const fixed<DP> & f);

            // shorthand operators
            fixed<DP> & operator += (const fixed<DP> & f);
            fixed<DP> & operator -= (const fixed<DP> & f);
            fixed<DP> & operator *= (const fixed<DP> & f);
            fixed<DP> & operator /= (const fixed<DP> & f);

            // comparison operators
            bool operator == (const fixed<DP> & f);
            bool operator != (const fixed<DP> & f);
            bool operator >= (const fixed<DP> & f);
            bool operator <= (const fixed<DP> & f);
            bool operator >  (const fixed<DP> & f);
            bool operator <  (const fixed<DP> & f);

            // absolute value
            fixed abs(const fixed<DP> & f);

        private:
            // integer containing fixed-point math
            int32_t m_value;

            // utility routines for multiplication and division
            int32_t mul(int32_t a, int32_t b);
            int32_t div(int32_t a, int32_t b);
    };

    template < int32_t DP >
    inline int32_t fixed<DP>::mul(int32_t a, int32_t b)
    {
        int64_t result = (int64_t)a * (int64_t)b;
        return (int32_t)(result >> DP);
    }

    template < int32_t DP >
    inline int32_t fixed<DP>::div(int32_t a, int32_t b)
    {
        int32_t result = 0x80000001;

        if (b != 0)
        {
            int32_t sign = 0x80000000 & a;

            if (sign)
                a = -a;

            result = ((int64_t)a << DP) / (int64_t)b;

            if (sign)
                result = -result;
        }

        return result;
    }

    template < int32_t DP >
    inline fixed<DP>::fixed()
      : m_value(0)
    {
        // nada
    }

    template < int32_t DP >
    inline fixed<DP>::fixed(const fixed<DP> & source)
      : m_value(source.m_value)
    {
        // nada
    }

    template < int32_t DP >
    fixed<DP>::fixed(const float & source)
      : m_value(0)
    {
        // assign pointer to avoid memcpy
        int32_t * f_as_int = (int32_t *)(&source);

        // get exponent
        int32_t f_exponent = ((*f_as_int) & 0x7f800000) >> 23;

        // if exponent is zero, so is the m_value
        if (f_exponent != 0)
        {
            // adjust exponent for bias and fixed decimal point
            f_exponent -= 127 + DP;

            // extract significand (mantissa)
            int32_t f_significand = ((*f_as_int) & 0x007fffff) + 0x00800000;

            // chose shift direction based on value of exponent
            if (f_exponent <= 23)
                m_value = f_significand >> (23 - f_exponent);
            else
                m_value = f_significand << (f_exponent - 23);

            // adjust for sign
            if ((*f_as_int) & 0x80000000)
                m_value = -m_value;
        }
    }

    template < int32_t DP >
    inline fixed<DP>::fixed(const int32_t & source)
      : m_value(source)
    {
        // nada
    }

    template < int32_t DP >
    inline void fixed<DP>::operator = (const fixed<DP> & source)
    {
        m_value = source.m_value;
    }

    template < int32_t DP >
    inline void fixed<DP>::operator = (const int32_t & source)
    {
        m_value = source;
    }

    template < int32_t DP >
    fixed<DP>::operator float()
    {
        // start with zero
        int32_t f = 0;

        // only compute for non-zero values
        if (m_value != 0)
        {
            // handle the sign first
            if (m_value < 0)
            {
                m_value = -m_value;
                f |= 0x80000000;
            }

            // compute exponent
            int32_t edx = 157 - DP;
            int32_t ebx = 0x40000000;

            while (!(ebx & m_value))
            {
                --edx;
                ebx >>= 1;
            }

            // store exponent
            f |= (edx << 23);

            // move bits for significand
            edx = 0x400000;

            while (true)
            {
                ebx >>= 1;

                if (ebx == 0)
                    break;

                if (m_value & ebx)
                    f |= edx;

                edx >>= 1;
            }
        }

        // use cast to force return of integer as a float
        return *((float *)&f);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::get_max_value()
    {
        return fixed(0x7fffffffL);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::get_min_value()
    {
        return fixed(0x80000001L);
    }

    template < int32_t DP >
    inline int32_t fixed<DP>::get_value()
    {
        return m_value;
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::operator - ()
    {
        return fixed(-m_value);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::operator + (const fixed<DP> & f)
    {
        return fixed(m_value + f.m_value);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::operator - (const fixed<DP> & f)
    {
        return fixed(m_value - f.m_value);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::operator * (const fixed<DP> & f)
    {
        return fixed(mul(m_value,f.m_value));
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::operator / (const fixed<DP> & f)
    {
        return fixed(div(m_value,f.m_value));
    }

    template < int32_t DP >
    inline fixed<DP> & fixed<DP>::operator += (const fixed<DP> & f)
    {
        m_value += f.m_value;
        return *this;
    }

    template < int32_t DP >
    inline fixed<DP> & fixed<DP>::operator -= (const fixed<DP> & f)
    {
        m_value -= f.m_value;
        return *this;
    }

    template < int32_t DP >
    inline fixed<DP> & fixed<DP>::operator *= (const fixed<DP> & f)
    {
        m_value = mul(m_value,f.m_value);
        return *this;
    }

    template < int32_t DP >
    inline fixed<DP> & fixed<DP>::operator /= (const fixed<DP> & f)
    {
        m_value = div(m_value,f.m_value);
        return *this;
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator == (const fixed<DP> & f)
    {
        return (m_value == f.m_value);
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator != (const fixed<DP> & f)
    {
        return (m_value != f.m_value);
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator >= (const fixed<DP> & f)
    {
        return (m_value >= f.m_value);
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator <= (const fixed<DP> & f)
    {
        return (m_value <= f.m_value);
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator >  (const fixed<DP> & f)
    {
        return (m_value > f.m_value);
    }

    template < int32_t DP >
    inline bool fixed<DP>::operator <  (const fixed<DP> & f)
    {
        return (m_value < f.m_value);
    }

    template < int32_t DP >
    inline fixed<DP> fixed<DP>::abs(const fixed<DP> & f)
    {
        return fixed((m_value < 0) ? -m_value : m_value);
    }
    
}

#endif
