/* ------------------------------------------------------------------------
 * $Id: Zorn.cc,v 1.5 2001/06/29 15:00:43 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-18 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <stdio.h>
#include <iostream>

// -- Zorn Includes
#include "Store.hh"
#include "EvalVisitor.hh"
#include "Zorn.hh"

// -- Function Prototypes
extern int zorn_parse();
extern void zorn_init(const char *filename, void (*output)(const char *msg));
extern FILE *yyin;

static void defaultErrorOutput(const char *msg);

// -- Code Segment

Zorn::Interpreter::Interpreter()
    : _initialized(true),
      _functionset(FunctionSet::instance()),
      _ruleset(RuleSet::instance()),
      _store(new Store()),
      _evaluator(new EvalVisitor(_store)),
      _output(defaultErrorOutput)
{
    // empty
}

Zorn::Interpreter::~Interpreter()
{
    // Delete allocated objects
    delete _evaluator;
    delete _store;
}

bool Zorn::Interpreter::parse(const std::string &filename)
{
    // Open the desired file
    if ((yyin = fopen(filename.c_str(), "r")) == NULL)
	return false;

    // Now, initialize the parser and parse the input file!
    zorn_init(filename.c_str(), _output);
    zorn_parse();

    // Remember to close the file, too
    fclose(yyin);

    // Mark the interpreter as uninitialized (might be pending inits
    // to evaluate)
    _initialized = false;

    return true;
}

void Zorn::Interpreter::addFunctionObject(FunctionObject *fobj)
{
    // Add the function object to the function set
    _functionset->addFunctionObject(fobj);
}

void Zorn::Interpreter::evaluate(const std::vector<NamedValue> &premises)
{
    // Are there any pending inits to evaluate?
    if (!_initialized) {

	// Yes, there are, evaluate them and then remove them
	_ruleset->visitAllInits(_evaluator);
	_ruleset->clearAllInits();
	_initialized = true;
    }

    // Define the premises in the store
    for (std::vector<NamedValue>::const_iterator i = premises.begin();
	 i != premises.end(); i++)
	_store->set(i->name(), i->value());
    
    // Evaluate the rules using the evaluation visitor
    _ruleset->visitAllRules(_evaluator);

    // Remove the premises again
    for (std::vector<NamedValue>::const_iterator i = premises.begin();
	 i != premises.end(); i++)
	_store->remove(i->name());    
}

void defaultErrorOutput(const char *msg)
{
    // Output the error string to standard error
    std::cerr << "Zorn:: " << msg << endl;
}
