/* ------------------------------------------------------------------------
 * $Id: RendererImpl.cc,v 1.33 2001/08/27 15:10:28 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-07 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000, 2001 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- OpenGL Includes
#include <GL/gl.h>
#include <GL/glu.h>

// -- Local Includes
#include "Celsius/debug.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Logger.hh"
#include "Nobel/Node.hh"
#include "Polhem/ImageImpl.hh"
#include "Polhem/GeometryImpl.hh"
#include "Polhem/LineGeometryImpl.hh"
#include "Polhem/PointGeometryImpl.hh"
#include "Polhem/TriangleGeometryImpl.hh"
#include "Polhem/RendererImpl.hh"

using namespace Nobel;

// -- Constants
const float default_fov = 60.0f;
const float default_near = 1.0f;
const float default_far = 1000.0f;

const char *renderer_version = "0.2";

// -- Code Segment

inline int triangleIndex(const IndexSeq &indexList, int n, int c)
{
    return indexList[n * 3 + c];
}

inline int lineIndex(const IndexSeq &indexList, int n, int c)
{
    return indexList[n * 2 + c];
}

RendererImpl::RendererImpl(int width, int height)
    : _width(width), _height(height)
{
    // Enable the depth buffer
    glEnable(GL_DEPTH_TEST);
    
    // Set buffer clear color
    glClearColor(0.0, 0.0, 0.0, 1.0);
    
    // Enable backface culling
    glEnable(GL_CULL_FACE);

    // Set up lighting
    glShadeModel(GL_SMOOTH);
    //glPolygonMode(GL_FRONT, GL_LINE);
    glEnable(GL_LIGHTING);
    
    // Add light zero and configure it
    glEnable(GL_LIGHT0);

    // Configure materials
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

    // Reset the viewport and configure the projection
    glViewport(0, 0, (GLsizei) _width, (GLsizei) _height);
    setProjection(default_fov, (float) width / (float) height, default_near,
		  default_far);
}

RendererImpl::~RendererImpl()
{
    // empty
}

void RendererImpl::setProjection(float fov, float aspect,
				 float near, float far)
{
    // Update project parameters
    _fov = fov;
    _aspect = aspect;
    _near = near;
    _far = far;
    
    // Set up a new projection mode
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(_fov, (GLfloat) _aspect, _near, _far);
    
    // Make sure we are in model-view matrix mode
    glMatrixMode(GL_MODELVIEW);
}

std::string RendererImpl::getName() const
{
    std::string name = "3Dwm Default OpenGL Renderer (OpenGL: ";
    name += (const char *) glGetString(GL_RENDERER);
    name += ")";
    return name;
}

std::string RendererImpl::getVendor() const
{
    std::string name = "http://www.3dwm.org (OpenGL: ";
    name += (const char *) glGetString(GL_VENDOR);
    name += ")";
    return name; 
}

std::string RendererImpl::getVersion() const
{
    std::string name = renderer_version;
    name += " (OpenGL: ";
    name += (const char *) glGetString(GL_VERSION);
    name += ")";
    return name; 
}

void RendererImpl::visit(Node_ptr n)
{
    n->render(Renderer_var(_this()));
}

void RendererImpl::renderPoints(const Nobel::PointMesh &mesh)
{
    glBegin(GL_POINTS);
    for (unsigned int i = 0; i < mesh.vertexList.length(); i++) {
	Vertex3D vtx = mesh.vertexList[i];
	glVertex3fv((GLfloat *) &vtx);
    }
    glEnd();
}

void RendererImpl::renderLines(const Nobel::LineMesh &mesh)
{
    int lineNum = mesh.vertexIndexList.length() / 2;
    glBegin(GL_LINES);
    for (int i = 0; i < lineNum; i++) {
	for (int j = 0; j < 2; j++) {
	    const Vertex3D &vtx =
		mesh.vertexList[lineIndex(mesh.vertexIndexList, i, j)];
	    glVertex3fv((GLfloat *) &vtx);
	}
    }
    glEnd();
}


void RendererImpl::renderTriangles(const Nobel::TriangleMesh &mesh)
{ 
    // @@@ Need to make this much more foolproof!
    
    // Compute the number of triangles to render
    int triangleNum = mesh.vertexIndexList.length() / 3;
    
    // Figure out which data we were supplied
    bool normals = mesh.normalList.length() != 0 &&
	mesh.normalIndexList.length() >= mesh.vertexIndexList.length();
    bool texcoords = mesh.texCoordList.length() != 0 &&
	mesh.texCoordIndexList.length() >= mesh.vertexIndexList.length();

    /*
    cerr << " #v: " << mesh.vertexList.length() 
	 << " #n: " << mesh.normalList.length()
	 << " #tc: " << mesh.texCoordList.length()
	 << " #vi: " << mesh.vertexIndexList.length()
	 << " #ni: " << mesh.normalIndexList.length()
	 << " #tci: " << mesh.texCoordIndexList.length()
	 << endl;
    */

    if (!normals) glEnable(GL_NORMALIZE);
    
    // Start emitting triangles
    glBegin(GL_TRIANGLES);
    
    // Loop through all triangles in the mesh
    for (int i = 0; i < triangleNum; i++) {
	
	// Three vertices to each triangle
	for (int j = 0; j < 3; j++) {
	    
	    // Did we get texture coordinates?
	    if (texcoords) {

		// Yes, we did, so apply it
		int tc_index = triangleIndex(mesh.texCoordIndexList, i, j);
		const TexCoord &tc = mesh.texCoordList[tc_index];
		glTexCoord2fv((GLfloat *) &tc);
	    }
	    
	    // Did we get normals?
	    if (normals) {
		
		// Yes, we did, so apply them
		int n_index = triangleIndex(mesh.normalIndexList, i, j);
		const Vertex3D &normal = mesh.normalList[n_index];
		glNormal3fv((GLfloat *) &normal);
	    }
	    
	    // Finally, send the vertex to OpenGL
	    int vtx_index = triangleIndex(mesh.vertexIndexList, i, j);
	    const Vertex3D &vtx = mesh.vertexList[vtx_index];
	    glVertex3fv((GLfloat *) &vtx);
	}
    }

    // Stop rendering triangles
    glEnd();

    if (!normals) glDisable(GL_NORMALIZE);
}

void RendererImpl::identityTransform()
{
    glLoadIdentity();
    //st_current.flags |= st_transform;
}

void RendererImpl::multiplyTransform(Transform_ptr t)
{
    // First, multiply transforms as usual
    VisitorImpl::multiplyTransform(t);

    Matrix3D m;
    m = _trafos.top()->matrix();
    
    // Premultiply to the view matrix
    m = m * _view;
    m.transpose();

    // Set it into OpenGL
    glLoadMatrixf(m.data());
}

void RendererImpl::save()
{
    VisitorImpl::save();
    
    // Push the current state to the stack and then clear it
    st_stack.push(st_current);
    st_current.flags = 0;
}

void RendererImpl::restore()
{ 
    VisitorImpl::restore();

    // If the transform stack is empty, load identity matrix
    if (_trafos.empty() == true) 
	glLoadIdentity(); // SteveH - I'm not sure we need to do this
    else {
	Matrix3D m = _trafos.top()->matrix();
	m.transpose();
	glLoadMatrixf(m.data());
    }
    
    // Retrieve the old state and pop it off the stack  
    st_current = st_stack.top();
    st_stack.pop();
    // @@@ Need to reinstate state here! 
} 

//
// updateTexture - called by Procedural Textures to load new texture data
//
void RendererImpl::updateTexture(const Rectangle &rect,
				 PixelType texFormat, 
				 const BinaryData &texData,
				 CORBA::Long texName)
{
    GLvoid *pixels;

    pixels = (GLvoid *) texData.get_buffer();

    glBindTexture(GL_TEXTURE_2D, texName);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
 
    if (texFormat == Nobel::RGB888) {
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, (GLsizei) rect.w, 
		     (GLsizei) rect.h, 0, GL_RGB, GL_UNSIGNED_BYTE, 
		     (GLubyte *) pixels);
    }
    else if (texFormat == Nobel::RGBA8888) {
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, (GLsizei) rect.w, 
		     (GLsizei) rect.h, 0, GL_RGBA, GL_UNSIGNED_BYTE, 
		     (GLubyte *) pixels);
    }
    
    GLenum err = glGetError();
    if (err != GL_NO_ERROR)
	DPRINTF("GL Error updating texture: %s\n", gluErrorString(err));
}

void RendererImpl::applyTexture(CORBA::Long texName)
{
    // TODO: cache tex state so we don't keep enabling unnecessarily
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, (GLuint)texName);
}

void RendererImpl::applyMaterial(const MaterialAttributes& ma)
{
    /*
    // Set the material state
    glMaterialfv(GL_FRONT, GL_AMBIENT, (GLfloat *) &ma.ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, (GLfloat *) &ma.diffuse);
    glMaterialfv(GL_FRONT, GL_EMISSION, (GLfloat *) &ma.emissive);
    glMaterialfv(GL_FRONT, GL_SPECULAR, (GLfloat *) &ma.specular);
    glMaterialf(GL_FRONT, GL_SHININESS, ma.shininess);
    */
    glColor4fv((GLfloat *) &ma.diffuse);
}

CORBA::Long RendererImpl::getMaxTexSize()
{
    GLint maxSize;
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &maxSize);
    return (CORBA::Long)maxSize;
}

void RendererImpl::disableTexture()
{
    glDisable(GL_TEXTURE_2D);
}

//
// Generate a single texture with the supplied pixel data
//
CORBA::Boolean RendererImpl::genTexture(CORBA::Long width,
					CORBA::Long height, 
					PixelType texFormat, 
					const BinaryData &texData, 
					CORBA::Long_out texName)
{
    GLenum err;
    GLuint tmpName;
    GLvoid *pixels;

    pixels = (GLvoid*) texData.get_buffer();

    DPRINT("Renderer::genTexture\n");    
    DUMPMEM(pixels, 256);
    DPRINT("\n");
    err = glGetError(); // Clear the error flag

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glGenTextures(1, &tmpName);

    // write CORBA out value
    texName = tmpName;

    err = glGetError();
    if (err != GL_NO_ERROR)
	return false;
    
    glBindTexture(GL_TEXTURE_2D, tmpName);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    // Currently we support 888RGB and 8888RGBA
    // May add 565RGB later
    if (texFormat == Nobel::RGB888) {
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, (GLsizei)width, 
		     (GLsizei)height, 0, GL_RGB, GL_UNSIGNED_BYTE, 
		     (GLubyte*) pixels);
    }
    else if (texFormat == Nobel::RGBA8888) {
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, (GLsizei)width, 
		     (GLsizei)height, 0, GL_RGBA, GL_UNSIGNED_BYTE, 
		     (GLubyte*) pixels);
    }

    // Make sure the texture was loaded successfully
    err = glGetError();
    if (err != GL_NO_ERROR)
	return false;

    return true;
}
