/* ------------------------------------------------------------------------
 * $Id: EventManager.cc,v 1.10 2001/07/18 07:48:37 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-11-28 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm includes
#include "Celsius/Mutex.hh"
#include "Celsius/Vector3D.hh"
#include "Celsius/Logger.hh"
#include "Polhem/View.hh"
#include "Polhem/Platform.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/EventManager.hh"

using namespace Nobel;

// -- Local Function Prototypes
static void loggerOutput(const char *msg);

// -- Code Segment

EventManager::EventManager()
    : _quit(false), _interpreter(new Zorn::Interpreter())
{
    // Configure error output function for the interpreter
    _interpreter->setErrorOutput(loggerOutput);
}

EventManager::~EventManager()
{
    delete _interpreter;
}

void EventManager::run()
{    
    Logger::log(LogGroup::Event) 
	<< "Event manager starting with " << _interpreter->getRuleCount()
	<< " rules in rule database." << std::endl;
	
    // Loop until the user exits
    while (_quit == false) {

	// Wait indefinitely for an event
	InputEvent event = Platform::getNextInputEvent();

	// Let the Zorn interpreter evaluate it
	_interpreter->evaluate(event);
    }
}

void EventManager::addEventMappingFunctions(Zorn::FunctionObject *fobj)
{
    // Add the event mapping function object to the interpreter
    _interpreter->addFunctionObject(fobj);
}

void EventManager::loadEventMappingRuleFile(const std::string &filename)
{
    Logger::log(LogGroup::Event)
	<< "Parsing rule file \"" << filename << "\"..." << std::endl;
    
    // Load the event mapping rule file (ignoring errors)
    if (_interpreter->parse(filename) == false) 
	Logger::log(LogGroup::Event)
	    << "Error loading rule file!" << std::endl;
    else 
	Logger::log(LogGroup::Event) << "Parsing completed." << std::endl;
}

void loggerOutput(const char *msg)
{
    // Output errors using the logging service
    Logger::log(LogGroup::Event) << msg << std::endl;
}
