/* ------------------------------------------------------------------------
 * $Id: Exception.hh,v 1.1 2001/08/16 11:32:21 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * This C++ SAX implementation (Simple API for XML) is derived from
 * David Megginson's original Java SAX that can be found on 
 * http://www.megginson.com/SAX/.
 * ------------------------------------------------------------------------
 * File created 2001-08-14 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------ 
 */

#ifndef _SAXException_hh_
#define _SAXException_hh_

// -- System Includes
#include <string>
#include <exception>

// -- Local Includes
#include "Celsius/SAX/LocatorImpl.hh"

namespace SAX {

    // -- Forward Declarations

    // -- Class Declarations

    /**
     * Encapsulate a general SAX error or warning. 
     **/
    class Exception : public std::exception {
    public:
	
	/**
	 * Constructor.
	 *
	 * @param error or warning message.
	 **/
	Exception(const string &msg) : _msg(msg), _e(0) {}
	
	/**
	 * Constructor. Create a new SAXException wrapping an existing
	 * exception.
	 *
	 * @param e exception to be wrapped in a SAXException.
	 **/
	Exception(std::exception &e) : _e(&e) { }

	/**
	 * Constructor. The existing exception will be embedded in the
	 * new one, but the new exception will have its own message.
	 *
	 * @param msg detail message.
	 * @param e exception to be wrapped in a SAXException.
	 **/
	Exception(const string &msg, std::exception &e) : _msg(msg), _e(&e) { }
	
	/**
	 * Destructor.
	 **/
	virtual ~Exception() { }
	
	/**
	 * Retrieve the exception message.
	 *
	 * @return exception detail message.
	 **/
	virtual const char *what() const {
	    return _msg.empty() ? _e->what() : _msg.c_str();
	}
	
	/**
	 * Retrieve the embedded exception.
	 * 
	 * @return pointer to the embedded exception
	 **/
	std::exception *getException() const {
	    return _e;
	}
	
    private:
	std::string _msg;
	std::exception *_e;	
    };

    /**
     * Encapsulate an XML parse error or warning. 
     **/
    class ParseException : public Exception {
    public:
	
	/**
	 * Constructor. Create a new SAXParseException from a message
	 * and a Locator.
	 *
	 * @param message error or warning message.
	 * @param locator locator object for the error or warning.
	 **/
	ParseException(const std::string &message, const Locator &locator)
	    : Exception(message), _locator(locator) { }

	/**
	 * Constructor. Wrap an existing exception in a
	 * SAXParseException.
	 *
	 * @param message error or warning message, empty to use the
	 *                message from the embedded exception.
	 * @param locator locator object for the error or warning.
	 * @param e exception to be wrapped in a SAXParseException.
	 **/
	ParseException(const std::string &message, const Locator &locator,
		       std::exception &e)
	    : Exception(message, e), _locator(locator) { }

	/**
	 * Create a new SAXParseException.
	 *
	 * @param message error or warning message.
	 * @param publicId public identifer of the entity that
	 *                 generated the error or warning.
	 * @param systemId system identifer of the entity that
	 *                 generated the error or warning.
	 * @param lineNumber line number of the end of the text that
	 *                   caused the error or warning.
	 * @param columnNumber column number of the end of the text
	 *                     that caused the error or warning.
	 **/
	ParseException(const std::string &message,
		       const std::string &publicId,
		       const std::string &systemId,
		       int lineNumber, int columnNumber) 
	    : Exception(message),
              _locator(publicId, systemId, lineNumber, columnNumber) { }

	/**
	 * Create a new SAXParseException.
	 *
	 * @param message error or warning message, empty to use the
	 *                message from the embedded exception.
	 * @param publicId public identifer of the entity that
	 *                 generated the error or warning.
	 * @param systemId system identifer of the entity that
	 *                 generated the error or warning.
	 * @param lineNumber line number of the end of the text that
	 *                   caused the error or warning.
	 * @param columnNumber column number of the end of the text
	 *                     that caused the error or warning.
	 * @param e exception to be wrapped in a SAXParseException.
	 **/
	ParseException(const std::string &message,
		       const std::string &publicId,
		       const std::string &systemId,
		       int lineNumber, int columnNumber, std::exception &e)
	    : Exception(message, e),
              _locator(publicId, systemId, lineNumber, columnNumber) { }
	
	/**
	 * Return the public identifier for the parse exception.
	 *
	 * @return string containing the public identifier, empty if
	 *         none is available.
	 **/
	const std::string &getPublicId() const {
	    return _locator.getPublicId(); 
	}
	
	/**
	 * Return the system identifier for the parse exception. 
	 *
	 * @return string containing the system identifier, empty if
	 *         none is available.
	 **/
	const std::string &getSystemId() const {
	    return _locator.getSystemId();
	}

	/**
	 * Return the line number where the parse exception ends. 
	 *
	 * @return line number, or -1 if none is available.
	 **/
	int getLineNumber() const {
	    return _locator.getLineNumber();
	}

	/**
	 * Return the column number where the parse exception ends.
	 * 
	 * @return column number, or -1 if none is available.
	 **/
	int getColumnNumber() const {
	    return _locator.getColumnNumber();
	}
	
    private:
	LocatorImpl _locator;
    };
};
	
#endif /* Exception.hh */
