/* ------------------------------------------------------------------------
 * $Id: Plane3D.hh,v 1.6 2001/08/02 14:22:07 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Plane3D_hh_
#define _Plane3D_hh_

// -- Celsius Includes
#include "Line.hh"
#include "Matrix3D.hh"
#include "Vector3D.hh"

/**
 * General three-dimensional plane class. Represented by the [A, B, C,
 * D] components in the conventional plane equation, i.e.
 *
 * 	A * x + B * y + C * z + D = 0
 * 
 * that holds true for all points [x, y, z] on the plane.
 **/
class Plane3D {
public:
    
    /**
     * Constructor.
     *
     * @param a value of A parameter.
     * @param d value of B parameter.
     * @param c value of C parameter.
     * @param d value of D parameter.
     **/
    Plane3D(float a = 0, float b = 0, float c = 0, float d = 0)
	: _a(a), _b(b), _c(c), _d(d) { }

    /**
     * Constructor. Creates the plane from a normal vector
     * (representing the facing of the plane) and a point on the
     * plane.
     *
     * @param n plane normal.
     * @param p point on the plane.
     **/
    Plane3D(const Vector3D &n, const Vector3D &p);

    /**
     * Constructor. Creates the plane from a triangle. The plane
     * parameters will be computed from the given triangle. The
     * triangle vertices are by default given in a counter-clockwise
     * winding; use the "ccw" parameter to change this behavior.
     *
     * @param p1 first triangle vertex.
     * @param p2 second triangle vertex.
     * @param p3 third triangle vertex.
     * @param ccw flags whether vertex winding is counter-clockwise or not.
     **/
    Plane3D(const Vector3D &p1, const Vector3D &p2, const Vector3D &p3,
	    bool ccw = true);

    /**
     * Destructor.
     **/
    ~Plane3D() { }
    
    /**
     * Classify a point against the plane. If the point is in front of
     * the plane, this function will return a positive value, if it is
     * behind, the value will be negative. Finally, if the point is on
     * the plane, the value will be zero. Note that you must take
     * numerical robustness into consideration when using this
     * function, so planes should ideally have a "thickness".
     *
     * @param p point to classify against the plane.
     * @return classification value (= a * x + b * y + c * z + d).
     **/
    float classify(const Vector3D &p) const;

    /**
     * Compute a plane/line intersection and return the intersection
     * point. If the line and plane are coincident, there is no single
     * intersection point. 
     *
     * @param p1 starting point of line.
     * @param p2 ending point of line.
     * @return intersection point (if any).
     **/
    Vector3D intersect(const Vector3D &p1, const Vector3D &p2) const;
    
    /**
     * Compute a plane/line intersection and return the line parameter
     * value for the intersection. If the line and plane are
     * coincident, there is no single intersection point.
     *
     * @param line starting point of line.
     * @return intersection parameter (if any).
     **/    
    float intersect(const Line<Vector3D> &line) const;
    
    /*
     * Retrieve plane parameters (read-only version).
     */ 
    float a() const { return _a; } 
    float b() const { return _b; }
    float c() const { return _c; }
    float d() const { return _d; }

    /*
     * Retrieve plane parameters.
     */ 
    float &a() { return _a; } 
    float &b() { return _b; }
    float &c() { return _c; }
    float &d() { return _d; }
    
    /**
     * Retrieve the plane normal.
     *
     * @return vector representing the normal of the plane.
     **/
    Vector3D getNormal() const {
	return Vector3D(_a, _b, _c);
    }

    /**
     * Flip the orientation of the plane. This is done simply by
     * flipping the plane normal (a, b, c).
     **/
    void flip() {
	_a = -_a; _b = -_b; _c = -_c;
    }

    /**
     * Transform the plane with a transformation matrix. The
     * transformation matrix should be inverted and transposed prior
     * to calling this method.
     *
     * @param inv_trafo matrix to transform the plane with (inverted
     *        and transposed).
     **/
    void transform(const Matrix3D &inv_trafo);

    /**
     * Plane output operator.
     *
     * @param f output stream to use.
     * @param p plane to output.
     * @return modified output stream.
     **/
    friend std::ostream & operator << (std::ostream &f, const Plane3D &p);
    
private:
    float _a, _b, _c, _d;
};

// -- Inlines

inline float Plane3D::classify(const Vector3D &p) const
{
    // Compute the dot product between the plane and the point
    return a() * p.x() + b() * p.y() + c() * p.z() + d(); 
}

inline Vector3D Plane3D::intersect(const Vector3D &p1,
				   const Vector3D &p2) const
{
    // Compute the direction vector
    Vector3D v = p2 - p1;
    v.normalize();

    // @@@need to watch out for singularities here! (= dot product zero)

    // Compute the intersection point between ray and vector and
    // return it
    float t = - classify(p1) / Vector3D::dot(getNormal(), v);
    return Vector3D(p1 + t * v);
}

inline float Plane3D::intersect(const Line<Vector3D> &line) const
{
    // Compute the intersection point between ray and vector and
    // return it
    return - classify(line.p()) /
	Vector3D::dot(getNormal(), line.getDirection());
}

inline std::ostream & operator << (std::ostream &f, const Plane3D &p)
{
    f << "[(" << p.a() << ", " << p.b() << ", " << p.c() << "), " << p.d() 
      << "]";
    return f;
}

#endif /* Plane3D.hh */ 
