/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.ResourceBundle;

import com.redhat.rhdb.vise.resources.ActionResources;

/**
 * GenericAction factory. Using objects of this class, one can get
 * wide variety of actions. All properties of available actions are
 * set in a resource file. Settings for a given action may be modified
 * after obtaining it from the factory. The image directory used for
 * storing icon files for the actions is can be set through a
 * constructor call when creating an ActionFactory or later via {@link
 * #setImageDirectory}. If the image directory is set to null,
 * graphics won't be used.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 *
 * @see com.redhat.rhdb.vise.GenericAction
 */
public class ActionFactory {
	private String imagedir;
	private ResourceBundle res;

	/** Default action. */
	public static final int ACTION_DEFAULT = -1;

	/** 'New' action. */
	public static final int ACTION_NEW = 0;

	/** 'Save' action. */
	public static final int ACTION_SAVE = 1;

	/** 'Open' action. */
	public static final int ACTION_OPEN = 2;

	/** 'Cut' action. */
	public static final int ACTION_CUT = 3;

	/** 'Copy' action. */
	public static final int ACTION_COPY = 4;

	/** 'Paste' action. */
	public static final int ACTION_PASTE = 5;

	/** 'About' action. */
	public static final int ACTION_ABOUT = 6;

	/** 'Exit' action. */
	public static final int ACTION_EXIT = 7;

	/** 'Explain' action. */
	public static final int ACTION_EXPLAIN = 8;

	/** 'Preferences' action. */
	public static final int ACTION_PREFERENCES = 9;

	/** 'Zoom In' action. */
	public static final int ACTION_ZOOMIN = 10;

	/** 'Zoom Out' action. */
	public static final int ACTION_ZOOMOUT = 11;

	/** 'Save As...' action. */
	public static final int ACTION_SAVEAS = 12;

	/** 'Delete' action. */
	public static final int ACTION_DELETE = 13;

	/** 'Edit' action. */
	public static final int ACTION_EDIT = 14;

	/** 'Ok' action. */
	public static final int ACTION_OK = 15;

	/** 'Cancel' action. */
	public static final int ACTION_CANCEL = 16;

	/** 'Manual' action. */
	public static final int ACTION_MANUAL = 17;

	/** 'Test' action. */
	public static final int ACTION_TEST = 18;

	/** 'Refresh' action. */
	public static final int ACTION_REFRESH = 19;

	/** 'Continue' action. */
	public static final int ACTION_CONTINUE = 20;

	/** 'Back' action. */
	public static final int ACTION_BACK = 21;

	/** 'Forward action. */
	public static final int ACTION_FORWARD = 22;

	/** 'Home' action. */
	public static final int ACTION_HOME = 23;

	/** 'Explain Analyze' action. */
	public static final int ACTION_ANALYZE = 24;

	/** 'Stop' action. */
	public static final int ACTION_STOP = 25;

	/** 'Databases' action. */
	public static final int ACTION_DATABASES = 26;

	/** 'PlannerOptions' action. */
	public static final int ACTION_PLANNEROPTIONS = 27;

	/**
	 * Creates a new <code>ActionFactory</code> instance. The image
	 * directory to be used is set to null.
	 */
	public ActionFactory()
	{
		res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");
		this.imagedir = null;
	}

	/**
	 * Creates a new <code>ActionFactory</code> instance using a given
	 * image directory.
	 *
	 * @param imagedir a <code>String</code> value
	 */
	public ActionFactory(String imagedir)
	{
		this();

		setImageDirectory(imagedir);
	}

	/**
	 * Get a certain action. The type of action to get is one of the
	 * constants defined in this class.
	 *
	 * @param type an <code>int</code> value
	 * @return a <code>GenericAction</code> value
	 */
	public GenericAction getAction(int type)
	{
		GenericAction genac;
		
		switch (type)
		{
			case ACTION_NEW:
				genac = new NewAction();
				break;
			case ACTION_SAVE:
				genac = new SaveAction();
				break;
			case ACTION_SAVEAS:
				genac = new SaveAsAction();
				break;
			case ACTION_OPEN:
				genac = new OpenAction();
				break;
			case ACTION_CUT:
				genac = new CutAction();
				break;
			case ACTION_COPY:
				genac = new CopyAction();
				break;
			case ACTION_PASTE:
				genac = new PasteAction();
				break;
			case ACTION_ABOUT:
				genac = new AboutAction();
				break;
			case ACTION_EXIT:
				genac = new ExitAction();
				break;
			case ACTION_EXPLAIN:
				genac = new ExplainAction();
				break;
			case ACTION_PREFERENCES:
				genac = new PreferencesAction();
				break;
			case ACTION_DATABASES:
				genac = new DatabasesAction();
				break;
			case ACTION_PLANNEROPTIONS:
				genac = new PlannerOptionsAction();
				break;
			case ACTION_ZOOMIN:
				genac = new ZoomInAction();
				break;
			case ACTION_ZOOMOUT:
				genac = new ZoomOutAction();
				break;
			case ACTION_DELETE:
				genac = new DeleteAction();
				break;
			case ACTION_EDIT:
				genac = new EditAction();
				break;
			case ACTION_OK:
				genac = new OkAction();
				break;
			case ACTION_CANCEL:
				genac = new CancelAction();
				break;
			case ACTION_MANUAL:
				genac = new ManualAction();
				break;
			case ACTION_TEST:
				genac = new TestAction();
				break;
			case ACTION_REFRESH:
				genac = new RefreshAction();
				break;
			case ACTION_CONTINUE:
				genac = new ContinueAction();
				break;
			case ACTION_BACK:
				genac = new BackAction();
				break;
			case ACTION_FORWARD:
				genac = new ForwardAction();
				break;
			case ACTION_HOME:
				genac = new HomeAction();
				break;
			case ACTION_ANALYZE:
				genac = new AnalyzeAction();
				break;
			case ACTION_STOP:
				genac = new StopAction();
				break;
			case ACTION_DEFAULT:
			default:
				genac = new DefaultAction();
				break;
		}

		return genac;
	}

	/**
	 * Get the current image directory.
	 *
	 * @return a <code>String</code> value
	 */
	public String getImageDirectory()
	{
		return imagedir;
	}

	/**
	 * Set the image directory. Doesn't affect previously generated
	 * actions.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setImageDirectory(String s)
	{
		imagedir = s;
		if (imagedir != null &&
			imagedir.lastIndexOf('/') == imagedir.length() - 1)
			imagedir = imagedir.substring(0, imagedir.length() - 1);
	}
	
	//
	// private methods
	//

	private Icon getIcon(String file)
	{
		if (imagedir == null)
			return null;

		String path = imagedir + "/" + file;

		ImageIcon i = new ImageIcon(getClass().getResource(path));

		return i;
	}

	//
	// private classes
	//

	private class NewAction extends GenericAction {
		public NewAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.NEW_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.NEW_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.NEW_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.NEW_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.NEW_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.NEW_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke)res.getObject(ActionResources.NEW_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.NEW_MNEMONIC));
		}
	}

	private class SaveAction extends GenericAction {
		public SaveAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.SAVE_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.SAVE_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.SAVE_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.SAVE_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.SAVE_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.SAVE_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.SAVE_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.SAVE_MNEMONIC));
		}
	}

	private class SaveAsAction extends GenericAction {
		public SaveAsAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.SAVEAS_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.SAVEAS_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.SAVEAS_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.SAVEAS_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.SAVEAS_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.SAVEAS_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.SAVEAS_MNEMONIC));
			putValue(MNEMONIC_KEY_INDEX,
					 (Integer)res.getObject(ActionResources.SAVEAS_MNEMONIC_INDEX));
		}
	}

	private class OpenAction extends GenericAction {
		public OpenAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.OPEN_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.OPEN_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.OPEN_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.OPEN_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.OPEN_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.OPEN_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.OPEN_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.OPEN_MNEMONIC));
		}
	}

	private class CutAction extends GenericAction {
		private Action action;

		public CutAction()
		{
			action = new javax.swing.text.DefaultEditorKit.CutAction();

			putValue(NAME,
					 res.getString(ActionResources.CUT_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.CUT_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.CUT_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.CUT_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.CUT_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.CUT_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.CUT_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.CUT_MNEMONIC));
		}

		public void actionPerformed(ActionEvent e)
		{
			super.actionPerformed(e);

			action.actionPerformed(e);
		}
	}
			
	private class CopyAction extends GenericAction {
		private Action action;

		public CopyAction()
		{
			action = new javax.swing.text.DefaultEditorKit.CopyAction();

			putValue(NAME,
					 res.getString(ActionResources.COPY_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.COPY_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.COPY_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.COPY_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.COPY_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.COPY_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.COPY_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.COPY_MNEMONIC));
		}

		public void actionPerformed(ActionEvent e)
		{
			super.actionPerformed(e);

			action.actionPerformed(e);
		}
	}
			
	private class PasteAction extends GenericAction {
		private Action action;

		public PasteAction()
		{
			action = new javax.swing.text.DefaultEditorKit.PasteAction();

			putValue(NAME,
					 res.getString(ActionResources.PASTE_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.PASTE_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.PASTE_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.PASTE_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.PASTE_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.PASTE_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.PASTE_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.PASTE_MNEMONIC));
		}

		public void actionPerformed(ActionEvent e)
		{
			super.actionPerformed(e);

			action.actionPerformed(e);
		}
	}

	private class AboutAction extends GenericAction {
		public AboutAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.ABOUT_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.ABOUT_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.ABOUT_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.ABOUT_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.ABOUT_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.ABOUT_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.ABOUT_MNEMONIC));
		}
	}

	private class ExitAction extends GenericAction {
		public ExitAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.EXIT_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.EXIT_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.EXIT_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.EXIT_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.EXIT_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.EXIT_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.EXIT_MNEMONIC));
		}
	}

	private class ExplainAction extends GenericAction {
		public ExplainAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.EXPLAIN_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.EXPLAIN_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.EXPLAIN_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.EXPLAIN_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.EXPLAIN_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.EXPLAIN_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.EXPLAIN_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.EXPLAIN_MNEMONIC));
		}
	}

	private class PreferencesAction extends GenericAction {
		public PreferencesAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.PREFERENCES_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.PREFERENCES_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.PREFERENCES_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.PREFERENCES_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.PREFERENCES_MNEMONIC));
		}
	}


	private class DatabasesAction extends GenericAction {
		public DatabasesAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.DATABASES_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.DATABASES_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.DATABASES_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.DATABASES_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.DATABASES_MNEMONIC));
		}
	}


	private class PlannerOptionsAction extends GenericAction {
		public PlannerOptionsAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.PLANNEROPTIONS_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.PLANNEROPTIONS_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.PLANNEROPTIONS_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.PLANNEROPTIONS_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.PREFERENCES_LARGEICON)));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.PLANNEROPTIONS_MNEMONIC));
		}
	}

	private class ZoomInAction extends GenericAction {
		public ZoomInAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.ZOOMIN_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.ZOOMIN_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.ZOOMIN_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.ZOOMIN_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.ZOOMIN_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.ZOOMIN_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.ZOOMIN_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.ZOOMIN_MNEMONIC));
		}
	}

	private class ZoomOutAction extends GenericAction {
		public ZoomOutAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.ZOOMOUT_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.ZOOMOUT_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.ZOOMOUT_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.ZOOMOUT_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.ZOOMOUT_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.ZOOMOUT_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.ZOOMOUT_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.ZOOMOUT_MNEMONIC));
		}
	}
	
	private class DeleteAction extends GenericAction {
		public DeleteAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.DELETE_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.DELETE_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.DELETE_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.DELETE_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.DELETE_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.DELETE_MNEMONIC));
		}
	}

	private class EditAction extends GenericAction {
		public EditAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.EDIT_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.EDIT_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.EDIT_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.EDIT_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.EDIT_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.EDIT_MNEMONIC));
		}
	}

	private class OkAction extends GenericAction {
		public OkAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.OK_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.OK_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.OK_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.OK_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.OK_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.OK_MNEMONIC));
		}
	}

	private class CancelAction extends GenericAction {
		public CancelAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.CANCEL_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.CANCEL_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.CANCEL_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.CANCEL_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.CANCEL_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.CANCEL_MNEMONIC));
		}
	}

	private class ManualAction extends GenericAction {
		public ManualAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.MANUAL_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.MANUAL_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.MANUAL_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.MANUAL_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.MANUAL_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.MANUAL_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.MANUAL_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.MANUAL_MNEMONIC));
		}
	}

	private class TestAction extends GenericAction {
		public TestAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.TEST_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.TEST_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.TEST_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.TEST_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.TEST_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.TEST_MNEMONIC));
		}
	}

	private class RefreshAction extends GenericAction {
		public RefreshAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.REFRESH_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.REFRESH_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.REFRESH_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.REFRESH_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.REFRESH_SMALLICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.REFRESH_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.REFRESH_MNEMONIC));
		}
	}

	private class ContinueAction extends GenericAction {
		public ContinueAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.CONTINUE_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.CONTINUE_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.CONTINUE_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.CONTINUE_LONGDESC));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.CONTINUE_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.CONTINUE_MNEMONIC));
		}
	}

	private class BackAction extends GenericAction {
		public BackAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.BACK_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.BACK_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.BACK_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.BACK_LONGDESC));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.BACK_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.BACK_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.BACK_MNEMONIC));
		}
	}

	private class ForwardAction extends GenericAction {
		public ForwardAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.FORWARD_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.FORWARD_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.FORWARD_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.FORWARD_LONGDESC));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.FORWARD_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.FORWARD_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.FORWARD_MNEMONIC));
		}
	}

	private class HomeAction extends GenericAction {
		public HomeAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.HOME_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.HOME_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.HOME_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.HOME_LONGDESC));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.HOME_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.HOME_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.HOME_MNEMONIC));
		}
	}

	private class AnalyzeAction extends GenericAction {
		public AnalyzeAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.ANALYZE_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.ANALYZE_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.ANALYZE_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.ANALYZE_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.ANALYZE_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.ANALYZE_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.ANALYZE_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.ANALYZE_MNEMONIC));
			putValue(MNEMONIC_KEY_INDEX,
					 (Integer)res.getObject(ActionResources.ANALYZE_MNEMONIC_INDEX));
		}
	}

	private class StopAction extends GenericAction {
		public StopAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.STOP_NAME));
			putValue(ACTION_COMMAND_KEY,
					 res.getString(ActionResources.STOP_COMMAND));
			putValue(SHORT_DESCRIPTION,
					 res.getString(ActionResources.STOP_SHORTDESC));
			putValue(LONG_DESCRIPTION,
					 res.getString(ActionResources.STOP_LONGDESC));
			putValue(SMALL_ICON,
					 getIcon(res.getString(ActionResources.STOP_SMALLICON)));
			putValue(LARGE_ICON,
					 getIcon(res.getString(ActionResources.STOP_LARGEICON)));
			putValue(ACCELERATOR_KEY,
					 (KeyStroke) res.getObject(ActionResources.STOP_ACCELERATOR));
			putValue(MNEMONIC_KEY,
					 (Integer)res.getObject(ActionResources.STOP_MNEMONIC));
		}
	}

	private class DefaultAction extends GenericAction {
		public DefaultAction()
		{
			putValue(NAME,
					 res.getString(ActionResources.DEFAULT_NAME));
		}
	}
}// ActionFactory
