/*
**  Penguin 0.10
**  by Rob Pelkey, 5 April 1997
**
**	Aug 14, 1999 - Tony Mantler <tonym@mac.linux-m68k.org>
**		- Removed bitdepth warning, as new kernels support all bitdepths
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <Aliases.h>
#include <Palettes.h>
#include <Gestalt.h> // JDJ
#include <Files.h>
#include <Fonts.h>
//#include <A4Stuff.h>
#include <Devices.h>
#include <Script.h>
#include <TextUtils.h>
#include <MacTypes.h>
#include <Traps.h>
#include <ToolUtils.h>
#include <Resources.h>
#include <Scrap.h>
#include "penguin_prototypes.h"
#include "penguin.h"
#include "prefs.h"
#include "prefs_prototypes.h"
#include "penguin_utils.h"
#include "bootstrap_prototypes.h"
#include "scsi_info.h"
#include "MMU_V2.h"
#include "machine.h"

QDGlobals qd;

/*
** Extra for Universal interfaces <= 2.1.4
*/
#ifdef UNIVERSAL_INTERFACES_VERSION
  #if UNIVERSAL_INTERFACES_VERSION <= 0x0214
    #define kFontIDMonaco monaco
  #endif
#endif /* UNIVERSAL_INTERFACES_VERSION */

#ifdef	AS_INIT
#include "ShowInitIcon.h"

#define	kDeactivatedIconID		131
#define	kErrorIconID			132
#endif

#define RESID_MBAR				128
#define RESID_ABOUT				131
#define RESID_ALERT_OK_CANCEL	128
#define RESID_ALERT				129
#define RESID_CONSOLE			128

#define MENUID_APPLE			128
#define ITEMID_ABOUT			1

#define MENUID_FILE				129
#define ITEMID_SETTINGS			1
#define ITEMID_OPEN_SETTINGS	3
#define	ITEMID_SAVE_SETTINGS	4
#define	ITEMID_DEFAULT_SETTINGS	5
#define ITEMID_BOOTNOW			7
#define ITEMID_QUIT				9

#define MENUID_EDIT				130
#define ITEMID_COPY				4
#define ITEMID_SELALL			8

#define MENUID_HARDWARE			131
#define ITEMID_PROFILE			1
#define ITEMID_SCSI				2
#define ITEMID_MMUDEV			3

#define CONSOLE_TEXT_LEFT		4
#define CONSOLE_TEXT_RIGHT		476
#define CONSOLE_TEXT_TOP		4
#define CONSOLE_TEXT_BOTTOM		284

#define AUTOBOOT_EVT_COUNT		5


/* For console window */
static TEHandle		gTErecord;
static Rect			gFrameRect;
static WindowPtr	gTheWindow;
static char			*textbuf;
static int			textbuf_len;

static Boolean		gReadyToQuit;
static short		lfd = -1;
static int			logging = 0;
static short		gRestoreDepth = 0;
static int			autoboot_evt_cnt;	/* 5 null events before checking autoboot on launch */

/* App globals */
penguin_config		config;
char				*gzip_rdbuf;

long				gCurrentMacID;

/* Prototypes */
static void			DoAbout (void);
static void			DoBootNow (void);
static void			CheckAutoBoot (void);
static void			HandleMenus ( long theChoice );
static void			HandleMouse ( const EventRecord *evPtr );
static void			HandleKeys ( const EventRecord *evPtr );
static void			HandleUpdate ( const EventRecord *evPtr );
static void			HandleOS ( const EventRecord *evPtr );
static pascal OSErr	HandleAEOpenApp(const AppleEvent *theAppleEvent, AppleEvent *reply, unsigned long handlerRefcon);
static pascal OSErr	HandleAEOpenDoc(const AppleEvent *theAppleEvent, AppleEvent *reply, unsigned long handlerRefcon);
static OSErr		CheckRequiredParams(const AppleEvent *theAppleEvent);
static void			DoEvents (void);
static void			InitToolbox(void);
static void			InitConsole(void);
static void			PrintLegalNotice(void);
static void			SetupMenus(void);
static short		EnvironOK(void);
static short		MainDeviceDepthOK(void);
static void			RestoreMainDevice(void);

/*
 *	cprintf
 *
 *	Print to the console. Save in result file
 *	if logging requested
 */
void cprintf (char *format, ...) {
	va_list		ap;
	long i;
	char *string;
	char buf[1024]; // JDJ
	
	//string = (char *) NewPtr(1024);
	string = buf; // JDJ
	va_start(ap, format);
	vsprintf(string, format, ap);
	va_end(ap);
	
#ifndef	AS_INIT
	strncat(textbuf, string, 1024);
	textbuf_len = strlen(textbuf);
	if (textbuf_len >= 32767) {
		strncpy(textbuf, (textbuf + textbuf_len - 24576), textbuf_len);
		textbuf_len = 24576;
	};
	
	TESetText(textbuf, textbuf_len, gTErecord);
	TESetSelect(textbuf_len, textbuf_len, gTErecord);
	SetPort( gTheWindow );      
//	EraseRect( &gTheWindow->portRect );
//	FrameRect( &gFrameRect );
	TEUpdate( &gTheWindow->portRect, gTErecord );
#endif

	i = strlen(string);
	if (logging)
		FSWrite(lfd, &i, string);
	//DisposePtr((Ptr) string);
}

/*
 *	start_logging
 *
 *	Create/append file and save logging results
 */
OSErr start_logging (void) {
	OSErr err;
	FSSpec log_spec;
	Str255 date, time;
	unsigned long now;
	Boolean junk;	

	memset(&log_spec, 0, sizeof(FSSpec));
	if ((err = ResolveAlias(&app_spec, config.logfile_alias, &log_spec, &junk)) != noErr)
		if (err == fnfErr) {
			if ((err = FSpCreate(&log_spec, 'R*ch', 'TEXT', smSystemScript)) != noErr)
				return err;
			else
				cprintf("\nCreated log file");
		} else
			return err;
	if ((err = FSpOpenDF(&log_spec, fsWrPerm, &lfd)) != noErr)
		return err;
	if ((err = SetFPos(lfd, (config.logfile_append) ? fsFromLEOF : fsFromStart, 0)) != noErr) 
		return err; 
	if (!config.logfile_append) {
		if ((err = SetEOF(lfd, 0)) != noErr) {
			return err;
		}
	}
	logging = 1;
	GetDateTime(&now);
	IUDateString(now, longDate, date);
	IUTimeString(now, true, time);
	cprintf ("\nLogging started %s %s\n", p2cstr(date), p2cstr(time));
#ifdef AS_INIT
	cprintf ("Penguin INIT version " kVersShortString "\n\n");
#else
	cprintf ("Penguin App version " kVersShortString "\n\n");
#endif

	return noErr;
}

/*
 *	stop_logging
 *
 *	Stop the logging to file
 */
void stop_logging (void) {
//	OSErr err;
	Str255 date, time;
	unsigned long now;

	GetDateTime(&now);
	IUDateString(now, longDate, date);
	IUTimeString(now, true, time);
	cprintf ("\nLogging ended %s %s\n", p2cstr(date), p2cstr(time));
	logging = 0;
	FSClose(lfd);
}

/*
 *	DoAbout
 *
 *	Show the somewhat familiar logo...
 */
static void DoAbout (void) {
	int i, depth;
	GDHandle theScreen;
	PixMapHandle theScreenMap;

	theScreen = GetMainDevice();
	theScreenMap = (*theScreen)->gdPMap;
	depth = (*theScreenMap)->pixelSize;
	if (depth >= 8)
		depth = 2;
	else if (depth >= 4)
		depth = 1;
	else
		depth = 0;
	i = Alert (RESID_ABOUT+depth, NULL);
}

/*
 *	DoBootNow
 *
 *	Take-off
 */
static void DoBootNow (void)
{
	autoboot_evt_cnt = 0;

	boot();
}

/*
 *	CheckAutoBoot
 *
 *	Called after a prefs file has been opened, will auto boot
 *	if that flag is set in the prefs and if NOT the <cmd> key
 *	is held down
 */
static void
CheckAutoBoot (void)
{
	KeyMap		kmap;

	autoboot_evt_cnt = 0;

	if (config.auto_boot) {
		/* Don't boot if <cmd> key down */
		GetKeys(kmap);
		if (!(kmap[1] & 0x00008000)) {
			DoBootNow();
		}
	}
}

/*
 *	HandleMenus
 *
 *	Handle application menus
 */
static void HandleMenus ( long theChoice ) {
	short theMenu, theItem; 
	Str255      theItemName;     
	short       theItemNumber;
	MenuHandle  theAppleMenu;

	theMenu = HiWord( theChoice ); 
	theItem = LoWord( theChoice ); 

	switch ( theMenu ) {
	case MENUID_APPLE:
		switch ( theItem ) {
		case ITEMID_ABOUT:
			DoAbout();
			break;
		default:
			theAppleMenu = GetMenuHandle( MENUID_APPLE );   
			GetMenuItemText( theAppleMenu, theItem, theItemName );   
			theItemNumber = OpenDeskAcc( theItemName );
			break;
		}
		break;
	case MENUID_FILE:
		switch ( theItem ) {
		case ITEMID_SETTINGS:
			do_prefs_dialog(&config);
			break;
		case ITEMID_OPEN_SETTINGS:
			if (open_prefs(&config) == noErr)
				CheckAutoBoot();
			break;
		case ITEMID_SAVE_SETTINGS:
			save_prefs(&config);
			break;		
		case ITEMID_DEFAULT_SETTINGS:
			save_prefs_as_default(&config);
			break;
		case ITEMID_BOOTNOW:
			DoBootNow();
			break;
		case ITEMID_QUIT:
			gReadyToQuit = true;
			break;
		}
		break;
	case MENUID_EDIT:
		switch ( theItem ) {
		case ITEMID_COPY:
			TECopy(gTErecord);
			ZeroScrap();
			TEToScrap();
			break;
		case ITEMID_SELALL:
			TESetSelect(0, textbuf_len-1, gTErecord);
			break;
		}
		break;
	case MENUID_HARDWARE:
		switch ( theItem ) {
		case ITEMID_PROFILE:
			ShowMachProfile();
			break;
		case ITEMID_SCSI:
			ShowSCSIInfo();
			break;
		case ITEMID_MMUDEV:
			ShowMMUDeviceMappings();
			break;
		}
		break;
	}
	HiliteMenu(0);  
}

/*
 *	HandleMouse
 *
 *	handle mouse events
 */
static void
HandleMouse ( const EventRecord *evPtr )
{
    WindowPtr   theWindow;
    short       thePart;
    long        theChoice;
	int         shiftKeyDown;
	Point		mLoc;

	thePart = FindWindow( evPtr->where, &theWindow );
	switch ( thePart ) {
	case inMenuBar:
		theChoice = MenuSelect( evPtr->where );
		if ( theChoice != 0 ) 
			HandleMenus ( theChoice );
		break;
	case inDrag:
		DragWindow( theWindow, evPtr->where, &qd.screenBits.bounds );
		break;         
	default:
		mLoc = evPtr->where;
		GlobalToLocal( &mLoc );
		if ( PtInRect( mLoc, &gFrameRect ) ) {
			shiftKeyDown = ( ( evPtr->modifiers & shiftKey ) != 0 );
			TEClick( mLoc, shiftKeyDown, gTErecord );
		};
		break;
	};
}

/*
 *	HandleKeys
 *
 *	handle keyboard events
 */
static void
HandleKeys ( const EventRecord *evPtr )
{
    short       theChar;
    long        theChoice;

	theChar = evPtr->message & charCodeMask;
	if ( ( evPtr->modifiers & cmdKey ) != 0 ) {
		if ( evPtr->what != autoKey ) {
			theChoice = MenuKey( theChar );
			HandleMenus ( theChoice );
		}
	} else {
		switch(theChar) {
			case 0x01:		/* Home */
				TEPinScroll(0, 32767, gTErecord);
				break;
			case 0x04:		/* End */
				TEPinScroll(0, -32767, gTErecord);
				break;
			case 0x0B:		/* PageUp */
				TEPinScroll(
					0,
					((*gTErecord)->viewRect.bottom - (*gTErecord)->viewRect.top - (*gTErecord)->lineHeight),
					gTErecord);
				break;
			case 0x0C:		/* PageDown */
				TEPinScroll(
					0,
					-((*gTErecord)->viewRect.bottom - (*gTErecord)->viewRect.top - (*gTErecord)->lineHeight),
					gTErecord);
				break;
			case 0x1E:		/* ArrowUp */
				TEPinScroll(0, (*gTErecord)->lineHeight, gTErecord);
				break;
			case 0x1F:		/* ArrowDown */
				TEPinScroll(0, -(*gTErecord)->lineHeight, gTErecord);
				break;
		}
	}
}

/*
 *	HandleUpdate
 *
 *	handle update events
 */
static void
HandleUpdate ( const EventRecord *evPtr )
{
    WindowPtr   theWindow;

	theWindow = (WindowPtr)evPtr->message;
	BeginUpdate( theWindow );
		SetPort( theWindow );      
		EraseRect( &theWindow->portRect );
		FrameRect( &gFrameRect );
		TEUpdate( &theWindow->portRect, gTErecord );
	EndUpdate( theWindow );
}

/*
 *	HandleOS
 *
 *	handle OS events
 */
static void
HandleOS ( const EventRecord *evPtr )
{
	switch ((evPtr->message >> 24) & 0xff) {
		case resumeFlag:
			if (evPtr->message & suspendResumeMessage) {
				TEActivate(gTErecord);
			} else {
				TEDeactivate(gTErecord);
			}
			break;

		default:
			break;
	}
}

/*
 *	HandleAEOpenApp
 *
 *	Handle open application event
 */
static pascal OSErr
HandleAEOpenApp(const AppleEvent *theAppleEvent, AppleEvent *reply, unsigned long handlerRefcon)
{
//#pragma unused(reply, handlerRefcon)

	OSErr		theErr;

	/* Do nothing */
	theErr = CheckRequiredParams(theAppleEvent);
	return theErr;
}

/*
 *	HandleAEOpenDoc
 *
 *	Handle open document event
 */
static pascal OSErr
HandleAEOpenDoc(const AppleEvent *theAppleEvent, AppleEvent *reply, unsigned long handlerRefcon)
{
//#pragma unused(reply, handlerRefcon)

	AEDescList		docList;
	AEKeyword		theKey;
	DescType		theType;
	Size			theSize;
	long			numDocs;
	OSErr			theErr;
	FSSpec			theFSS;

	/* Get data from apple event */
	theErr = AEGetParamDesc(theAppleEvent, keyDirectObject, typeAEList, &docList);
	if (theErr == noErr) {
		theErr = CheckRequiredParams(theAppleEvent);
		if (theErr == noErr) {
			theErr = AECountItems(&docList, &numDocs);
			if (theErr == noErr) {
				/* If there are more than one document, open the first one
				 * and silently ignore the rest
				 */
				if (numDocs >= 1) {
					theErr = AEGetNthPtr(&docList, 1, typeFSS, &theKey, &theType,
									(Ptr)&theFSS, sizeof(theFSS), &theSize);
					if (theErr == noErr) {
						theErr = read_prefs(&theFSS, &config);
						if (theErr == noErr) {
							CheckAutoBoot();
						} else
							cprintf("Error loading prefs (read_prefs() : %hd\n", theErr);
					} else
						cprintf("Error loading prefs (AEGetNthPtr() : %hd\n", theErr);
				}
			} else
				cprintf("Error loading prefs (AECountItems() : %hd\n", theErr);
		} else
			cprintf("Error loading prefs (CheckRequiredParams() : %hd\n", theErr);
		
		AEDisposeDesc(&docList);

	} else
		cprintf("Error loading prefs (AEGetParamDesc() : %hd\n", theErr);

	/* Do nothing */
	return theErr;
}

/*
 *	CheckRequiredParams
 *
 *	Check for missing apple event parameter(s)
 */
static OSErr
CheckRequiredParams(const AppleEvent *theAppleEvent)
{
	DescType		theType;
	Size			theSize;
	OSErr			theErr;

	theSize = 0;
	theErr = AEGetAttributePtr(theAppleEvent, keyMissedKeywordAttr,
				typeWildCard, &theType, nil, 0, &theSize);
	
	if (theErr == errAEDescNotFound)
		theErr = noErr;
	else if (theErr == noErr)
		theErr = errAEParamMissed;

	return theErr;
}

/*
 *	DoEvents
 *
 *	the main event (loop)
 */
static void DoEvents (void)
{
	EventRecord theEvent;

	WaitNextEvent( everyEvent, &theEvent, 15L, nil );
	switch ( theEvent.what ) {

		case nullEvent:
			TEIdle( gTErecord );
			break;

		case activateEvt:
			if ( ( theEvent.modifiers & activeFlag ) != 0 )
				TEActivate( gTErecord );
			else
				TEDeactivate( gTErecord );
			break;

		case updateEvt:
			HandleUpdate(&theEvent);
			break;

		case mouseDown:
			HandleMouse(&theEvent);
			break;

		case keyDown:
		case autoKey:
			HandleKeys(&theEvent);
			break;

		case osEvt:
			HandleOS(&theEvent);
			break;

		case kHighLevelEvent:
			AEProcessAppleEvent(&theEvent);
			break;

		default:
			break;
	}
}

/*
 *	InitToolbox
 *
 *	Usual Mac setup stuff
 */
static void InitToolbox(void)
{
	InitGraf( &qd.thePort );
	InitFonts();
	FlushEvents(everyEvent, 0L);
	InitWindows();
	InitMenus();
	TEInit();
	InitDialogs( 0L );
	InitCursor();

	MaxApplZone();
	MoreMasters();
	MoreMasters();
	MoreMasters();

	/* Install out event handlers, ignore errors */
	AEInstallEventHandler(kCoreEventClass, kAEOpenApplication, (AEEventHandlerUPP)HandleAEOpenApp, 0, false);
	AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, (AEEventHandlerUPP)HandleAEOpenDoc, 0, false);
}

/*
 *	InitConsole
 *
 *	Create TextEdit based console window
 */
static void InitConsole(void)
{
	Rect		theDestRect;
	Rect		theViewRect;
	
	/* set up TextEdit-based console window */
	textbuf = (char *) NewPtrClear(40960);
	if (textbuf == NULL) return; // JDJ
	gTheWindow = GetNewWindow( RESID_CONSOLE, nil, (WindowPtr)-1L );
	if (gTheWindow == NULL) return; // JDJ
	SetWTitle(gTheWindow, "\pPenguin Console");
	SetPort( gTheWindow );
	ShowWindow( gTheWindow );
	SetRect( &theDestRect, CONSOLE_TEXT_LEFT, CONSOLE_TEXT_TOP,
		CONSOLE_TEXT_RIGHT, CONSOLE_TEXT_BOTTOM );
	theViewRect = theDestRect;
	TextFont (kFontIDMonaco);
	TextSize (9);
	gTErecord = TENew( &theDestRect, &theViewRect );
	if (gTErecord == NULL) return; // JDJ
	TEAutoView( true, gTErecord );
	gFrameRect = theDestRect;
	InsetRect( &gFrameRect, -5, -5 );
}

/*
 *	PrintLegalNotice
 *
 *	Tell 'em who we are and what we've done!
 */
static void PrintLegalNotice(void)
{
	/* say hi */
	cprintf ("Penguin version " kVersShortString " - " kBuildDateString);
	cprintf ("\nby Tony Mantler\n");
	cprintf ("with Carsten Neumann, Andrew McPherson,\n");
	cprintf ("   Rob Pelkey, Joshua Juran, Randy Thelen and Mikael Forselius.\n");
	cprintf ("   - Penguin INIT by Jrgen Mellinger.\n");
	cprintf ("   - ShowInitIcon by Peter N Lewis, Jim Walker and Francois Pottier\n");
	cprintf ("   - Additions & corrections by Bryan Baker, Marc LaViolette.\n\n");

	cprintf ("Based on Linux/m68k Atari Bootstrap version 1.8\n");
	cprintf ("Copyright 1993, 1994, 1995 by Arjan Knor, Robert de Vries,\n");
	cprintf ("    Roman Hodek, and Andreas Schwab\n");

	cprintf ("Code from MacBSD Booter copyright 1995, 1996 by Brian R. Gacke\n");
	cprintf ("    Redistributed under GNU GPL with author's permission.\n");

	cprintf ("Code from OpenBSD effort copyright 1993 by\n");
	cprintf ("    Allen K. Briggs, Chris P. Caputo, Michael L. Finch,\n");
	cprintf ("    Bradley A. Grantham and Lawrence A. Kesteloot.\n\n");

	cprintf ("Penguin is distributed under the GNU General Public License, and comes with\n");
	cprintf ("ABSOLUTELY NO WARRANTY; see the accompanying file COPYING for details.\n\n");
}

/*
 *	SetupMenus
 *
 *	Create our menus
 */
static void SetupMenus(void)
{
	Handle      theMenuBar;  
	MenuHandle  theAppleMenu;
	
	/* install our menus */
	ClearMenuBar();
	theMenuBar = GetNewMBar( 128 );
	SetMenuBar( theMenuBar );

	//DisposeHandle( theMenuBar );
	ReleaseResource(theMenuBar); // JDJ
	theAppleMenu = GetMenuHandle( 128 );
	AppendResMenu( theAppleMenu, 'DRVR' );
	DrawMenuBar();
}

/*
 *	EnvironOK
 *
 *	Validate this Mac, abort if not Linux capable :(
 */
static short EnvironOK(void)
{
	short	envFail;
	OSErr	err;
	long	response;

	envFail = 0;

	if (!have_trap(_Gestalt)) {

		/* Fail if no _Gestalt */
#ifndef	AS_INIT
		ParamText ("\pError: Gestalt() trap missing.\n",
			"\pSystem too old, unable to continue...\n",
			"\pUpdate your System and try again!\n", "\p");
		StopAlert (RESID_ALERT, NULL);
#endif
		++envFail;

	} else {

		/* Fail if System version < 7.0 */
		err = Gestalt(gestaltSystemVersion, &response);
		if (err == noErr) {
			if (response < 0x0700) {
#ifndef	AS_INIT
				ParamText ("\pFailure: Needs System 7.0 or later.\n", "\p", "\p", "\p");
				StopAlert (RESID_ALERT, NULL);
#endif
				++envFail;
			}
		} else {
#ifndef	AS_INIT
			ParamText ("\pFailure: Needs System 7.0 or later.\n", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
#endif
			++envFail;
		}

		/* Warn if 24-bit mode */
		err = Gestalt(gestaltAddressingModeAttr, &response);
		if (err == noErr) {
			if (!GestaltBitSet(response, gestalt32BitAddressing)) {
#ifndef	AS_INIT
				ParamText ("\pWarning: System in 24-bit addressing mode.\n",
					"\pFor best results, turn 32-bit addressing on and reboot.\n",
					"\p", "\p");
				NoteAlert (RESID_ALERT, NULL);
#endif
			}
		} else {
#ifndef	AS_INIT
			ParamText ("\pFailure: Can't determine addressing mode (24/32-bit).\n", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
#endif
			++envFail;
		}

		/* Fail if VM on */
		err = Gestalt(gestaltVMAttr, &response);
		if (err == noErr) {
			if (GestaltBitSet(response, gestaltVMPresent)) {
#ifndef	AS_INIT
				ParamText ("\pFailure: Virtual memory activated.\n",
					"\pTurn virtual memory off, reboot and try again.\n",
					"\p", "\p");
				StopAlert (RESID_ALERT, NULL);
#endif
				++envFail;
			}
		} else {
#ifndef	AS_INIT
			ParamText ("\pFailure: Can't determine state of virtual memory.\n", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
#endif
			++envFail;
		}

		/* Fail if not 68K architecture */
		err = Gestalt(gestaltSysArchitecture, &response);
		if (err == noErr) {
			if (response != gestalt68k) {
#ifndef	AS_INIT
				ParamText ("\pFailure: Requires MC680x0, not PowerPC.\n", "\p", "\p", "\p");
				StopAlert (RESID_ALERT, NULL);
#endif
				++envFail;
			}
		}

		/* Fail if not 68020, 68030, 68040 */
		err = Gestalt(gestaltProcessorType, &response);
		if (err == noErr) {
			if ( (response < gestalt68020) || (response > gestalt68040) ) {
#ifndef AS_INIT
				ParamText ("\pFailure: Requires 68020, 68030 or 68040 processor.\n",
					"\p", "\p", "\p");
				StopAlert (RESID_ALERT, NULL);
#endif
				++envFail;
			}
		} else {
#ifndef	AS_INIT
			ParamText ("\pFailure: Can't determine processor type.\n", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
#endif
			++envFail;
		}

		/* Fail if no PMMU, 68030 MMU or 68040 MMU */
		err = Gestalt(gestaltMMUType, &response);
		if (err == noErr) {
			if ( (response < gestalt68851) || (response > gestalt68040MMU) ) {
#ifndef	AS_INIT
				ParamText ("\pFailure: Requires memory management unit (MMU).\n",
					"\p", "\p", "\p");
				StopAlert (RESID_ALERT, NULL);
#endif
				++envFail;
			}
		} else {
#ifndef	AS_INIT
			ParamText ("\pFailure: Can't determine MMU type.\n", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
#endif
			++envFail;
		}


	}


	return (envFail == 0);
}

/*
 *	MainDeviceDepthOK
 *
 *	Validate device depth, ask to change depth if > 8 bits
 *	abort if depth change not verified.
 */
short
MainDeviceDepthOK(void)
{
#if 1 /* Kernel supports all bitdepths */
	return 1;
#else
	GDHandle		hDevice;
	PixMapHandle	hPixMap;
	short			depth, result;
#ifndef	AS_INIT
	short			newDepth, i;
#endif
	result = 0;

	hDevice = GetMainDevice();
	hPixMap = (*hDevice)->gdPMap;
	depth = (*hPixMap)->pixelSize;
	if (depth > 8) {
#ifndef	AS_INIT
		for(newDepth = 8; newDepth > 0; newDepth >>= 1) {
			if (HasDepth(hDevice, newDepth, 0, 0))
				break;
		}
		if (newDepth > 0) {
			ParamText ("\pCurrently only supporting 256-colors or less.\nChange to compatible bit-depth?", "\p", "\p", "\p");
			i = NoteAlert (RESID_ALERT_OK_CANCEL, NULL);
			if (i == 1) {
				if (SetDepth(hDevice, newDepth, 0, 0) == noErr) {
					gRestoreDepth = depth;
					result = 1;
				} else {
					ParamText ("\pFailure: Unable to set number of colors.", "\p", "\p", "\p");
					StopAlert (RESID_ALERT, NULL);
				}
			} else if (i == 3)
				result = 1;

		} else {
			ParamText ("\pFailure: No compatible screen depth found. Unable to set number of colors.", "\p", "\p", "\p");
			StopAlert (RESID_ALERT, NULL);
		}
#else
		cprintf( "Failure: Screen Depth must be set to 256 colors or less.\n" );
#endif
	} else
		result = 1;

	return result;
#endif
}

/*
 *	RestoreMainDevice
 *
 *	Return main device to original state (if changed)
 */
static void
RestoreMainDevice(void)
{
	GDHandle		hDevice;

	if (gRestoreDepth > 0) {
		hDevice = GetMainDevice();
		SetDepth(hDevice, gRestoreDepth, 0, 0);
		gRestoreDepth = 0;
	}
}

/*
 *	main
 *
 */
void main (void) {
	OSErr		err; // JDJ
	KeyMap		kmap;
	
#ifndef	AS_INIT
	InitToolbox();
#endif
	
	/* Ignore startup autoboot if <cmd> key down else wait
	 * AUTOBOOT_EVT_COUNT events (to process AppleEvents, 'odoc')
	 * before checking default settings file for autboot flag
	 */
	GetKeys(kmap);
	if (kmap[1] & 0x00008000)
		autoboot_evt_cnt = 0;
	else
		autoboot_evt_cnt = AUTOBOOT_EVT_COUNT;

	/* Allocate buffer for gzip'd images
	 * Might as well do it here and carry the
	 * buffer around while app is executing -
	 * The memory must be available anyway
	 */
	gzip_rdbuf = NewPtr(GZIP_RDBUF_SIZE);
	if (gzip_rdbuf == nil) {
#ifndef	AS_INIT
		ParamText ("\pFailure: Unable to allocate buffers.\n",
			"\pIncrease the memory partition.\n",
			"\p", "\p");
		StopAlert (RESID_ALERT, NULL);
#else
		ShowInitIcon( kErrorIconID, true );
#endif
		
		/* exit application... */
		return;
	}

	if (EnvironOK()) {

		if (MainDeviceDepthOK()) {

			Gestalt(gestaltMachineType, &gCurrentMacID);

			gReadyToQuit = false; // JDJ
		 	
#ifndef	AS_INIT
			InitConsole();
			PrintLegalNotice();
			
			SetupMenus();
#endif

			err = init_app_spec();
			if (err)
				cprintf("init_app_spec() returned error %d in main.\n", err);
			err = init_prefs(&config);
			if (err)
				cprintf("init_prefs(&config) returned error %d in main.\n", err);

#ifndef	AS_INIT
			/* Sit and do stuff */
			while (!gReadyToQuit) {
				if (autoboot_evt_cnt > 0) {
					if (--autoboot_evt_cnt == 0) {
						if (config.auto_boot)
							DoBootNow();
					}
				}
				DoEvents(); // JDJ
			}

			RestoreMainDevice();
		}
#else
			if ( config.boot_into_linux && autoboot_evt_cnt )
			{
				DoBootNow();
				ShowInitIcon( kErrorIconID, true );
			}
			else
				ShowInitIcon( kDeactivatedIconID, true );
		}
#endif


	}
}

