// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/los.cpp,v 1.11 2002/01/11 15:47:38 weldon Exp $
//

 
#include "gc_for_orp.h"
 
#include "los.h"
#include "object_layout.h"
#include "remembered_set.h"
#include "descendents.h"
#include "gc_hooks.h"
#include "generation.h"
#include "gc_consts.h"
 
#include "gc_globals.h"
#include "gc_plan.h"
#include "gc_debug.h"
 
#include "gc_consts.h"
#include "gc_perf.h"
#include "los_container.h"
#include "root_set_enum.h"
#include "orp_threads.h"
#include "orp_synch.h"
#include "block_list.h"

// Forward reference.
block_info *p_get_new_los_single_block(unsigned int object_size, bool extend, bool collect, bool in_a_gc);
// Allocate a single block for each LOS object size.
void init_large_object_stores() {
    for (int i = 0; i < GC_LOS_BUCKETS; i++) {
        // Allocate a single block for each size in the LOS.
        los_buckets[i] = p_get_new_los_single_block((POINTER_SIZE_INT)(64<<i), true, false, true); 
        // Extend instead of collecting.
        los_buckets_free_hint[i] = los_buckets[i]; // When allocating this size start looking here.
        assert ((POINTER_SIZE_INT)(64<<i) < GC_BLOCK_ALLOC_SIZE);
    }
}

Java_java_lang_Object *gc_los_malloc_noclass (unsigned size) 
{
    //
    // The size can't be zero, since there is a requisite VTable pointer
    // word that is the part of every object, and the size supplied
    // to us in this routine should include that VTable pointer word.
    //
    assert(size!=0);
    // Classes loaded before java.lang.Class must use this API, because the
    // vtable for java.lang.Class hasn't been constructed yet.
    // Currently only three classes should be allocated through this call:
    // java.lang.Object, java.io.Serializable and java.lang.Class.
    
    assert ((size % GC_OBJECT_ALIGNMENT) == 0);
    
#ifdef SPACE_DEMOGRAPHICS
    record_for_space_demographics(size);
#endif
    
    int attempt = 0;
    unsigned int real_size_bytes = size;
    Object_Gc_Header *result_start = NULL;
//    p_gc_lock->_lock_enum();                     // vvvvvvvvvvvvv
    int bucket_index = 0;

    unsigned int i;
    for (i=64; i < GC_BLOCK_ALLOC_SIZE; i = i*2) {
        if (size < i) {
            break;
        }
        bucket_index++;
    }

    // 
    // The object goes into the bucket at bucket_index.
    block_info *right_size = los_buckets_free_hint[bucket_index];
    bool if_collect_return_null = true;
    bool try_extending = false;
    while (result_start == NULL) {
        if (right_size->free) {
            // Race condition here result hdr could be zero!!!!!
            result_start = (Object_Gc_Header *)right_size->free;
            if (result_start) {
                los_free_link *next = ((los_free_link *)result_start)->next;
                if ( result_start != InterlockedCompareExchangePointer (&right_size->free, next, result_start)) {
                    result_start = NULL; // Try again it didn't work.
                }
            }
        } else {
            if (right_size->next) {
                right_size = right_size->next;
                los_buckets_free_hint[bucket_index] = right_size; // This could race but it is just a hint.
            } else {
                block_info *temp = p_get_new_los_single_block(i, try_extending, if_collect_return_null, false);
                // Allocate and link a block of the correct size free objects.
                if_collect_return_null = false; // If we get back here we extend force the getting of a block.
                try_extending = true;
                
                if (temp) { // We got a new block instead of collecting.
                    block_info *temp_right_size = right_size;
                    // old code.
                    while (InterlockedCompareExchangePointer((PVOID *)&temp_right_size->next, (PVOID)temp, (PVOID)NULL)) {
                        temp_right_size = temp_right_size->next;
                    }
                    
                    right_size = temp_right_size;
                } else {
                        
                    right_size = los_buckets[bucket_index];;
                    //
                    // We did a collection so we need to just try again. If it fails next time we do an extend.
                    //
                }
            }
        }
    }
    *result_start = NULL; // Clear the next field.
#ifdef _DEBUG
    // Make sure the object is clear.
    unsigned int j;
    for (j = 0; j < right_size->los_object_size; j++) {
        assert (*((char *)result_start) == 0);
    }
#endif 
//    p_gc_lock->_unlock_enum();    // ^^^^^^^

    return P_OBJ_FROM_START(result_start);
}
 
#ifdef SPACE_DEMOGRAPHICS
void record_for_space_demographics(unsigned obj_size)
{
    obj_size += sizeof(Object_Gc_Header);

    if (obj_size < DEMOGRAPHICS_ARRAY_SIZE) {
        space_demographics_size[obj_size] += 1;
    } else {
        cout << "OBJECT OF SIZE " << obj_size << " GREATER THAN SPACE DEMOGRAPHICS LIMIT " << endl;
    }
}
#endif

void forced_reclaim_heap()
{
    p_gc->reclaim_heap(0, true);
}

#ifdef GC_STATS
//
// The LOS is being give a chance to set up the instrumentation
// to do things like space accounting, just before a GC hits.
//
void 
Large_Object_Store::prepare_for_collection() 
{
    _stat_marked_space    = 0;
    _stat_swept_free      = 0;
    _stat_swept_live      = 0;
    _stat_swept_reclaimed = 0;
    _stat_free_space_not_recovered = 0;
}
#else
void 
Large_Object_Store::prepare_for_collection() 
{
}
#endif // GC_STATS



Large_Object_Store::Large_Object_Store(Gc_Interface  *p_gc_interface,
                                       Gc_Fast_Hooks *p_gc_hooks,
                                       Gc_Plan       *p_gc_plan,
									   Generation    *p_container,
		                               Card_Table    *p_card_table,
					                   Block_Store   *p_bs,
                                       LOS_Container *p_los_container)
					   : Gc_Space(0, 
                                  p_gc_hooks,
                                  p_gc_plan,
					              p_container, 
								  NULL, 
					              p_card_table, 
								  p_bs)
{
	_los_block_size_bytes = p_gc_plan->los_block_size_bytes();
    // 
    assert (0); // We should be calling init_large_object_store instead."
}

//
// When the Large Object Store gets an evict call, it marks and scans 
// all the descendents of the object, but does not move that object.
// Currently unreachable objects are not reclaimed from this store.
//
Java_java_lang_Object *
mark_and_scan_object(Java_java_lang_Object *p_obj)
{
	//
	// If the object is marked, this means that it has already 
	// been visited. The forwarded bit and the mark bit are the same.
    //               BUT DO NOT DEPEND ON THIS IN THE CODE.
	//
	if (is_object_marked(p_obj)) {
		return p_obj;
	}
    //
    // A mark bit signals that this object has been scanned.
    // After the stop-the-world collection, the cleanup routine should
    // turn the mark bit off.
    //
	set_object_marked(p_obj);
#ifdef GC_STATS
    //
    // Update the statistics.
    //
    _stat_marked_space += get_real_object_size_bytes(p_obj);
#endif
    // What is the default place to move objects refered to by LOS. It will be the first train. 
    
	scan_object(p_obj);
	return p_obj;
}

//
// Mrl_Gc_Vxx calls us to clean up after a stop-the-world collection.
// This is our chance to clear the mark bits in all the objects.
//
void 
Large_Object_Store::cleanup()
{
    assert(0);
}
   

//
// An object is being allocated from the fixed Large Object Space. Check to
// see if it will fit into this store. If not just return null since it may
// be able to fit into some other Large_Object_Store.
//
// If return_null_on_fail is true we need to return without waiting for locks
// so the JITed code can be moved to a GC safepoint.
//  
Java_java_lang_Object *
Large_Object_Store::gc_pinned_malloc(unsigned size, 
									 VTable *p_vtable,
									 bool return_null_on_fail,
                                     bool double_alignment_required                               
                                     )
{
    assert(0);
    return NULL;
}
 
// 
// Get a single block that will hold pinned object that don't
// get their own block.
// If collect is true we can return NULL to indicate that we did a collection instead of
// getting a new block. Hopefully this freed up some fixed objects.
//
block_info *p_get_new_los_single_block(unsigned int object_size, bool extend, bool collect_returns_null, bool in_a_gc)
{
    block_info *los_block = p_get_new_block (extend, collect_returns_null, in_a_gc);
    if (los_block == NULL) {
        return NULL;
    }
    
    assert (gc_block_status_table[los_block->block_status_table_index] == block_in_free);
    gc_block_status_table[los_block->block_status_table_index] = block_in_los;
    los_block->in_los_p = true;
    los_block->in_free_p = false;
    assert (los_block->car_birthday == 0);
    los_block->c_area_p = true;
    los_block->los_object_size = object_size;
    los_block->free = GC_BLOCK_ALLOC_START(los_block);
    memset (los_block->free, 0, GC_BLOCK_ALLOC_SIZE); // Clear the objects.
    // Create a linked list of objects.
    los_free_link *tmp;
    los_free_link *next_object;
    tmp = (los_free_link *)los_block->free;
    next_object = (los_free_link *)((POINTER_SIZE_INT)tmp + (POINTER_SIZE_INT)object_size);
    while (((POINTER_SIZE_INT)next_object + (POINTER_SIZE_INT)object_size) <= 
           ((POINTER_SIZE_INT)GC_BLOCK_ALLOC_START(los_block) + (POINTER_SIZE_INT)GC_BLOCK_ALLOC_SIZE)) {
        // There is room for the next object.
        tmp->next = (los_free_link *)next_object;
        tmp = tmp->next;
        next_object = (los_free_link *)((POINTER_SIZE_INT)tmp + (POINTER_SIZE_INT)object_size);
    }

    tmp->next = NULL;
    return los_block;
}

//
// This is a special case version of gc_pinned_malloc, which is used
// at bootstrap time to create objects which don't have classes
// at that time. (The classes come later, and the VTable pointers
// are appropriately patched.) Need to build a common routine
// out of this and the previous one.
//
Java_java_lang_Object *
Large_Object_Store::gc_pinned_malloc_noclass(unsigned size)
{
    return gc_los_malloc_noclass (size);
}
//
// We just ran out of LOS. Notify the GC Interface.
//
void
Large_Object_Store::_notify_large_object_store_exhausted(unsigned int size)
{
	//
	// (Finally!) Notify enclosing GC Interface that the LOS is out of space
	//
    assert (0);
// 	_p_gc_interface->notify_out_of_space(999); // NOW CALLED BY CONTAINER LOGIC.
}

//
// No more LOS, even after a collection.
//
void
Large_Object_Store::_notify_large_object_store_exhausted_error(unsigned int size)
{
	cout << "Error: Large Object Space Exhausted when allocating ";
	cout << size << " bytes" << endl;

	orp_exit(1);
}

void *
Large_Object_Store::_skip_free_block(void *p_thing)
{
	unsigned int free_block_size = get_free_block_size(p_thing);

	return (void *)((char *)p_thing + free_block_size);
}

//
// The generation may not be available when the first LOS
// is created. Therefore it may need to be patched later.
// Earlier we created LOSs first because they were large,
// monolithic beasts. Now that we have non-contiguous,
// variable sized loses that grow on demand, we don't need
// to create them first. Need to refine this.
//
void 
Large_Object_Store::set_generation(Generation *p_gen) 
{
	p_container = p_gen;
}

void *
Large_Object_Store::_sweep_thing(void *p_thing)
{
        assert (0);

	return NULL;
}
 
void 
Large_Object_Store::_update_allocation_stats(Partial_Reveal_VTable *p_vtable, 
							                 unsigned int real_size_bytes)
{

	_resident_object_count++;
	_resident_occupied_bytes += real_size_bytes;
 
	if (real_size_bytes < _smallest_object_size)
		_smallest_object_size = real_size_bytes;

	if (real_size_bytes > _largest_object_size)
		_largest_object_size = real_size_bytes;
}

 
bool 
Large_Object_Store::_verify_object_is_in_large_object_store(Java_java_lang_Object *p_obj)
{
    assert(0);
	return false;
}

ORP_thread *Large_Object_Store::get_lock_owner()
{
    return this_los_lock;
}   

bool Large_Object_Store::try_lock ()
{
    if (InterlockedCompareExchangePointer (
        (PVOID *)&this_los_lock,
        (PVOID)p_TLS_orpthread, 
        (PVOID)NULL) == 
        (PVOID)NULL) {
        return true; // success
    } else {
        return false;
    }
}

void Large_Object_Store::release_lock()
{
    assert (p_TLS_orpthread == this_los_lock);
    this_los_lock = NULL;
}

//
// Scan to see if there is room for this block.
//
bool Large_Object_Store::is_block_available(unsigned int size)
{
    return p_global_bs->is_block_available(size);
}
