// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_lang_reflect_Field.cpp,v 1.4 2001/12/10 09:41:25 gwu2 Exp $
//



#include "platform.h"
#include "environment.h"
#include "root_set_enum.h"
#include "native_utils.h"
#include "jni_direct.h"
#include "jni_utils.h"
#include "java_lang_reflect_Field.h"

#include "gnu_classpath_jni_utils.h"
#include "jni_utils.h"

/* According to new Classpath Field native interface, we :
 *   1. Removed Java_java_lang_reflect_Field_getName(), Classpath do this
 *   2. Removed All arguments of "jclass caller_class" in method declarations
 *   3. Remove "jclass type" argument in Java_java_lang_reflect_Field_set()
 * /


/*
 * Class:     java_lang_reflect_Field
 * Method:    getName
 * Signature: ()Ljava/lang/String;
 */
/*
JNIEXPORT jstring JNICALL Java_java_lang_reflect_Field_getName
  (JNIEnv *jenv, jobject jfield)
{
	Field *f = GetField(jenv, jfield);
	assert(f);
	const String *name = f->get_name();
	assert(name);
	jstring ret_string = jenv->NewStringUTF(name->bytes);
	assert(ret_string);
	return ret_string;
} // Java_java_lang_reflect_Field_getName
*/

/*
 * Class:     java_lang_reflect_Field
 * Method:    getType
 * Signature: ()Ljava/lang/Class;
 */

JNIEXPORT jclass JNICALL Java_java_lang_reflect_Field_getType
  (JNIEnv *jenv, jobject jfield)
{
	Field *f = GetField(jenv, jfield);
	assert(f);
	// We must do some mapping from signature to valid class name 
	//  that could be loaded
	//Class *clss = f->get_class();
	//assert(clss);
	//jclass clazz = jenv->FindClass(clss->name->bytes);
	jclass clazz;
	const char *buf = f->get_signature()->descriptor->bytes;
	if(*buf != JAVA_TYPE_CLASS){
		switch(*buf){
		case 'Z': buf = "boolean"; break;
		case 'B': buf = "byte"; break;
		case 'C': buf = "char"; break;
		case 'D': buf = "double"; break;
		case 'F': buf = "float"; break;
		case 'I': buf = "int"; break;
		case 'J': buf = "long"; break;
		case 'S': buf = "short"; break;
		case '[': break;
		}
		clazz = jenv->FindClass(buf);
	}else{
		assert(*buf == 'L');
		int len = strlen(buf);
		char *name = new char[len];
		strncpy(name, buf+1, len-2);
		name[len-2] = '\0';
		//clazz = jenv->FindClass(name);
		Class *caller = f->get_class();
		clazz = FindClassWithClassLoader(name, caller->class_loader);		
		delete name;
	}
	assert(clazz);
	return clazz;
} // Java_java_lang_reflect_Field_getType


/*
 * Class:     java_lang_reflect_Field
 * Method:    getModifiers
 * Signature: ()I
 */

JNIEXPORT jint JNICALL Java_java_lang_reflect_Field_getModifiers
  (JNIEnv *jenv, jobject jfield)
{
	Field* f = GetField(jenv, jfield);
	assert(f);
	jint modifiers = f->get_access_flags();
	return modifiers;
} // java_lang_reflect_Field_getModifiers



/*
 * Class:     java_lang_reflect_Field
 * Method:    getBoolean
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)Z
 */


JNIEXPORT jboolean JNICALL Java_java_lang_reflect_Field_getBoolean
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jboolean value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a boolean type:
	if (!PrimitiveIsAssignableFrom('Z', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jboolean)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticBooleanField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetBooleanField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getBoolean



/*
 * Class:     java_lang_reflect_Field
 * Method:    getChar
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)C
 */


JNIEXPORT jchar JNICALL Java_java_lang_reflect_Field_getChar
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jchar value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a char type:
	if (!PrimitiveIsAssignableFrom('C', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jbyte)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticCharField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetCharField(obj, field_id);
	}
	return (jchar) value;
} // Java_java_lang_reflect_Field_getChar



/*
 * Class:     java_lang_reflect_Field
 * Method:    getByte
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)B
 */

JNIEXPORT jbyte JNICALL Java_java_lang_reflect_Field_getByte
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jbyte value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('B', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jbyte)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticByteField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetByteField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getByte


/*
 * Class:     java_lang_reflect_Field
 * Method:    getShort
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)S
 */

JNIEXPORT jshort JNICALL Java_java_lang_reflect_Field_getShort
  (JNIEnv *jenv, jobject jfield, jobject obj)
{	
	jshort value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('S', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jshort)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticShortField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetShortField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getShort



/*
 * Class:     java_lang_reflect_Field
 * Method:    getInt
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)I
 */

JNIEXPORT jint JNICALL Java_java_lang_reflect_Field_getInt
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jint value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('I', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jint)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticIntField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetIntField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getInt



/*
 * Class:     java_lang_reflect_Field
 * Method:    getLong
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)J
 */


JNIEXPORT jlong JNICALL Java_java_lang_reflect_Field_getLong
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jlong value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('J', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jlong)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticLongField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetLongField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getLong


/*
 * Class:     java_lang_reflect_Field
 * Method:    getFloat
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)F
 */


JNIEXPORT jfloat JNICALL Java_java_lang_reflect_Field_getFloat
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jfloat value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('F', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jfloat)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticFloatField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetFloatField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getFloat


/*
 * Class:     java_lang_reflect_Field
 * Method:    getDouble
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)D
 */


JNIEXPORT jdouble JNICALL Java_java_lang_reflect_Field_getDouble
  (JNIEnv *jenv, jobject jfield, jobject obj)
{
	jdouble value = 0;
    jfieldID field_id;

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	char type = f->get_signature()->descriptor->bytes[0];

	// If field cannot be converted to a byte type:
	if (!PrimitiveIsAssignableFrom('D', type)) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return (jdouble)0;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);

		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		value = jenv->GetStaticDoubleField(dclazz, field_id);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		value = jenv->GetDoubleField(obj, field_id);
	}
	return value;
} // Java_java_lang_reflect_Field_getDouble



/*
 * Class:     java_lang_reflect_Field
 * Method:    get
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;)Ljava/lang/Object;
 */


JNIEXPORT jobject JNICALL Java_java_lang_reflect_Field_get
  (JNIEnv *jenv, jobject field, jobject obj)
{
	jobject retobj = 0;
	jfieldID  field_id;
	jclass dclazz = 0;
	jvalue args[1];

	Field* f = GetAndVerifyField(jenv, field, obj); 
	assert(f);

	char type = f->get_signature()->descriptor->bytes[0];
	
	if (f->is_static()) {
		// Access handle of class declaring this field
	    dclazz = GetFieldClass(jenv, field);
		assert(dclazz);
		field_id = jenv->GetStaticFieldID(dclazz,
									 f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
	} else {
		// Access handle of parameter object's class:
	    jclass oclazz = GetObjectClass(jenv, obj);
		assert(oclazz);
		field_id = jenv->GetFieldID(oclazz,
			                   f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
	} 
	assert(field_id);

	// If the field has a primitive type:
	if ((type != 'L') && (type != '[')) {

		jclass clazz;	
		jmethodID method_id;

		if (type == 'B') {
				
					args[0].b = (f->is_static()) ?
						jenv->GetStaticByteField(dclazz, field_id) :
						jenv->GetByteField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Byte object
					clazz = jenv->FindClass("java/lang/Byte");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(B)V");
		
		} else if (type == 'C') {
				
					args[0].c = (f->is_static()) ?
						jenv->GetStaticCharField(dclazz, field_id) :
						jenv->GetCharField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Character object
					clazz = jenv->FindClass("java/lang/Character");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(C)V");

		} else if (type == 'D') {
					
					args[0].d = (f->is_static()) ?
						jenv->GetStaticDoubleField(dclazz, field_id) :
						jenv->GetDoubleField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Double object
					clazz = jenv->FindClass("java/lang/Double");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(D)V");

		} else if (type == 'F') {

					args[0].f = (f->is_static()) ?
						jenv->GetStaticFloatField(dclazz, field_id) :
						jenv->GetFloatField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Float object
					clazz = jenv->FindClass("java/lang/Float");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(F)V");

		} else if (type == 'I')	{
					
					args[0].i = (f->is_static()) ?
						jenv->GetStaticIntField(dclazz, field_id) :
						jenv->GetIntField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Integer object
					clazz = jenv->FindClass("java/lang/Integer");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(I)V");

		} else if (type == 'J') {

					args[0].j = (f->is_static()) ?
						jenv->GetStaticLongField(dclazz, field_id) :
						jenv->GetLongField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Long object
					clazz = jenv->FindClass("java/lang/Long");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(J)V");

		} else if (type == 'S') {

					args[0].s = (f->is_static()) ?
						jenv->GetStaticShortField(dclazz, field_id) :
						jenv->GetShortField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Short object
					clazz = jenv->FindClass("java/lang/Short");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(S)V");

		} else if (type == 'Z') {
			
					args[0].z = (f->is_static()) ?
						jenv->GetStaticBooleanField(dclazz, field_id) :
						jenv->GetBooleanField(obj, field_id);

					if (jenv->ExceptionOccurred()) 
						return (jobject)0;   

					// Allocate java/lang/Boolean object
					clazz = jenv->FindClass("java/lang/Boolean");	
					method_id = jenv->GetMethodID(clazz, "<init>",	"(Z)V");
		} else {
			// Wrong value for "type"	
			assert(0);
		}
		
		// Construct new Java object and initialize it with the field value: 
		assert(clazz);
		assert(method_id);
		retobj = jenv->NewObjectA(clazz, method_id, args);

	} else { // If the field is of reference type:
		retobj = (f->is_static()) ? 
			jenv->GetStaticObjectField(dclazz, field_id) :
			jenv->GetObjectField(obj, field_id);	
	} 

	if (jenv->ExceptionOccurred()) 
		return (jobject)0;   

	assert(retobj);
	return retobj;
} // Java_java_lang_reflect_Field_get



/*
 * Class:     java_lang_reflect_Field
 * Method:    setByte
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;B)V
 */


JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setByte
  (JNIEnv *jenv, jobject jfield, jobject obj, jbyte byte)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a byte type:
	if (!PrimitiveIsAssignableFrom(type, 'B')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticByteField(dclazz, field_id, byte);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetBooleanField(obj, field_id, byte);
	}
} // Java_java_lang_reflect_Field_setByte



/*
 * Class:     java_lang_reflect_Field
 * Method:    setShort
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;S)V
 */


JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setShort
  (JNIEnv *jenv, jobject jfield, jobject obj, jshort s)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a short type:
	if (!PrimitiveIsAssignableFrom(type, 'S')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticShortField(dclazz, field_id, s);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetShortField(obj, field_id, s);
	}
} // Java_java_lang_reflect_Field_setShort


/*
 * Class:     java_lang_reflect_Field
 * Method:    setInt
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;I)V
 */


JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setInt
  (JNIEnv *jenv, jobject jfield, jobject obj, jint i)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from an int type:
	if (!PrimitiveIsAssignableFrom(type, 'I')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticIntField(dclazz, field_id, i);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetIntField(obj, field_id, i);
	}
} // Java_java_lang_reflect_Field_setInt



/*
 * Class:     java_lang_reflect_Field
 * Method:    setLong
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;J)V
 */

JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setLong
  (JNIEnv *jenv, jobject jfield, jobject obj, jlong l)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a long type:
	if (!PrimitiveIsAssignableFrom(type, 'J')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticLongField(dclazz, field_id, l);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetLongField(obj, field_id, l);
	}
} // Java_java_lang_reflect_Field_setLong



/*
 * Class:     java_lang_reflect_Field
 * Method:    setFloat
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;F)V
 */

JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setFloat
  (JNIEnv *jenv, jobject jfield, jobject obj, jfloat ff)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a float type:
	if (!PrimitiveIsAssignableFrom(type, 'F')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticFloatField(dclazz, field_id, ff);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetFloatField(obj, field_id, ff);
	}

} // Java_java_lang_reflect_Field_setFloat



/*
 * Class:     java_lang_reflect_Field
 * Method:    setDouble
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;D)V
 */

JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setDouble
  (JNIEnv *jenv, jobject jfield, jobject obj, jdouble d)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a double type:
	if (!PrimitiveIsAssignableFrom(type, 'D')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticDoubleField(dclazz, field_id, d);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetDoubleField(obj, field_id, d);
	}
} // Java_java_lang_reflect_Field_setDouble


/*
 * Class:     java_lang_reflect_Field
 * Method:    setChar
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;C)V
 */


JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setChar
  (JNIEnv *jenv, jobject jfield, jobject obj, jchar c)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a char type:
	if (!PrimitiveIsAssignableFrom(type, 'C')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticCharField(dclazz, field_id, c);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetCharField(obj, field_id, c);
	}
} // Java_java_lang_reflect_Field_setChar



/*
 * Class:     java_lang_reflect_Field
 * Method:    setBoolean
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;Z)V
 */


JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setBoolean
  (JNIEnv *jenv, jobject jfield, jobject obj, jboolean z)
{
	jfieldID field_id = 0;
	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
                             "field is final");
		return;
	}
	
	char type = f->get_signature()->descriptor->bytes[0];
	assert(type);
	
	// If field is not assignable from a boolean type:
	if (!PrimitiveIsAssignableFrom(type, 'Z')) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
			                 "field type mismatch");
		return;
	}

	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);

		// If field is static, acquire pointer to the field in the
		// class static data block:
 		field_id = jenv->GetStaticFieldID(dclazz,
		                             f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetStaticBooleanField(dclazz, field_id, z);
	}
	else { // else acquire pointer to the field in the object itself:

		// Access handle of parameter object's class:
	    jclass oclazz = jenv->GetObjectClass(obj);
		assert(oclazz);

		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
		assert(field_id);
		jenv->SetBooleanField(obj, field_id, z);
	}
} // Java_java_lang_reflect_Field_setBoolean


/*
 * Class:     java_lang_reflect_Field
 * Method:    set
 * Signature: (Ljava/lang/Class;Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Class;)V
 */

JNIEXPORT void JNICALL Java_java_lang_reflect_Field_set
  (JNIEnv *jenv, jobject jfield, jobject obj, jobject value)
{
    jfieldID field_id;
	jclass dclazz =0;  // Class declaring this field
	jclass tclazz =0;  // Class of the type of this field
	jclass vclazz =0;  // Class of the value argument

	Field* f = GetAndVerifyField(jenv, jfield, obj); 
	assert(f);

	// Field cannot be set if it is final:
	if (f->is_final()) {
		throw_exception_from_jni(jenv, "java/lang/IllegalAccessException",
					         "field is final");
		return;
	}

	// Check if the passed value object is null:
	if (!value) {
		//throw_exception_from_jni(jenv, "java/lang/NullPointerException", 0);
		//return;
        //assert it's object field
        if(f->is_static()){
            dclazz = GetFieldClass(jenv, jfield);
			assert(dclazz);
            jenv->SetStaticObjectField(dclazz, (jfieldID)f, NULL);
        }else
            jenv->SetObjectField(obj, (jfieldID)f, NULL);
        return;
	}

	// Access the class pointer of the value object:
	vclazz = GetObjectClass(jenv, value);
	assert(vclazz);

	Object_Handle_Struct *vclazz_h = (Object_Handle_Struct *)vclazz;
    Class *vcl = (Class *)vclazz_h->java_reference;
	assert(vcl);
    
    /*
	// Access the type class pointer of this field:
	jclass field_clazz = jenv->FindClass("java/lang/reflect/Field");	
	assert(field_clazz);
    jfieldID type_id = jenv->GetFieldID(field_clazz, "type","Ljava/lang/Class;");
	assert(type_id);
    tclazz = jenv->GetObjectField(jfield, type_id);
    Object_Handle_Struct *tclazz_h = (Object_Handle_Struct *)tclazz;
    Class *tcl = (Class *)tclazz_h->java_reference;
    */
    tclazz = Java_java_lang_reflect_Field_getType(jenv, jfield);
    Object_Handle_Struct *tclazz_h = (Object_Handle_Struct *)tclazz;
    Class *tcl = (Class *)tclazz_h->java_reference;
    
	// If the field has a primitive type:
	int fprim = tcl->is_primitive;
	char vsig;

	// Check if the value object argument represents a wrapper
    // class for a primitive type:
	int	vprim = ((vsig = IsWrapperClass(vcl->name->bytes)) != '\0');

	// If one is primitive, but the other is not:
	if (fprim && !vprim) {
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException",
							 "unwrapping conversion failed");
		return;
	}	


	// Get the field's fieldID:
	if (f->is_static()) {
		// Access handle of class declaring this field
	    jclass dclazz = GetFieldClass(jenv, jfield);
		assert(dclazz);
		field_id = jenv->GetStaticFieldID(dclazz,
			                         f->get_signature()->name->bytes,
									 f->get_signature()->descriptor->bytes);
	}
	else {
		// Access handle of parameter object's class:
	    jclass oclazz = GetObjectClass(jenv, obj);
		assert(oclazz);
		field_id = jenv->GetFieldID(oclazz,
		                       f->get_signature()->name->bytes,
							   f->get_signature()->descriptor->bytes);
	}
	assert(field_id);

	if (!fprim) {
        // Field is not primitive:
        // See if the field is assignable by the argument value object:
		// check if the value object is an instance of the type of field:

		if (!(jenv->IsAssignableFrom(vclazz, tclazz))) {
			throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
				                 "field type mismatch");
			return;
		}	
		else {
			if (f->is_static()) {
				// Access handle of class declaring this field
				dclazz = GetFieldClass(jenv, jfield);
				assert(dclazz);
				// Set the field in the class static data block:
				jenv->SetStaticObjectField(dclazz, field_id, value);
			}
			else { // Set the field in the object itself:
				jenv->SetObjectField(obj, field_id, value);
			}
		}
	}
	else { // Field is primitive and argument value object represents a primitive:type

		char type = f->get_signature()->descriptor->bytes[0];

		if (!PrimitiveIsAssignableFrom(type, vsig)) {
			throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException", 
				                 "field type mismatch");
			return;
		}

        switch (vsig) { // Check receiver field type
        case 'B':
			{
				// Get the fieldID of the value field of a Byte object:
				jclass byte_clazz = jenv->FindClass("java/lang/Byte");	
				assert(byte_clazz);
				jfieldID value_id = jenv->GetFieldID(byte_clazz, "value", "B");
				jbyte b = jenv->GetByteField(value, value_id);
				
				if (f->is_static()) {
					// If field is static, acquire pointer to the field in the
					// class static data block:
					switch (type) {
					case 'B':
						jenv->SetStaticByteField(dclazz, field_id, b);
						break;
					case 'C':
						jenv->SetStaticCharField(dclazz, field_id, b);
						break;
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, b);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, b);
						break;
					case 'I':
						jenv->SetStaticIntField(dclazz, field_id, b); 
						break;
					case 'J': 
						jenv->SetStaticLongField(dclazz, field_id, b);
						break;
					case 'S':
						jenv->SetStaticShortField(dclazz, field_id, b);
						break;
					}
				}
				else { // else acquire pointer to the field in the object itself:
					switch (type) {
					case 'B':
						jenv->SetByteField(obj, field_id, b);
						break;
					case 'C':
						jenv->SetCharField(obj, field_id, b);
						break;
					case 'D':
						jenv->SetDoubleField(obj, field_id, b);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, b);
						break;
					case 'I':
						jenv->SetIntField(obj, field_id, b);
						break;
					case 'J': 
						jenv->SetLongField (obj, field_id, b);
						break;
					case 'S':
						jenv->SetShortField(obj, field_id, b);
						break;
					}
				}
				break;
			}
		case 'C':
			{
				// Get the fieldID of the value field of a Char object:
				jclass char_clazz = jenv->FindClass("java/lang/Char");	
				assert(char_clazz);
				jfieldID value_id = jenv->GetFieldID(char_clazz, "value", "C");
				jchar c = jenv->GetCharField(value, value_id);
			
				if (f->is_static()) {
					switch (type) {
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, c);
						break;
					case 'J': 
						jenv->SetStaticLongField(dclazz, field_id, c);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, c);
						break;
					case 'I':
						jenv->SetStaticIntField(dclazz, field_id, c);
						break;
					case 'S':
						jenv->SetStaticShortField(dclazz, field_id, c);
						break;
					case 'C':
						jenv->SetStaticCharField(dclazz, field_id, c);
						break;
					}
					
				}
				else {
					switch (type) {
					case 'D':
						jenv->SetDoubleField(obj, field_id, c);
						break;
					case 'J': 
						jenv->SetLongField(obj, field_id, c);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, c);
						break;
					case 'I':
						jenv->SetIntField(obj, field_id, c);
						break;
					case 'S':
						jenv->SetShortField(obj, field_id, c);
						break;		
					case 'C':
						jenv->SetCharField(obj, field_id, c);
						break;
					}
				}
				break;
			}
        case 'S':
			{
				// Get the fieldID of the value field of a Short object:
				jclass short_clazz = jenv->FindClass("java/lang/Short");	
				assert(short_clazz);
				jfieldID value_id = jenv->GetFieldID(short_clazz, "value", "S");
				jshort s = jenv->GetShortField(value, value_id);
				
				if (f->is_static()) {
					switch (type) {
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, s);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, s);
						break;
					case 'I':
						jenv->SetStaticIntField(dclazz, field_id, s);
						break;
					case 'J': 
						jenv->SetStaticLongField(dclazz, field_id, s);
						break;
					case 'S':
						jenv->SetStaticShortField(dclazz, field_id, s);
						break;
					}
					
				}
				else {
					switch (type) {
					case 'D':
						jenv->SetDoubleField(obj, field_id, s);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, s);
						break;
					case 'I':
						jenv->SetIntField(obj, field_id, s);
						break;
					case 'J': 
						jenv->SetLongField(obj, field_id, s);
						break;
					case 'S':
						jenv->SetShortField(obj, field_id, s);
						break;
					}
				}
				break;
			}
        case 'I':
			{
				// Get the fieldID of the value field of a Int object:
				jclass int_clazz = jenv->FindClass("java/lang/Int");	
				assert(int_clazz);
				jfieldID value_id = jenv->GetFieldID(int_clazz, "value", "I");
				jint i = jenv->GetIntField(value, value_id);
				
				if (f->is_static()) {
					switch (type) {
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, i);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, (jfloat)i);
						break;
					case 'I':
						jenv->SetStaticIntField(dclazz, field_id, i);
						break;
					case 'J':
						jenv->SetStaticLongField(dclazz, field_id, i);
						break;
					}
				}
				else {
					switch (type) {
					case 'D':
						jenv->SetDoubleField(obj, field_id, i);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, (jfloat)i);
						break;
					case 'I':
						jenv->SetIntField(obj, field_id, i);
						break;
					case 'J':
						jenv->SetLongField(obj, field_id, i);
					}
				}
				break;
			}
        case 'J':
			{
				// Get the fieldID of the value field of a Int object:
				jclass long_clazz = jenv->FindClass("java/lang/Long");	
				assert(long_clazz);
				jfieldID value_id = jenv->GetFieldID(long_clazz, "value", "J");
				jlong j = jenv->GetLongField(value, value_id);
				
				if (f->is_static()) {
					switch (type) {
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, (jdouble)j);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, (jfloat)j);
						break;
					case 'J':
						jenv->SetStaticLongField(dclazz, field_id, j);
						break;
					}
				}
				else {
					switch (type) {
					case 'D':
						jenv->SetDoubleField(obj, field_id, (jdouble)j);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, (jfloat)j);
						break;
					case 'J':
						jenv->SetLongField(obj, field_id, j);
					}
				}	
				break;
			}
        case 'F':
			{
				// Get the fieldID of the value field of a Float object:
				jclass float_clazz = jenv->FindClass("java/lang/Float");	
				assert(float_clazz);
				jfieldID value_id = jenv->GetFieldID(float_clazz, "value", "F");
				jfloat fval = jenv->GetFloatField(value, value_id);

				if (f->is_static()) {
					switch (type) {
					case 'D':
						jenv->SetStaticDoubleField(dclazz, field_id, fval);
						break;
					case 'F':
						jenv->SetStaticFloatField(dclazz, field_id, fval);
						break;
					}
				}
				else {
					switch (type) {
					case 'D':
						jenv->SetDoubleField(obj, field_id, fval);
						break;
					case 'F':
						jenv->SetFloatField(obj, field_id, fval);
						break;
					}
				}
				break;
			}
        case 'D':
			{
				// Get the fieldID of the value field of a Double object:
				jclass double_clazz = jenv->FindClass("java/lang/Double");	
				assert(double_clazz);
				jfieldID value_id = jenv->GetFieldID(double_clazz, "value", "D");
				jdouble d = jenv->GetDoubleField(value, value_id);

				if (f->is_static()) {
					jenv->SetStaticDoubleField(dclazz, field_id, d);
				}
				else {
					jenv->SetDoubleField(obj, field_id, d);
				}
				break;
			}
        }
	}
} // Java_java_lang_reflect_Field_set

