
// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/ir.h,v 1.24 2002/01/10 03:12:09 xhshi Exp $
//

#ifndef _IR_H_
#define _IR_H_

#include <iostream.h>
#include "assert.h"
#include "Mem_Manager.h"
#include "../x86/x86.h"
#include "local_reg_alloc.h"
#include "dlink.h"

#include "jit_intf.h"
#include "jit_types.h"
#include "x86_emitter.h"
#include "code_emitter.h"

#define IS_64BIT(ty) ((ty) == JAVA_TYPE_DOUBLE || (ty) == JAVA_TYPE_LONG)
#define IS_FP_DBL_TYPE(ty) ((ty) == JIT_TYPE_FLOAT || (ty) == JIT_TYPE_DOUBLE)
#define IS_BYTE_SIZE(ty) ((ty) == JIT_TYPE_BYTE || (ty) == JIT_TYPE_BOOLEAN)
#define SAME_DST_SRC_REG(dst,src) \
        ((dst) == (src) || ((dst)->is_reg() && (dst)->assigned_preg() == (src)->assigned_preg()))

#define CONVERT_TO_INT_JAVA(t)   \
do { switch (t)             \
{                           \
case JAVA_TYPE_CHAR:        \
case JAVA_TYPE_BYTE:        \
case JAVA_TYPE_SHORT:       \
case JAVA_TYPE_BOOLEAN:     \
    t = JAVA_TYPE_INT;      \
    break;                  \
case JAVA_TYPE_ARRAY:       \
    t = JAVA_TYPE_CLASS;    \
    break;                  \
default:                    \
    break;                  \
}} while (0)

#define CONVERT_TO_INT_JIT(t)   \
do { switch (t)             \
{                           \
case JIT_TYPE_CHAR:         \
case JIT_TYPE_BYTE:         \
case JIT_TYPE_SHORT:        \
case JIT_TYPE_BOOLEAN:      \
    t = JIT_TYPE_INT;       \
    break;                  \
case JIT_TYPE_ARRAY:        \
    t = JIT_TYPE_CLASS;     \
    break;                  \
default:                    \
    break;                  \
}} while (0)


//#ifdef INLINE_NATIVE
struct NativeCode{
	char** code ;
} ;
//#endif

class Cfg_Node;       // forward declaration
class Cfg_Extra_Info; // forward declaration
class Flow_Graph;
class Expressions;

class Operand {
public:
    enum Kind {
        GCTrack,  // something whose live range needs to be tracked - Reg_Operand and Spill_Operand
        Immediate,
        Field, 
        Array,
        Static,
        Const,
        Status
    };
    const /*Kind*/unsigned char kind;
    const O3_Jit_Type type;
    unsigned /*short*/ bvp;  // bit vector position
    Operand(Kind k, O3_Jit_Type t) : kind((unsigned char)k), type(t), _hi_opnd(NULL), bvp((unsigned)-1) 
	{
//#ifdef INLINE_NATIVE
        foldable = true ;
//#endif
	}
//#ifdef INLINE_NATIVE
    virtual void set_foldable(bool b) { foldable = b ;}
    virtual bool get_foldable() { return foldable ; }
//#endif
	virtual bool is_reg()                 {return false;}
	virtual bool is_aliased_across_call() {return false;}
	virtual bool is_mem()                 {return false;}
    virtual bool is_array()               {return false;}
    virtual bool is_physical_reg()        {return false;}
    virtual bool is_temp_reg()            {return false;}
    virtual bool is_single_def_temp_reg() {return false;}
    virtual bool is_vreg()                {return false;}
    virtual bool is_ret()                 {return false;}
    virtual bool is_arg()                 {return false;}
    virtual bool is_foldable()            {return false;}
    virtual bool is_status_flags()        {return false;}
    virtual bool is_fp_stk()              {return false;}
    virtual bool contain_reg(unsigned reg_id) {return false;}
    virtual bool was_originally_array_opnd()  {return false;}
    virtual void print(ostream& cout) = 0;
	virtual bool need_kill_set()          {return true;}
    virtual void starts_live_range(Local_Reg_Manager& rm) {rm.starts_live_range(this);}
    virtual void starts_live_range_from_dst(Local_Reg_Manager& rm) {}
    virtual void find_local_reg_cand(int src_in_reg, unsigned marker) {}
    virtual void fold_operand(Folding& fold) {}
    virtual Operand *base() { return NULL; }
    virtual Operand *index() { return NULL; }
    virtual X86_Reg_No assigned_preg() {return n_reg;}
    void    set_hi_opnd(Operand *hi) {_hi_opnd = hi;}
    Operand *hi_opnd() {return _hi_opnd;} 
    bool  is_hi() {return IS_64BIT(type) && _hi_opnd == NULL;}
    unsigned bv_position() { assert(bvp != (unsigned)-1); return bvp; }
    virtual Operand *fp_return_loc() {return NULL;}
    //
    // code emission routines
    //
    virtual void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
                 const R_Opnd *const dst)   {assert(0);}
    virtual void emit_cmov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
                 const R_Opnd *const dst, X86_CC cc, unsigned is_signed)   {assert(0);}
	virtual void emit_mov_to_mem(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
                 const M_Opnd *const dst, X86_Opnd_Size sz=opnd_32){assert(0);}
    virtual void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 X86_ALU_Opcode opc, Operand* dst) {assert(0);}
//#ifdef INLINE_NATIVE
	virtual void emit_mul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 Operand*dst) {assert(0);}
//#endif
    virtual void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 Operand*dst) {assert(0);}
    virtual void emit_imul_inst_3(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 Operand*dst, Operand *imm) {assert(0);}
    virtual void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& xopnds){assert(0);}
    virtual void emit_pop(O3_Emitter& emitter,X86_Opnd_Pool& xopnds) {assert(0);}
    virtual void emit_test(O3_Emitter& emitter,X86_Opnd_Pool& xopnds){assert(0);}
    virtual void emit_shift(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 X86_Shift_Opcode opc,Operand*dst) {assert(0);}
    virtual void emit_fld(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds) {assert(0);}
    virtual void mov_fp_stk_to(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds, unsigned pop) {assert(0);}
    virtual void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
                 X86_FP_Opcode opc, bool pop_stk0, Operand* dst) {assert(0);}
    virtual Inst *lookup_exp(Expressions& exprs, Inst *inst_head) {assert(0);return NULL;}
private:
    Operand *_hi_opnd; // hi part of operands with long type
//#ifdef INLINE_NATIVE
    bool foldable ;
//#endif
};

class Imm_Operand : public Operand {
public:
    Imm_Operand(unsigned i, O3_Jit_Type ty) : _imm(i), Operand(Immediate,ty) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    unsigned imm()  {return _imm;}
    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const R_Opnd *const dst,bool ok_with_carry);
private:
    unsigned _imm;

    void print(ostream& cout) {cout << _imm;}
	//
	// Since imm cannot be killed, no kill set is required
	//
	bool  need_kill_set() {return false;} 

    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const R_Opnd *const dst);
	void emit_mov_to_mem(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const M_Opnd *const dst, X86_Opnd_Size sz);
    void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& xopnds);
    void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_ALU_Opcode opc, Operand*dst);
    void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
    void emit_shift(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_Shift_Opcode opc,Operand*dst);
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
};

class GCTrack_Operand : public Operand {
public:
    GCTrack_Operand(unsigned i, O3_Jit_Type t) : id(i), orig_id(i), Operand(GCTrack, t),
        which_subr(NULL), _home_location((unsigned)-1), _use_arg_home_loc(0),
        _home_location_assigned(0) { bvp = id;}
//    const unsigned id; // unique id
    unsigned id; // unique id
    const unsigned orig_id;
#if 0
    unsigned bv_position() {
        Operand *fp_loc = fp_return_loc();
        if (fp_loc != NULL) return fp_loc->bv_position();

        X86_Reg_No regno = assigned_preg();
        return (regno < n_reg) ? (unsigned) regno : id; }
#endif // 0
    virtual void set_assigned_preg(X86_Reg_No no) { assert(0); }
    virtual int  global_reg_alloc_cand() {return 1;}
    virtual bool is_spill() { return false; }
    virtual bool is_global_fp_cand() { return IS_FP_DBL_TYPE(type); }
    Cfg_Node *which_subr; // if type=RETADDR, which subroutine it returns from
    int home_location() { return _home_location; }
    void set_home_location(int loc) { _home_location = loc; }
    void set_use_arg_home_loc() {_use_arg_home_loc = 1;}
    bool use_arg_home_loc() {return _use_arg_home_loc == 1;}
    bool home_location_assigned() { return _home_location_assigned == 1; }
    void set_home_location_assigned() { _home_location_assigned = 1; }
protected:
    int _home_location : 30;
    unsigned _use_arg_home_loc:1;
    unsigned _home_location_assigned:1;
};

//
// register operand (local and global)
//
class Reg_Operand : public GCTrack_Operand {
public:
    Reg_Operand(unsigned i,unsigned is_rcand, O3_Jit_Type t)
      :  GCTrack_Operand(i,t)
      // : 
      //_global_reg_alloc_cand(is_rcand), GCTrack_Operand(i,t) {
      {
	xxx = 0;
	yyy._global_reg_alloc_cand = is_rcand;
	yyy._assigned_physical_reg = (is_physical_reg())? id : n_reg;
          if (yyy._assigned_physical_reg != n_reg) bvp = yyy._assigned_physical_reg;
          yyy._assigned_entry = NOT_YET_ASSIGNED;
          yyy._assign_local_reg = NULL;
          yyy._temp_reg_has_multiple_defs = NULL;
          yyy._no = NULL;
          _globally_allocated_fp = false;
      }
    bool is_reg() {return true;}
    bool contain_reg(unsigned reg_id) {return id == reg_id;}
    void set_global_reg_cand()   {yyy._global_reg_alloc_cand = 1;}
    int  global_reg_alloc_cand() {return yyy._global_reg_alloc_cand;}
    unsigned assigned_entry() {return yyy._assigned_entry;}
    void set_assigned_entry(unsigned i) {
        assert(i < NOT_YET_ASSIGNED);
        yyy._assigned_entry = i;
    }
    void set_assigned_preg(X86_Reg_No no) {
        assert(!is_physical_reg());
        yyy._assigned_physical_reg = no;
        bvp = no;
    }
    void set_assign_local_reg() {yyy._assign_local_reg = 1;}
    void set_no(unsigned n) {assert(n < 0xFFFF); yyy._no = n;}
    unsigned get_no() {return yyy._no;}
    void print(ostream& cout) {
        if (yyy._assigned_physical_reg != n_reg && !is_arg() && !is_ret()) 
             cout << X86_Reg_Str[yyy._assigned_physical_reg];
        else print_symbolic(cout);
    }
    void copy_attr_from(Reg_Operand* reg) {
        xxx = reg->xxx;
        yyy._assigned_physical_reg = reg->yyy._assigned_physical_reg;
        if (yyy._assigned_physical_reg != n_reg) bvp = yyy._assigned_physical_reg;
    }
    virtual void print_symbolic(ostream& cout) = 0;
    virtual Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
    X86_Reg_No assigned_preg() {return (X86_Reg_No) yyy._assigned_physical_reg;}
    bool is_globally_allocated_fp() { return _globally_allocated_fp; }
    void set_globally_allocated_fp() { _globally_allocated_fp = true; }
protected:
    union {
        struct {
            unsigned _assigned_physical_reg:8;
            unsigned _global_reg_alloc_cand:1; // set if it is a global reg alloc candidate
            unsigned _assign_local_reg:1; // set, then local reg alloc will try to find a reg
            unsigned _assigned_entry:5;   // for local register allocation
            unsigned _temp_reg_has_multiple_defs:1; // if set, we should consider it as a vreg
            unsigned _no:16; // for splitting cfg_node (temp_reg been split different blocks)
        } yyy;
        unsigned xxx;
    };
    //xxxxxx
    bool _globally_allocated_fp; // This should be moved into an "xxx" bit above.
    //
    // routines for emitting code
    //
    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst);
    void emit_cmov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst, X86_CC cc, unsigned is_signed);
	void emit_mov_to_mem(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const M_Opnd *const dst, X86_Opnd_Size sz);
    void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_ALU_Opcode opc, Operand*dst);
//#ifdef INLINE_NATIVE
    void emit_mul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
//#endif
    void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
    void emit_imul_inst_3(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst,Operand*imm);
    void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& xopnds);
    void emit_pop(O3_Emitter& emitter,X86_Opnd_Pool& xopnds);
    void emit_test(O3_Emitter& emitter, X86_Opnd_Pool& xopnds);
    void emit_shift(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_Shift_Opcode opc,Operand*dst);
    void emit_fld(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void mov_fp_stk_to(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds, unsigned pop);
    void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_FP_Opcode opc, bool pop_stk0, Operand* dst);
};

//
// if global register allocation fails to assign a physical reg to the virtual reg,
// the access of the virtual reg is turned into a memory access
//
class Virtual_Reg : public Reg_Operand {
public:
    Virtual_Reg(unsigned i,O3_Jit_Type t) : Reg_Operand(i,1,t) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    bool is_vreg() {return true;}
private:
    void print_symbolic(ostream& cout) {
        if (type == JAVA_TYPE_DOUBLE)   cout << "d" << id;
        else if (type == JAVA_TYPE_FLOAT)    cout << "f" << id;
        else cout << "r" << id;
    }
};

//
// by default, Temp_Regs are not global reg allocation candidates
//
class Temp_Reg : public Reg_Operand {
public:
    Temp_Reg(unsigned i,O3_Jit_Type t) : Reg_Operand(i,0,t), _inst(NULL) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    void set_inst(Inst *i) {
		_inst = i;
	}
    Inst *inst() {return _inst;}
    void set_temp_reg_has_multiple_defs() {yyy._temp_reg_has_multiple_defs = 1;}
    bool  is_temp_reg() {return true;}
private:
    Inst *_inst; // the inst that defines the temp register
    void print_symbolic(ostream& cout) {cout << "t" << id;}
//    bool  is_temp_reg() {return yyy._temp_reg_has_multiple_defs == NULL;}
    bool  is_single_def_temp_reg() {return yyy._temp_reg_has_multiple_defs == 0;}
    bool  is_foldable();
    void find_local_reg_cand(int src_in_reg, unsigned marker);
};

class Physical_Reg : public Reg_Operand {
public:
    Physical_Reg(X86_Reg_No reg_no, O3_Jit_Type ty) : Reg_Operand((unsigned)reg_no,0,ty) {
        yyy._assigned_physical_reg = reg_no; spill = NULL;
    }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    X86_Reg_No assigned_preg() {return (X86_Reg_No)id;}
    Operand *spill;
private:
    void print_symbolic(ostream& cout) {cout << X86_Reg_Str[id];}
	bool  is_aliased_across_call() {return is_scratch_x86reg((X86_Reg_No)id) == 1;}
    bool  is_physical_reg() {return true;}
    bool  is_global_fp_cand() {return false;}
};

//
// argument (incoming and outgoing)
//
class Arg_Operand : public Reg_Operand {
public:
    Arg_Operand(unsigned arg_i,unsigned i, O3_Jit_Type t) 
        : num((unsigned short)arg_i), Reg_Operand(i,0,t) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
	const unsigned short num;  // ith argument
private:
    void print_symbolic(ostream& cout) {
        cout << "arg" << num;
        if (yyy._assigned_physical_reg != n_reg)
             cout << ":" << X86_Reg_Str[yyy._assigned_physical_reg];
    }
	bool  is_aliased_across_call() {return true;}
    bool  is_arg() {return true;}
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
};

//
// return value
//
class Ret_Operand : public Reg_Operand {
public:
    Ret_Operand(unsigned i, O3_Jit_Type t) 
        : _fp_return_loc(NULL), Reg_Operand(i,0,t) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    void assign_call_convention_reg() {
        if (yyy._assigned_physical_reg != n_reg || IS_FP_DBL_TYPE(type))
            return;
        if (type == JIT_TYPE_LONG && hi_opnd() == NULL) // hi 32bit
            bvp = yyy._assigned_physical_reg = edx_reg;
        else
            bvp = yyy._assigned_physical_reg = eax_reg;
    }
    void set_fp_return_loc(Operand *fp_loc) {_fp_return_loc = fp_loc; bvp = fp_loc->bv_position(); }
    Operand *fp_return_loc() {return _fp_return_loc;}
private:
    Operand *_fp_return_loc;
    void print_symbolic(ostream& cout) {
        cout << "Ret." << (char)type;
        if (yyy._assigned_physical_reg != n_reg)
             cout << ":" << X86_Reg_Str[yyy._assigned_physical_reg];
    }
	bool  is_aliased_across_call() {return true;}
    bool  is_ret() {return true;}
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
};

//
// FP stack operand 
//
class Fp_Stk : public Reg_Operand {
public:
    Fp_Stk(unsigned j, unsigned i, O3_Jit_Type t) : stk(j),Reg_Operand(i,0,t){}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    const unsigned stk;
private:
    void print_symbolic(ostream& cout) {cout << "fp_stk(" << stk << ")";}
	bool is_aliased_across_call() {return true;}
    bool is_fp_stk() {return true;}
    void mov_fp_stk_to(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds, unsigned pop);
    void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_FP_Opcode opc, bool pop_stk0, Operand* dst);
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head) {assert(0);return NULL;} // never happen
    bool is_global_fp_cand() {return false;}
};

class Mem_Operand : public Operand {
public:
    Mem_Operand(Operand::Kind k, O3_Jit_Type ty) : Operand (k,ty) {}

	bool  is_aliased_across_call() {return true;}
	bool  is_mem() {return true;}
    //
    // routines for emitting code
    //
    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst);
    void emit_cmov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst, X86_CC cc, unsigned is_signed);
    void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_ALU_Opcode opc,Operand*dst);
    void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
    void emit_imul_inst_3(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst,Operand*imm);
    void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_pop(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_fld(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void mov_fp_stk_to(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds, unsigned pop);
    void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_FP_Opcode opc, bool pop_stk0, Operand* dst);
};

//
// array operand 
//
class Array_Operand : public Mem_Operand {
public:
    Array_Operand(Reg_Operand *base, Reg_Operand *index, Operand *bound,
        unsigned s, unsigned o, O3_Jit_Type t) : _shift((unsigned char)s), _offset(o), 
        _base(base), _index(index), _bound(bound), Mem_Operand(Array,t) {}
    bool contain_reg(unsigned reg_id) {
        return _base->contain_reg(reg_id) || _index->contain_reg(reg_id) ;
    }
    Operand *base()  { return _base; }
    Operand *index() { return _index;}
    Operand *bound() { return _bound;}
    unsigned shift() { return _shift;}
    unsigned offset(){ return _offset;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    //
    // private fields
    //
    Reg_Operand *_base;
    Reg_Operand *_index;
    Operand *_bound;
    unsigned char _shift;
    unsigned _offset;

    bool is_array() {return true;}
    void print(ostream& cout) {        // print [b+i*1+8]
        cout << "["; _base->print(cout);
        cout << "+"; _index->print(cout);
        cout << "*" << (1<<_shift) << "+" << _offset << "]";
    }
    void starts_live_range(Local_Reg_Manager& rm) {
        rm.starts_live_range(_base);
        rm.starts_live_range(_index);
    }
    void starts_live_range_from_dst(Local_Reg_Manager& rm) {starts_live_range(rm);}
    void find_local_reg_cand(int src_in_reg, unsigned marker) {
        assert(src_in_reg == 0);
        _base->find_local_reg_cand(1,marker);  // both base and index must be in regs
        _index->find_local_reg_cand(1,marker);
    }
    void fold_operand(Folding& fold);
};

typedef void *FIELD_UID;

//
// field operand
//
class Field_Operand : public Mem_Operand {
public:
    Field_Operand(Reg_Operand *base,unsigned o,O3_Jit_Type t,FIELD_UID f,bool e) :
      _offset(o), _base(base), _was_originally_array_opnd(false), fid(f),
      _may_throw_excp(e), Mem_Operand(Field,t) {}
    bool contain_reg(unsigned reg_id) {return _base->contain_reg(reg_id);}
    Operand *base()   {return _base;}
    unsigned offset() {return _offset;}
    void set_was_originally_array_opnd() {_was_originally_array_opnd = true;}
    bool was_originally_array_opnd() {return _was_originally_array_opnd;}
    void reset_throw_null_ptr_excp() {_may_throw_excp = false;}
    bool may_throw_null_ptr_excp()   {return _may_throw_excp;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    FIELD_UID const fid;  
private:
    //
    // private fields
    //
    Reg_Operand *_base;
    unsigned _offset;

    bool _was_originally_array_opnd;
    bool _may_throw_excp;  // may throw null pointer exception

    void print(ostream& cout) { // print [b+16]   
        cout << "["; _base->print(cout); cout << "+" << _offset << "]";
    }
    void starts_live_range(Local_Reg_Manager& rm) {
        rm.starts_live_range(_base);
    }
    void starts_live_range_from_dst(Local_Reg_Manager& rm) {starts_live_range(rm);}
    void find_local_reg_cand(int src_in_reg, unsigned marker) {
        assert(src_in_reg == 0);
        _base->find_local_reg_cand(1,marker);  // base must be in reg
    }
    void fold_operand(Folding& fold) {
        fold.replace_reg_opnd(_base);
        if (type == JIT_TYPE_DOUBLE && !is_hi() != 0)
            hi_opnd()->fold_operand(fold);
    }
};


//
// static operand 
//
class Static_Operand : public Mem_Operand {
public:
    Static_Operand(void *a, O3_Jit_Type t, Field_Handle fh) : 
      addr(a), _fh(fh), Mem_Operand(Static,t) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}

    void *const addr;
    Field_Handle get_field_handle() { return _fh; }
private:
    void print(ostream& cout) {cout << "S[" << addr << "]";}
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
    Field_Handle _fh;
};

//
// spill_loc operand 
//
class Spill_Operand : public GCTrack_Operand {
public:
    Spill_Operand(unsigned i, O3_Jit_Type t) : GCTrack_Operand(i,t) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    int global_reg_alloc_cand() {return 0;}
    void print(ostream& cout) {cout << "spill_loc(" << id << ")";}
	bool is_mem() {return true;}
    bool is_spill() {return true;}
    bool is_global_fp_cand() {return false;}
    //
    // routines for emitting code
    // should use multiple inheritance (GCTrack and Mem)
    //
    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst);
    void emit_cmov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         const R_Opnd *const dst, X86_CC cc, unsigned is_signed);
    void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_ALU_Opcode opc,Operand*dst);
    void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
    void emit_imul_inst_3(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst,Operand*imm);
    void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_pop(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_fld(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void mov_fp_stk_to(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds, unsigned pop);
    void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_FP_Opcode opc, bool pop_stk0, Operand* dst);
};

union Value {
	int i;
	float f;
	double d;
    void *addr;
	struct l {
		unsigned lo;
		unsigned hi;
	} l;
};

//
// constant operands (double or float constants) 
//
class Const_Operand : public Operand {
public:
    Const_Operand(Value* v, O3_Jit_Type t) 
        : Operand(Const,t), val(*v), _data_label(NULL) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}

    const Value val;
    void set_lo(Imm_Operand* lo) {_lo = lo;}
    void set_hi(Imm_Operand* hi) {_hi = hi;}
    Imm_Operand *lo() {return _lo;}
    Imm_Operand *hi() {return _hi;}
    bool need_data_space() {
        return type == JIT_TYPE_FLOAT || type == JIT_TYPE_DOUBLE;
    }
    unsigned data_space_needed() {
        if (type == JIT_TYPE_FLOAT) return 1;
        else if (type == JIT_TYPE_DOUBLE) return 3; // 1 extra for alignment
        assert(0);
        return 0;
    }
    void set_data_label(char *label) {_data_label = label;}
    char *data_label() { return _data_label; }
private:
    Imm_Operand *_lo; // for long operand
    Imm_Operand *_hi; // for long operand
    void print(ostream& cout);
	//
	// Since constant cannot be killed, no kill set is required
	//
	bool  need_kill_set() {return false;} 
    char *_data_label;  // label in data block
    //
    // routines for emitting code
    //
    void emit_mov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const R_Opnd *const dst);
    void emit_cmov_to_reg(O3_Emitter& emitter,X86_Opnd_Pool& xopnds,
         const R_Opnd *const dst, X86_CC cc, unsigned is_signed);
    void emit_alu_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_ALU_Opcode opc, Operand*dst);
    void emit_imul_inst(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst);
    void emit_imul_inst_3(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,Operand*dst,Operand*imm);
    void emit_push(O3_Emitter& emitter,X86_Opnd_Pool& xopnds);
    void emit_pop(O3_Emitter& emitter,X86_Opnd_Pool& xopnds);
    void emit_data(O3_Data_Emitter& data_emitter);
    void emit_fld(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_fp_op(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds,
         X86_FP_Opcode opc, bool pop_stk0, Operand* dst);
    Inst *lookup_exp(Expressions& exprs, Inst *inst_head);
};

extern unsigned next_global_id;

//
// status flags generated by compare instruction 
//
class Status_Flags : public Operand {
public:
    Status_Flags(unsigned i, O3_Jit_Type t) : id(i), _lt(false), _eq(false), _gt(false),
        Operand(Status,t) { bvp = i; }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    const unsigned id;
    bool is_known() { return (_lt || _eq || _gt); }
    bool is_lt() { return _lt; }
    bool is_eq() { return _eq; }
    bool is_gt() { return _gt; }
    void set_lt() { _lt = true; _eq = false; _gt = false; }
    void set_eq() { _lt = false; _eq = true; _gt = false; }
    void set_gt() { _lt = false; _eq = false; _gt = true; }
private:
    void print(ostream& cout) {cout << "sflags" << id;}
    bool  is_status_flags() {return true;}
    bool _lt, _eq, _gt;
};

#define MAX_SRCS 3
#ifdef PRINTABLE_O3
#define INIT_INST() \
    for (int i=n_srcs; i<MAX_SRCS; i++) _srcs[i] = NULL; \
    bc_index = ~0u; \
    emitter_offset = (unsigned)-1; \
    is_spill = 0; \
    _dead = 0;  \
    _fp_spilled = 0; \
    live_ranges_ended = 0; \
    region_live_ranges_ended = 0; \
    region_dst_live_range_ends = 0; \
    _been_expanded = 0; \
    global_id = next_global_id++; \
    _newly_created = 1; \
    _marker = 1; \
    _gc_unsafe = 0; \
    _target_of_optimized_sw_br = 0; \
    _dst = NULL
#else // PRINTABLE_O3
#define INIT_INST() \
    for (int i=n_srcs; i<MAX_SRCS; i++) _srcs[i] = NULL; \
	_dead = 0;  \
    _fp_spilled = 0; \
    live_ranges_ended = 0; \
    region_live_ranges_ended = 0; \
    region_dst_live_range_ends = 0; \
    _been_expanded = 0; \
    _newly_created = 1; \
    _marker = 1; \
    _gc_unsafe = 0; \
    _target_of_optimized_sw_br = 0; \
    _dst = NULL
#endif // PRINTABLE_O3

class Exp;         // forward declaration
class Expressions; // forward declaration
class Inst : public Dlink {
public:
    struct Info {
        char *fmt_string;
        char src_in_reg[MAX_SRCS];
        char is_unary;  // requires dst==src(0) [inc,dec,not,neg; unlike mov]
    };

	Exp *const exp;
    const unsigned char n_srcs; // num of source operands
    unsigned char live_ranges_ended;
    unsigned char region_live_ranges_ended;
#ifdef PRINTABLE_O3
    unsigned bc_index;  // bytecode that this instruction originated from
    unsigned emitter_offset;
    char is_spill;
    unsigned global_id;
#endif // PRINTABLE_O3

    Inst() : n_srcs(0), exp(0) {
        INIT_INST();
    }
    Inst (Exp *e, Inst *inst_head) : n_srcs(0), exp(e) {
        insert_before(inst_head); // append to the end of the inst list
        INIT_INST();
    }
    Inst(Operand *src0,Exp *e, Inst *inst_head) : n_srcs(1), exp(e){
        insert_before(inst_head); // append to the end of the inst list
        _srcs[0] = src0;
        INIT_INST();
    }
    Inst(Operand *src0,Operand *src1, Exp *e, Inst *inst_head) : 
	n_srcs(2), exp(e) {
        insert_before(inst_head); // append to the end of the inst list
        _srcs[0] = src0; _srcs[1] = src1;
        INIT_INST();
    }
    Inst(Operand *src0, Operand *src1, Operand *src2, Exp *e, Inst *inst_head) : 
	n_srcs(3), exp(e) {
        insert_before(inst_head); // append to the end of the inst list
        _srcs[0] = src0; _srcs[1] = src1; _srcs[2] = src2;
        INIT_INST();
    }
    void set_dst(Operand *dst) {
        if (dst->is_temp_reg())
            ((Temp_Reg*)dst)->set_inst(this);
        _dst = dst;
    }
    virtual Operand *dst() {return _dst;}
    Operand *src(unsigned i) {assert(i < n_srcs); return _srcs[i];}
    void replace_src(unsigned i, Operand *src) {assert(i <= n_srcs);_srcs[i] = src;}
    bool is_64bit();  // if the inst produces a 64-bit value
    void print(ostream& cout);
    const char *fmt_string() {return info()->fmt_string;}
    bool is_imm_assignment() {
        return is_assignment() && _srcs[0]->kind == Operand::Immediate;
    }
    bool is_reg_assignment() {
        return is_assignment() && _srcs[0]->is_reg();
    }
    bool is_const_assignment() {
        return is_assignment() && _srcs[0]->kind == Operand::Const;
    }
    bool is_outgoing_arg_assignment() {
        return is_assignment() && _dst->is_arg();
    }
    bool is_same_reg_copying() {
        return is_reg_assignment() && 
            ((_dst == _srcs[0]  && !_dst->is_arg())||
            (((Reg_Operand*)_dst)->assigned_preg() == ((Reg_Operand*)_srcs[0])->assigned_preg() &&
            ((Reg_Operand*)_dst)->assigned_preg() != n_reg));
    }
    bool is_gc_unsafe()  {return _gc_unsafe;}
    void set_gc_unsafe() {_gc_unsafe = 1;}
    void mark_dead()     {_dead = 1;}
    void unmark_dead()   {_dead = 0;}
    int  is_dead()       {return _dead;}
    int  been_expanded() {return _been_expanded; }
    Inst *default_long_expansion(Mem_Manager& mem);
    Inst *expand(Expressions& exprs);
    virtual void bounds_expansion(Expressions& exprs, Cfg_Node *node, Flow_Graph *fg) {}
    void set_expanded() {_been_expanded = 1;}
    bool is_newly_created() {return _newly_created == 1;}
    void reset_newly_created() {_newly_created = 0;}
    bool is_marked() {return _marker == 1;}
    void reset_marker() {_marker = 0;}
    bool fp_spilled() { return _fp_spilled; }
    void set_fp_spilled() { _fp_spilled = 1; }
    void set_target_of_optimized_switch_br() {_target_of_optimized_sw_br = 1;}
    bool is_target_of_optimized_sw_br() {return _target_of_optimized_sw_br == 1;}
    //
    // return true, if any operand (src or dst) may throw null pointer exception
    //
    bool may_throw_null_ptr_excp() {
        return false;
        int i;
        for (i = 0; i < n_srcs; i++)
            if (_srcs[i]->kind == Operand::Field &&
                ((Field_Operand*)_srcs[i])->may_throw_null_ptr_excp())
                return true;
        if (_dst != NULL && _dst->kind == Operand::Field &&
            ((Field_Operand*)_dst)->may_throw_null_ptr_excp())
            return true;
        return false;
    }

    // 
    // virtual functions
    //
#ifdef STAT_INDIRECT_CALL
	virtual bool is_stat_call()		{return false ;}
#endif
    virtual O3_Jit_Type type();
    virtual bool contain_reg(unsigned reg_id) {
        int i;
        for (i = 0; i < n_srcs; i++)
                if (_srcs[i]->contain_reg(reg_id)) return true;
        return false;
    }
    virtual bool is_ret_from_jsr()        {return false;}
    virtual bool is_jump()                {return false;}
    virtual void escape(ostream& cout)    {}
    virtual bool is_add()                 {return false;}
    virtual bool is_sub()                 {return false;}
    virtual bool is_branch()              {return false;}
    virtual bool is_inlinable_call()      {return false;}
    virtual bool is_call()                {return false;}
    virtual bool is_type_inst()           {return false;}
    virtual bool is_assignment()          {return false;}
    virtual bool is_pseudo_asgn()         {return false;}
    virtual bool is_switch()              {return false;}
    virtual bool is_widen()               {return false;}
    virtual bool is_iinc()                {return false;}
    virtual bool is_field_inc()           {return false;}
    virtual bool is_iinc_and()            {return false;}
    virtual bool is_nextpc()              {return false;}
    virtual bool is_push()                {return false;}
    virtual bool exits_method()           {return false;}
    virtual bool changes_ret_addr()       {return false;}
    virtual bool is_fp_pop()              {return false;}
    virtual bool is_compare()             {return false;}
    virtual bool is_bitwise()             {return false;}
    virtual bool is_fprem()               {return false;}
    virtual bool is_fneg()                {return false;}
    virtual bool is_idiv()                {return false;}
    virtual bool is_mul()                 {return false;}
    virtual bool is_div_inst()            {return false;}
    virtual bool is_obj_info()            {return false;}
    virtual bool can_use_lea()            {return false;}
    virtual bool must_use_lea()           {return false;}
    virtual bool is_commutable()          {return false;}
    virtual bool affects_flags()          {return true;}  // whether it modifies status flags
    virtual bool sets_zero_flag()         {return false;} // whether it ZERO flag
//#ifdef INLINE_NATIVE
	virtual bool is_native_inline()       {return false;} // candidate for native inlining
	virtual char** get_native_code()	  {return NULL;}
	virtual Operand** get_native_args(Expressions& exprs, int& len)       {return NULL;} //expand/modify all long srcs for native-inline
//#endif
    virtual bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1,
        Operand *src2, Operand *&new_rhs) {return false;}

    virtual const Info *info()            {return NULL;}
    virtual bool need_special_expansion() {return false;}
    virtual bool is_deref()               {return false;}
    virtual bool can_eliminate()          {return !may_throw_null_ptr_excp();}
    virtual void dont_eliminate()         {assert(0);}
    virtual void set_changes_ret_addr(bool value) {}
    virtual void set_fp_pop_after_compute(){assert(0);}
    virtual void set_must_use_lea()       {assert(0);}
    virtual int  esp_effect()             {return 0;}
    virtual Inst *special_expansion(Expressions& exprs) {assert(0); return NULL;}
    virtual Inst *expand_inst(Mem_Manager& mem,Operand *dst,Operand *s0,
        Exp *e,Inst *i){assert(0); return this;}
    virtual Inst *expand_inst(Mem_Manager& mem,Operand *dst,Operand *s0,
        Operand *s1,Exp *e,Inst *i) {assert(0); return this;}
    virtual Inst *expand_inst(Mem_Manager& mem,Operand *dst,Operand *s0,
        Operand *s1,Operand *s2,Exp *e,Inst *i){assert(0); return this;}
    virtual Inst *simplify(Mem_Manager &mem, Expressions &exprs);
    void emit_inst(O3_Emitter& xe, X86_Opnd_Pool& opnds) {
#ifdef PRINTABLE_O3
        emitter_offset = xe.get_size();
#endif // PRINTABLE_O3
        emit(xe, opnds); }

    Inst *next() {return (Inst *)_next;}
    Inst *prev() {return (Inst *)_prev;}

    //
    // estimated size of memory that is needed 
    //
    unsigned char region_dst_live_range_ends:1;
protected:
    virtual void emit(O3_Emitter& xe, X86_Opnd_Pool& opnds) {assert(0);}
    unsigned char _dead:1;
    unsigned char _fp_spilled:1;
    unsigned char _been_expanded:1;
    unsigned char _marker:1;
    unsigned char _gc_unsafe:1;
    unsigned char _newly_created:1;
    unsigned char _target_of_optimized_sw_br:1;
    Operand *_srcs[MAX_SRCS];
    Operand *_dst;
};

#ifdef STAT_INDIRECT_CALL
class Stat_Call_Inst : public Inst {
public:
    Stat_Call_Inst(Exp *e, Inst *inst_head) :  Inst(e,inst_head) {} // void return
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
	bool is_stat_call()		{return true; }
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds) ;
};
#endif

#ifdef O3_VTune_Support
class VTune_Call_Inst : public Inst {
public:
    VTune_Call_Inst(Exp *e, Inst *inst_head, bool enter) :  Inst(e,inst_head){ m_enter = enter; } // void return
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
	bool is_stat_call()		{return true; }
	bool m_enter;
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds) ;
    Inst *special_expansion(Expressions& exprs);
};
#endif

//
// Widen_Inst (widen a source inst)
//
class Widen_Inst : public Inst {
public:
    Widen_Inst(Operand *src, Exp *e, Inst *inst_head);
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    bool is_signed() { return _is_signed; }
    bool is_half() { return _is_half; }
    void set_signed(bool sign) { _is_signed = sign; }
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
    bool is_widen() { return true; }
    bool affects_flags() { return false; }
    O3_Jit_Type type() {return _dst->type;}
    //
    // private fields
    //
    bool _is_signed;  // signed extended
    bool _is_half;
};

//
// assignment inst
//
class Assign_Inst : public Inst {
public:
    Assign_Inst(Operand *dst, Operand *src, Exp *e, Inst *inst_head) : 
      Inst(src,e,inst_head), real_type(JIT_TYPE_UNKNOWN) {
		//
		// To avoid globle register being setted a cross-bb inst,
		// add more conditions //::sxh 2001.6.27
		//
		if (dst->is_temp_reg()){
			if(dst->is_single_def_temp_reg())
				((Temp_Reg*)dst)->set_inst(this);
			//else
			//	((Temp_Reg*)dst)->set_inst(NULL);
		}
        _dst = dst; _srcs[0] = src;
        _pseudo_asgn = false;  // by default, generate the assignment
        _use_pseudo_asgn_value = false;
        _ok_with_carry = true;
        _is_fp_pop = false;
        _can_eliminate = true;
        _do_pop = true;
        _aastore_call = false;
        _cmov = false;
//#ifdef INLINE_NATIVE
        is_copy_prop = true ;
//#endif
      }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    O3_Jit_Type real_type;
    bool is_assignment() {return true;}
    void arg_expansion(Expressions& exprs, Method_Handle mh,
                       Inst*& new_arg1, Inst*& new_arg2); // expand arg0 = t
    void disable_pseudo_asgn() {_pseudo_asgn = false;}
    void enable_pseudo_asgn()  {_pseudo_asgn = true;}
    bool is_pseudo_asgn() {return _pseudo_asgn /*|| !_use_pseudo_asgn_value*/;}
    bool use_pseudo_asgn_value() {return _use_pseudo_asgn_value;}
    void reset_ok_with_carry() { _ok_with_carry = false;}
    void set_is_fp_pop() { _is_fp_pop = true; }
    bool is_fp_pop() { return _is_fp_pop; }
    void set_no_pop_on_fp_store() { _do_pop = false; }
    bool is_iinc_and();
    bool is_field_inc();
    void set_aastore_call() {_aastore_call = true;}
    void set_cmov(unsigned char cc, unsigned is_signed) {
        _cmov = true; _condition = cc; _is_signed_cmov = is_signed; }
//#ifdef INLINE_NATIVE
    void set_copy_prop(bool b) { is_copy_prop = b ;}
//#endif
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool _pseudo_asgn;
    bool _use_pseudo_asgn_value;
    unsigned char _condition;
    bool _changes_ret_addr:1;
    bool _ok_with_carry:1;
    bool _is_fp_pop:1;
    bool _can_eliminate:1;
    bool _aastore_call:1;
    bool _do_pop:1;
    bool _cmov:1;
    unsigned _is_signed_cmov:1;
//#ifdef INLINE_NATIVE
    bool is_copy_prop ;
//#endif

    bool can_eliminate()  {return _can_eliminate && !may_throw_null_ptr_excp();}
    void dont_eliminate() { _can_eliminate = false; }
    bool changes_ret_addr() { return _changes_ret_addr;}
    bool affects_flags() { return false; }
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs) {
//#ifdef INLINE_NATIVE
        if (!is_copy_prop) return false ;
//#endif
        if (is_reg_assignment() || is_imm_assignment()) { new_rhs = src0; return true; }
        return false;
    }
    void set_changes_ret_addr(bool value) { _changes_ret_addr = value; }
    Operand *dst() {
        if (_pseudo_asgn == true)
            return _srcs[0];
        else {
            _use_pseudo_asgn_value = true;
            return _dst;
        }
    }
    O3_Jit_Type type() {return _dst->type;}
    bool need_special_expansion();
    Inst *special_expansion(Expressions& exprs);
    Inst *expand_inst(Mem_Manager& mem, Operand *dst, Operand *src, Exp *e, Inst *i) {
        return new (mem) Assign_Inst(dst,src,e,i);
    }
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
};

//
// info about an oject, e.g., length of array and vtable 
//
class Obj_Info_Inst : public Inst {
public:
	enum Kind {length,vtable,intfcvtable,n_obj_info};
	const Kind kind;
    Obj_Info_Inst(Kind op,Operand *base, Exp *e, Inst *inst_head) : 
	kind(op), Inst(base,e,inst_head) {}
    Obj_Info_Inst(Kind op,Operand *ch, Operand *ths, Exp *e, Inst *inst_head) : 
    kind(op), Inst(ch,ths,e,inst_head) {assert(op == intfcvtable);}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    bool need_special_expansion() {return true;}
    bool is_obj_info()            {return true;}
    Inst *special_expansion(Expressions& exprs);
    const static Info _info[n_obj_info];
    const Info *info() {return &_info[kind];}
};

//
// branch against zero
//
class Branch_Inst : public Inst {
public:
    enum Kind {beq,bne,blt,bge,bgt,ble,bp,bnp,n_br};
    const bool is_signed;
    Branch_Inst(Kind k,bool is_signed,Operand *src, Exp *e,
        Inst *inst_head, Cfg_Node *target=NULL) : 
        _kind((unsigned char)k), is_signed(is_signed), _branch_fallthrough(false), _bound_branch(false),
            _target(target), _cfg_idx(-1), Inst(src,e,inst_head),
            _jump_over_next_br(false), _can_commute(true) {}
    Kind kind() {return (Kind)_kind;}
    void set_fallthrough() {_branch_fallthrough = true;}
    void set_cfg_idx(int idx) {_cfg_idx = idx;}
    bool branch_fallthrough() {return _branch_fallthrough;}
    void set_bound_branch() {_bound_branch = true;}
    bool is_bound_branch() {return _bound_branch;}
    void set_jump_over_next_branch() {_jump_over_next_br = true;}
    void commute_condition();
    bool can_commute() { return _can_commute; }
    void dont_commute() { _can_commute = false; }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    const static X86_CC _br_cc[n_br];
private:
    /*Kind*/unsigned char _kind;
    bool _branch_fallthrough;
    bool _bound_branch; // true if this is a bound branch
    bool _jump_over_next_br; // the target is the following inst of the next branch
    bool _can_commute;
    int  _cfg_idx;
    Cfg_Node *_target; // must be NULL unless it's a bounds-check branch
    const static Info _info[n_br];
    const Info *info() {return &_info[_kind];}
    bool  is_branch() {return true;}
    void  emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
};

//
// compare 
//
class Compare_Inst : public Inst {
public:
    enum Kind {cmp,cmp_lt,cmp_gt,test,n_cmp};
    const /*Kind*/unsigned char kind;
    const bool is_cmpg;  // Whether it's fcmpg/dcmpg versus fcmpl/dcmpl
    Compare_Inst(Kind k,bool g,Operand *src0, Operand *src1, Exp *e, bool gen_branch, Inst *inst_head) : 
      kind((unsigned char)k), is_cmpg(g), gen_branch(gen_branch), Inst(src0,src1,e,inst_head),
          _two_pops(false), _compare_vtable_of_obj(NULL), _method_inlined(NULL),
          _can_eliminate(true) {}
    Compare_Inst(Kind k,bool g,Operand *src, Exp *e, bool gen_branch, Inst *inst_head) : 
      kind((unsigned char)k), is_cmpg(g), gen_branch(gen_branch), Inst(src,e,inst_head),
          _two_pops(false), _compare_vtable_of_obj(NULL), _method_inlined(NULL),
          _can_eliminate(true) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    const bool gen_branch; // Generate a branch inst for bounds checking during expansion.
    void set_double_pop() { _two_pops = true; }
    bool is_vtable_compare() { return (_compare_vtable_of_obj != NULL); }
    Operand *object_compared_with_vtable() { return _compare_vtable_of_obj; }
    void set_object_compared_with_vtable(Operand *o) { _compare_vtable_of_obj = o; }
    Method_Handle method_inlined() { return _method_inlined; }
    void set_method_inlined(Method_Handle mh) { _method_inlined = mh; }
private:
    const static Info _info[n_cmp];
    const Info *info() {return &_info[kind];}
    const static Branch_Inst::Kind br_lcmp_map[Branch_Inst::n_br][3];
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
    void  emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void bounds_expansion(Expressions& exprs, Cfg_Node *node, Flow_Graph *fg);
    bool is_compare() { return true; }
    bool can_eliminate() { return _can_eliminate && !may_throw_null_ptr_excp(); }
    void dont_eliminate() { _can_eliminate = false; }
    bool _two_pops;
    bool _can_eliminate;
    Operand *_compare_vtable_of_obj;
    Method_Handle _method_inlined;
    Inst *simplify(Mem_Manager &mem, Expressions &exprs);
};

//
// add instruction
//
class Add_Inst : public Inst {
public:
    enum Kind {add,adc,fadd,n_add};
    const /*Kind*/unsigned char kind;

    Add_Inst(Kind k,Operand *src0, Operand *src1,Exp *e, Inst *inst_head) : 
      kind((unsigned char)k), _can_eliminate(true), _fp_pop_after_compute(false), _must_use_lea(false),
          Inst(src0,src1,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
	const static Info _info[n_add];
    const static X86_ALU_Opcode _ia32[n_add];
    const Info *info() {return &_info[kind];}
    bool  is_iinc() {return _dst == _srcs[0];}
    bool  is_add() {return true;}
    Inst *expand_inst(Mem_Manager& mem, Operand *dst, Operand *s0, Operand *s1,
        Exp *e, Inst *i) {
        assert(kind == add);
        return new (mem) Add_Inst(adc,s0,s1,e,i);
    }
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void dont_eliminate() { _can_eliminate = false; }
    bool can_eliminate() { return _can_eliminate && !may_throw_null_ptr_excp(); }
    bool _can_eliminate;
    bool _fp_pop_after_compute;
    bool _must_use_lea;
    void set_fp_pop_after_compute() { _fp_pop_after_compute = true; }
    int esp_effect();
    bool can_use_lea();
    bool must_use_lea() { return _must_use_lea; }
    void set_must_use_lea() { _must_use_lea = true; }
    bool is_commutable() { return true; }
    bool affects_flags();
    bool sets_zero_flag() {return kind != fadd;} // add and adc
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// sub instruction
//
class Sub_Inst : public Inst {
public:
    enum Kind {sub,sbb,fsub,n_sub};
    const Kind kind;

    Sub_Inst(Kind k,Operand *src0, Operand *src1, Exp *e, Inst *inst_head) : 
      kind(k),_fp_pop_after_compute(false),  _must_use_lea(false),
          Inst(src0,src1,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
	const static Info _info[n_sub];
    const Info *info() {return &_info[kind];}
    bool  is_sub() {return true;}
    Inst *expand_inst(Mem_Manager& mem, Operand *dst, Operand *s0, Operand *s1,
        Exp *e, Inst *i) {
        assert(kind == sub);
        return new (mem) Sub_Inst(sbb,s0,s1,e,i);
    }
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool _fp_pop_after_compute;
    bool _must_use_lea;
    void set_fp_pop_after_compute() { _fp_pop_after_compute = true; }
    int esp_effect();
    bool can_use_lea();
    bool must_use_lea() { return _must_use_lea; }
    void set_must_use_lea() { _must_use_lea = true; }
    bool affects_flags();
    bool sets_zero_flag() {return kind != fsub;} // sub and sbb
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// mul instruction
//
class Mul_Inst : public Inst {
public:
//#ifdef INLINE_NATIVE
    enum Kind {mul,fmul,smul,mul_const,n_mul}; //smul is a special mul, which has two I type src and J type dst. 
									 //Now it's used for native-inlining.
#if 0
    enum Kind {mul,fmul,n_mul};
#endif
    const Kind kind;
	
    Mul_Inst(Kind k,Operand *src0, Operand *src1, Exp *e, Inst *inst_head) : 
      kind(k), _fp_pop_after_compute(false), Inst(src0,src1,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
	const static Info _info[n_mul];
    const Info *info() {return &_info[kind];}
//#ifdef INLINE_NATIVE
	static char** native_code[n_mul]; 
	bool is_mul_const;
	char** get_native_code() { 
		if (is_mul_const) 
			return native_code[mul_const]; // for mul_const, it's special
		else 
			return native_code[kind];
	}
	bool is_native_inline()  ; // long mul is replaced with native code
	Operand** get_native_args(Expressions& exprs, int& len) ;
    bool can_eliminate()  {return kind!=smul ;}
//#endif
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool _fp_pop_after_compute;
    void set_fp_pop_after_compute() { _fp_pop_after_compute = true; }
    bool is_commutable() { return true; }
    bool affects_flags();
    bool is_mul() { return true; }
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// div instruction
//
class Div_Inst : public Inst {
public:
    enum Kind {div,fdiv,rem,frem,n_div};
    const Kind kind;

    Div_Inst(Kind k,Operand *s0, Operand *s1, Exp *e, Inst *inst_head) : 
      kind(k),_fp_pop_after_compute(false),Inst(s0,s1,e,inst_head) {}
    Div_Inst(Kind k,Operand *s0, Operand *s1, Operand *s2, Exp *e, Inst *inst_head) : 
      kind(k),_fp_pop_after_compute(false),Inst(s0,s1,s2,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
	const static Info _info[n_div];
    const Info *info() {return &_info[kind];}
//#ifdef INLINE_NATIVE
	static char** native_code[n_div];
	char** get_native_code() {return native_code[kind];}
	bool is_native_inline()  ; // long mul is replaced with native code
	Operand** get_native_args(Expressions& exprs, int& len) ;
//#endif
    bool  need_special_expansion() {return true;}
    bool can_eliminate() { return false; } // could throw exception
    Inst *special_expansion(Expressions& exprs);
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool is_fprem() { return (kind == frem); }
    bool is_div_inst() { return (kind == div || kind == rem); }
    bool _fp_pop_after_compute;
    void set_fp_pop_after_compute() { _fp_pop_after_compute = true; }
    bool affects_flags();
};

//
// negate instruction
//
class Neg_Inst : public Inst {
public:
    enum Kind {neg,fneg,n_neg};
    const Kind kind;

    Neg_Inst(Kind k,Operand *src,  Exp *e, Inst *inst_head) : 
      kind(k),Inst(src,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
	const static Info _info[n_neg];
    const Info *info() {return &_info[kind];}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool affects_flags();
    bool sets_zero_flag() {return kind == neg;} 
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// bitwise instruction
//
class Bitwise_Inst : public Inst {
public:
    enum Kind {k_and,k_or,k_xor,shl,shr,sar,n_bitwise};
    const Kind kind;

    Bitwise_Inst(Kind k,Operand *src0, Operand *src1, Exp *e, Inst *inst_head) : 
      kind(k),Inst(src0,src1,e,inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    bool is_bitwise() { return true; }
private:
	const static Info _info[n_bitwise];
    const Info *info() {return &_info[kind];}
//#ifdef INLINE_NATIVE
	static char** native_code[n_bitwise];
	char** get_native_code() { 
		return native_code[kind];
	}
	bool is_native_inline(); // lshl is replaced with native code
	Operand** get_native_args(Expressions& exprs, int& len);
//#endif
    bool  need_special_expansion() {return true;}
    Inst *expand_inst(Mem_Manager& mem, Operand *dst, Operand *s0, Operand *s1,
        Exp *e, Inst *i) {
        assert(kind >= k_and && kind <= k_xor);
        return new (mem) Bitwise_Inst(kind,s0,s1,e,i);
    }
    Inst *special_expansion(Expressions& exprs);
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool is_commutable() { return (kind == k_and || kind == k_or || kind == k_xor); }
    bool sets_zero_flag() {return true;} 
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// method call 
//
class Call_Inst : public Inst {
public:
    enum Kind {virtual_call,    // invokevirtual
               special_call,    // invokespecial
               new_call,        // new
               newarray_call,   // newarray
               anewarray_call,  // anewarray
               multinew_call,   // multianewarray
               static_call,     // invokestatic
               monenter_call,   // monitor enter
               monexit_call,    // monitor exit
               monenter_static_call,   // monitor enter for static methods
               monexit_static_call,    // monitor exit  for static methods
               athrow_call,     // athrow
               athrow_lazy_call,     // lazy athrow
               interface_call,  // invokeinterface
               llsh_call,       // long shift left
               lrsh_call,       // long arith shift right
               lrsz_call,       // long unsigned shifht right
               f2i_call,        // convert float to int
               d2i_call,        // convert double to int
               f2l_call,        // convert float to long
               d2l_call,        // convert double to long
			   frem_call,       // single floating-point remainder
			   drem_call,       // double floating-point remainder
               lmul_call,       // long multiplication
#ifdef ORP_LONG_OPT
			   lmul_const_multiplier_call,    //long multiplication with const multiplier
#endif
               ldiv_call,       // long division
               lrem_call,       // long remainder
               getstring_call,  // string of constant pool
               bounds_call,     // throw ArrayIndexOutOfBoundsException
               checkcast_call,  // checkcast
               checkcast_notnull_call,  // checkcast (obj is not null)
               instanceof_call, // is instance of
               classinit_call,  // initialize class
               resintfc_call,   // resolve interface call
               aastore_call,
               writebarrier_call,
#ifdef JIT_SAPPHIRE
			   readbarrier_call, // READ BARRIER
               acmp_call,       // compare two pointers for sapphire gc
#endif
#ifdef STAT_INDIRECT_CALL
			   stat_indirect_call, //Special call for instrument before indirect branches
#endif
#ifdef O3_VTune_Support
				vtune_method_call,
#endif
				const_ldiv_call , // long division with constant divisor.
				const_lrem_call,  // long rem with constant divisor.
               n_call}; 
    const /*Kind*/ unsigned char kind;

#ifdef STAT_INDIRECT_CALL
	Inst* stat_inst ; // points to the instrumented StatIndirectCall_Inst
#endif

    Call_Inst(Kind op,Operand *addr, Exp *e, void *h, bool is_mh, Inst *inst_head) : 
        kind((unsigned char)op), Inst(addr, e, inst_head), _args(NULL), 
        _n_args(0), _ret(NULL), _handle(h), _is_mh(is_mh) 
#ifdef STAT_INDIRECT_CALL
		,stat_inst(NULL)
#endif
		{}
    Call_Inst(Kind op,Exp *e, void *h, bool is_mh, Inst *inst_head) : 
        kind((unsigned char)op), Inst(e, inst_head), _args(NULL),
        _n_args(0), _ret(NULL), _handle(h), _is_mh(is_mh) 
#ifdef STAT_INDIRECT_CALL
		,stat_inst(NULL)
#endif
		{}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}

    Method_Handle get_mhandle() {return (_is_mh)?(Method_Handle)_handle:NULL;}
    Class_Handle  get_chandle() {return (_is_mh)?NULL:(Class_Handle)_handle;}
    void set_args(Inst **args, unsigned n_args, Inst *ret) {
        _args = args; _n_args = (unsigned short)n_args; _ret = ret; _sz_of_arg_array = (unsigned short)n_args;}
    void set_args(Inst **args, unsigned n_args, unsigned sz_array, Inst *ret) {
        _args = args; _n_args = (unsigned short)n_args; _ret = ret; _sz_of_arg_array = (unsigned short)sz_array;}
    unsigned n_args() { return _n_args; }
    Inst *get_arg(unsigned n) { return _args[n]; }
    Operand *get_arg_opnd(unsigned n);
    void replace_arg(unsigned n, Inst *i) { _args[n] = i;}
    Inst *get_ret()           { return _ret; }
    void kill_ret() { _ret = NULL; }

#ifdef JIT_SAPPHIRE
	O3_Jit_Type wb_kind;  // what kind of writebarrier/readbarrier calls
#endif

private:
    const static Info _info[n_call];
    const Info *info() {return &_info[kind];}
    bool is_call() {return true;}
    bool is_inlinable_call(){
    	Method_Handle h = (_is_mh)?(Method_Handle)_handle:NULL;
 	if (h && method_is_native(_handle)) return false;

        return kind == virtual_call ||
               kind == special_call ||
               kind == static_call;
    }
    bool exits_method() {
        return (kind == athrow_call || kind == athrow_lazy_call || kind == bounds_call); }
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
    bool need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);

    Inst **_args;
    unsigned short _n_args;
    unsigned short _sz_of_arg_array;
    Inst *_ret;
    //
    // true if _handle is a method handle (virtual, special, static, 
    // or interface). Otherwise, _handle is a class handle (new or newarray)
    //
    bool _is_mh;  
    void *_handle; // method handle or class handle

    int esp_effect();
};

//
// return 
//
class Return_Inst : public Inst {
public:
    Return_Inst(Exp *e, Inst *inst_head) :  Inst(e,inst_head) {} // void return
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
};

//
// conversion inst (e.g., long to float)
//
class Convt_Inst : public Inst {
public:
    Convt_Inst(Operand *src, Exp *e, Inst *inst_head) : Inst(src, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void  escape(ostream& cout);
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs);
};

//
// the next PC of the _inst (for jsr)
//
class NextPC_Inst : public Inst {
public:
    NextPC_Inst(Exp *e, Inst *inst_head) : Inst(e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    void set_inst(Inst *i) {_inst = i;}
private:
    Inst *_inst;
    bool is_nextpc() { return true; }
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    const static Info _info;
    const Info *info() {return &_info;}
};

class lookupswitch_info;
//
// jump instruction (for jsr and goto)
//
class Jump_Inst : public Inst {
public:
    enum Kind {jump,jump_s,jump_switch,n_jump};
    const /*Kind*/unsigned char kind;
    Jump_Inst(Exp *e, Inst *inst_head) : kind((unsigned char)jump),
        _branch_fallthrough(false), _sw(NULL), Inst(e, inst_head) {}
    Jump_Inst(Kind k, Operand *src,Exp *e, Inst *inst_head) : kind((unsigned char)k),
        _branch_fallthrough(false), _sw(NULL), Inst(src,e, inst_head) {}
    void set_fallthrough() {_branch_fallthrough = true;}
    bool branch_fallthrough() {return _branch_fallthrough;}
    bool is_ret_from_jsr() { return (kind == jump_s); }
    void set_switch_info(lookupswitch_info *s) {_sw = s;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    bool _branch_fallthrough;
    lookupswitch_info *_sw;
    const static Info _info[n_jump];
    const Info *info() {return &_info[kind];}
    void emit(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_jump(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_jump_s(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    void emit_jump_switch(O3_Emitter& emitter,X86_Opnd_Pool& x86_opnds);
    bool is_jump() {return true;}
};

class tableswitch_info;
class lookupswitch_info; 
//
// switch instruction (tableswitch and lookupswitch)
//
class Switch_Inst : public Inst {
public:
    enum Kind {table,lookup,n_switch};
    const Kind kind;
    Switch_Inst(Kind op, Operand *src,Exp *e,Inst *inst_head,
        Cfg_Extra_Info *sw): kind(op), sw_info(sw), Inst(src, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    Inst *special_expansion(Expressions& exprs, Cfg_Node *node);
private:
    Cfg_Extra_Info *const sw_info;
    const static Info _info[n_switch];
    const Info *info() {return &_info[kind];}
    bool is_switch() { return true; }
    bool  need_special_expansion() {return true;}
    Inst *expand_tableswitch(Expressions& exprs, Cfg_Node *node, int low, int high);
    Inst *expand_lookupswitch(Expressions& exprs, Cfg_Node *node,
                              lookupswitch_info *sw);
    Inst *turn_into_switch_table(Expressions& exprs, Cfg_Node *node,
                                 lookupswitch_info *sw);
    Inst *simplify(Mem_Manager &mem, Expressions &exprs) { return this; }
    bool is_copy_prop_assignment(Expressions &exprs, Operand *src0, Operand *src1, Operand *src2, Operand *&new_rhs) {
        return false; }
};

//
// string of the constant pool
//
class String_Inst : public Inst {
public:
    String_Inst(Operand *src0, Operand *src1, Exp *e, Inst *inst_head) 
        : Inst(src0, src1, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};

//
// Class initialization call
//
class Classinit_Inst : public Inst {
public:
    Classinit_Inst(Operand *src, Exp *e, Inst *inst_head)
        : Inst(src, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};

//
// Write barrier call
//
class Writebarrier_Inst : public Inst {
public:
#ifndef JIT_SAPPHIRE
    Writebarrier_Inst(Operand *src1, Operand *src2, Exp *e, Inst *inst_head)
        : Inst(src1, src2, e, inst_head) {}
#else
    Writebarrier_Inst(Operand *src1, Operand *src2, Exp *e, Inst *inst_head)
        : wb_kind(JIT_TYPE_INVALID), Inst(src1, src2, e, inst_head) {}
    Writebarrier_Inst(Operand *src0, Operand *src1, Operand *src2, Exp *e, 
		              O3_Jit_Type wbkind, Inst *inst_head)
        : wb_kind(wbkind), Inst(src0, src1, src2, e, inst_head) {}
	O3_Jit_Type wb_kind; // what kind of writebarriers
#endif
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};

//::
// Read barrier call
//::
#ifdef JIT_SAPPHIRE
class Readbarrier_Inst : public Inst {
public:
    Readbarrier_Inst(Operand *src1, Exp *e, Inst *inst_head)
        : wb_kind(JIT_TYPE_INVALID), Inst(src1, e, inst_head) {}
    Readbarrier_Inst(Operand *src0, Operand *src1, Exp *e, 
		              O3_Jit_Type wbkind, Inst *inst_head)
        : wb_kind(wbkind), Inst(src0, src1, e, inst_head) {}
	O3_Jit_Type wb_kind; // what kind of readbarriers
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};
#endif

//
// Instrumentation for indirect branch statistics
//
#ifdef STAT_INDIRECT_CALL
class StatIndirectCall_Inst : public Inst {
public:
    StatIndirectCall_Inst(Operand *src0, Exp *e, Inst *inst_head)
        : Inst(src0, e, inst_head) {}
    StatIndirectCall_Inst(Operand *src0, Operand *src1, Exp *e, Inst *inst_head)
        : Inst(src0, src1, e, inst_head) {}

    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};
#endif

//
// Type conversion (instanceof/checkcast)
//
class Type_Inst : public Inst {
public:
    enum Kind {cast,instance,n_type};
    const Kind kind;
    Type_Inst(Kind op, Operand *src0, Operand *src1, Exp *e, Inst *inst_head) 
        : kind(op), not_null(false), Inst(src0, src1, e, inst_head) {
//#ifdef INLINE_NATIVE
        offset = -1 ;
//#endif
    }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    void set_obj_not_null() {not_null = true;}
private:
//#ifdef INLINE_NATIVE
    int offset ; //for fast_checkcast
	static char** native_code[n_type];
	char** get_native_code() {return native_code[kind];}
    bool is_native_inline()  ; 
	Operand** get_native_args(Expressions& exprs, int& len) ;
//#endif
    bool  not_null;
    const static Info _info[n_type];
    const Info *info() {return &_info[kind];}
    bool  is_type_inst()           {return true;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
};

//
// push value onto run-time stack (inc esp)
//
class Push_Inst : public Inst {
public:
    Push_Inst(Operand *src, Exp *e, Inst *inst_head) : Inst(src,  e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    bool  need_special_expansion() {return true;}
    Inst *special_expansion(Expressions& exprs);
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
    int esp_effect() { return -4; }
    bool is_push() {return true;}
};

//
// pop value from run-time stack (dec esp)
//
class Pop_Inst : public Inst {
public:
    Pop_Inst(Operand *dst,Exp *e, Inst *inst_head) : Inst(e, inst_head) {
    _dst = dst;
    }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
    int esp_effect() { return 4; }
};

class Fxch_Inst : public Inst {
public:
    // src0 should be stk(0), src1 should not be stk(0)
    Fxch_Inst(Operand *src0, Operand *src1, Exp *e, Inst *inst_head) :
      Inst(src0, src1, e, inst_head) { _dst = src0; }
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info;
    const Info *info() {return &_info;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
};

class Math_Inst: public Inst {
public:
    enum Kind {sin, cos, sqrt, rndint, abs, n_math};
    const Kind kind;
    Math_Inst(Kind op, Operand *src0, Exp *e, Inst *inst_head)
        : kind(op), Inst(src0, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info[n_math];
    const Info *info() {return &_info[kind];}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
};

//
// This instruction just dereferences its source operand to maintain
// precise null-pointer exceptions.
//
class Deref_Inst: public Inst {
public:
    Deref_Inst(Operand *src0, Exp *e, Inst *inst_head) : Inst(src0, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    bool is_deref() { return true; }
    const static Info _info;
    const Info *info() {return &_info;}
    bool can_eliminate() {return false;}
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
};

//
// x86 native instructions 
// Those instructions are created during instruction expansion.
//
class Native_Inst : public Inst {
public:
    enum Kind {cdq,idiv,fchs,fnstsw,sahf,n_native};
    const Kind kind;
    Native_Inst(Kind op, Operand *src0, Exp *e, Inst *inst_head) 
        : kind(op), Inst(src0, e, inst_head) {}
    Native_Inst(Kind op, Operand *src0, Operand *src1, Operand *src2, Exp *e, Inst *inst_head) 
        : kind(op), Inst(src0, src1, src2, e, inst_head) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    const static Info _info[n_native];
    const Info *info() {return &_info[kind];}
    bool can_eliminate() { return (kind != idiv) && !may_throw_null_ptr_excp(); }
    void emit(O3_Emitter& emitter, X86_Opnd_Pool& x86_opnds);
    bool is_fneg() { return (kind == fchs); }
    bool is_idiv() { return (kind == idiv); }
};


class Expressions; 
class Stack;
class Eh_Node;
extern void     build_IR(
    Compile_Handle   compilation_handle,
    Class_Handle  class_handle,
    Method_Handle method_handle,
    bool          gc_requires_write_barriers,
    Mem_Manager&  mem_manager,
    Expressions&  expressions,
    Stack&        stack,  // mimic stack
    Inst          *inst_head,
    const unsigned char *first_bc,      // first bytecode of method
    Cfg_Node      *node,  // current BB
    char *stack_sig_in,
    int stack_sig_in_size,
    char *stack_sig_out,
    int &stack_sig_out_size,
    int &ends_in_return,
    int is_exception_entry,
    int is_extended_bb);

extern void build_incoming_arg_assignment(
    Class_Handle c_handle,
    Method_Handle m_handle, 
    Mem_Manager& mem, 
    Expressions& exprs, 
    Inst *inst_head); 

extern void inline_methods(Compile_Handle cmpl_handle,
                           Mem_Manager&   mem_manager,
                           Expressions&   exprs, 
                           Flow_Graph     *fg,
                           unsigned       accumulated_inlined_bc_size,
                           unsigned&      global_inlined_bc_size,
                           bool           gc_requires_write_barriers);

Inst *ir_make_epilog(Expressions &exprs, Cfg_Node *node, Mem_Manager &mem_manager);

extern JIT_Result build_IR_of_method(
        Compile_Handle cmpl_handle, 
        Class_Handle   c_handle, 
        Method_Handle  m_handle,
        bool           gc_requires_write_barriers,
        Mem_Manager&   mem_manager,
        Expressions&   exprs,
        Flow_Graph*&   fg,  // OUT 
        Eh_Node        *caller_eh,
        Cfg_Node       *caller_subr,
        Flow_Graph     *caller_fg,
        unsigned       caller_bc_idx,
        const Byte     *bytecode_addr, 
        size_t         bytecode_size, 
        unsigned       max_locals,
        unsigned       max_stack,
        unsigned       accumulated_inlined_bc_size,
        unsigned&      global_inlined_bc_size);

extern unsigned local_reg_allocation(Flow_Graph *fg, Expressions &exprs);

#define CAFFEINE_MARK

extern void code_emission(Compile_Handle cmpl_handle, Class_Handle c_handle, 
                      Method_Handle m_handle, Expressions& exprs, 
                      Flow_Graph *fg, unsigned bytecode_size,
                      unsigned args_on_stack
#ifdef CAFFEINE_MARK
                      , bool do_block_padding
#endif
                          );


#endif  //_IR_H_
