// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/inlining.cpp,v 1.28 2002/01/14 10:18:55 xhshi Exp $
//


#include "defines.h"
#include <iostream.h>
#include "ir.h"
#include "expression.h"
#include "flow_graph.h"
#include "inlining.h"
#include "mtable.h"
#include "build_ir_routines.h"
#include "handles.h"
#include "is_subclass_of.h"
#include "../ia32_o1_jit/profiling.h"
#include "tail_recursion.h"
#include "bit_vector.h"
#include "internal_jit_intf.h"

//#ifdef INLINE_NATIVE
#include "nativeparser.h"
#include "nativegen.h"
//#endif

extern O3_Method_Table *O3_itable;

Profile_Rec *o1_method_get_profile_info(Method_Handle m_handle) {
    Small_Method_Info *smi = (Small_Method_Info*)method_get_info_block(m_handle,o1_jit);
    if (smi == NULL) return NULL;
    return smi->prof_rec;
}

static bool search_back_edge_count(Profile_Rec *prof,
                                   unsigned loop_idx,
                                   PROF_COUNTER& cnt) {
    if (loop_idx == NO_BC_IDX) return false;
    unsigned n_entries = prof->n_back_edge;
    unsigned short *bc_idx = (unsigned short*)&prof->back_edge[n_entries];
    unsigned i;
    for (i = 0; i < n_entries; i++) {
        if (bc_idx[i] == loop_idx) break;
        if (i != n_entries) {
            cnt = prof->back_edge[i];
            return true;
        }
    }
    return false;
}

void search_loop_count(Profile_Rec *prof,
                       Cfg_Node *node,
                       PROF_COUNTER& cnt) {
    //
    // find loop tail 
    //
    Cfg_Node *l_hd = node->loop_header;
    Cfg_Int k;
    for (k = 0; k < l_hd->in_edge_size();k++)
        if (l_hd->in_edges(k)->loop_header == l_hd) break;
    assert(k != l_hd->in_edge_size());
    Cfg_Node *n = l_hd->in_edges(k);
    bool found = search_back_edge_count(prof,n->first_bc_idx(),cnt);
    if (!found)
        search_back_edge_count(prof,l_hd->first_bc_idx(),cnt);
}

bool safe_to_inline(Method_Handle mh, Call_Inst *cinst)
{
    return (method_is_static(mh) || 
        method_is_final(mh)  || 
        method_is_private(mh)||
        (class_get_flags(method_get_class(mh)) & ACC_FINAL) ||
        cinst->kind == Call_Inst::special_call);
}

static bool should_deref_before_inline(Method_Handle mh, Call_Inst *cinst)
{
    if (cinst->kind == Call_Inst::special_call) // invokespecial
        return (strcmp(method_get_name(mh), "<init>") ? true : false); // never for <init>
    if (method_is_static(mh))
        return false;  // never for static method
    return true;  // always for anything virtual
}

//
// decide if we want to inline the method
// cur_fg is the flow graph of the current method, NOT the one of the 
// inlined method
static bool inline_policy(Flow_Graph *cur_fg,   // flow graph of the current method 
                          Cfg_Node *node,       // node containing inlined call
                          Method_Handle mh,     // method handle of inlined method
                          unsigned& inline_bc_size,
                          unsigned& global_inline_bc_size,
                          Profile_Rec *inlined_prof,
                          Call_Inst *cinst) {
    if (false ||         // turn off inlining by changing false --> true
        mh == NULL ||    // native method
        node->is_cold()) // infrequently executed path
        return false;
    //
    // get the class handles for caller and callee methods
    //
    Class_Handle s_class = method_get_class(mh);
    Class_Handle c_class = cur_fg->c_handle();
    static Class_Handle security = NULL;
    //
    // find the class handle for the securityManager
    //
    if (security == NULL) {
        Loader_Exception lexc;
        security = class_load_class_by_name_using_system_class_loader(
                   "java/lang/SecurityManager",&lexc);
    }
    //
    // The Java Language Spec. pp 337
    // A method invocation within class C is to invoke a method m within 
    // class S.  If class C and class S have the same class loader and class 
    // S is not SecurityMananger or a subclass of SecurityManager, then
    // we consider inlining the method m.  There is another condition that
    // we can also safely inline a method.  However, we don't consider 
    // implementing that for now.
    // 
    if (class_get_class_loader(c_class) != class_get_class_loader(s_class) ||
        O3_is_subclass_of(s_class, security))
        return false;
    //
    // Don't inline the constructor for an Exception object.
    //
    if (cinst->kind == Call_Inst::special_call &&
        O3_is_subclass_of(s_class, cached_class_handles[ch_java_lang_Throwable]))
        return false;
    //
    // we are only interested in the methods whose byte code sizes are
    // less than 20 bytes (prevent code explorsion).
    //
    unsigned bc_size = method_get_byte_code_size(mh);

    bool hot_method = true;
    if (instrumenting) {
        Profile_Rec *cur_prof = cur_fg->prof_rec();
        if (cur_prof == NULL) {
            cur_prof = o1_method_get_profile_info(cur_fg->m_handle());
            cur_fg->set_prof_rec(cur_prof);
        }
        PROF_COUNTER init_m_policy = recompilation_policy_method;
        PROF_COUNTER m_cnt = init_m_policy; 
        PROF_COUNTER l_cnt = recompilation_policy_loop;
        bool prof_method = false;
        if (cur_prof != NULL) {
            init_m_policy = cur_prof->m_policy;
            if (node->loop_header != NULL && 
                node->loop_header->first_bc_idx() != NO_BC_IDX) {
                //
                // find out execution frequency of the loop
                //
                search_loop_count(cur_prof,node,l_cnt);
                if (l_cnt < recompilation_policy_loop && 
                    l_cnt > recompilation_policy_loop*8/10)
                    hot_method = false;
            } else if (node->loop_header == NULL) {
                m_cnt = cur_prof->m_entry;
                prof_method = true;
                //
                // find out execution frequency of the entry
                //
                if (m_cnt < init_m_policy && 
                    m_cnt > init_m_policy*8/10)
                    hot_method = false;
            }
        }
        //
        // inlined_prof == NULL:  the method is never executed
        // if 200 < m_entry < recompilation_policy_method, then the method is 
        // not called very often
        //
        if (inlined_prof == NULL || 
            (((unsigned)inlined_prof->m_entry) < inlined_prof->m_policy && 
             ((unsigned)inlined_prof->m_entry) > inlined_prof->m_policy*2/10))
            hot_method = false; 
        //
        // use profiling info to determine if the current block is cold so
        // that we can lay out blocks properly
        // inlined_prof == NULL doesn't mean that the current block is never
        // executed because some subclass overwrites the method.
        //
        if (!(method_get_flags(mh) & ACC_NATIVE)) { // not native method
            PROF_COUNTER cnt = 0;
            if (prof_method) {
                if (m_cnt > init_m_policy)
                    cnt = ((unsigned)-1) - m_cnt + init_m_policy;
                else
                    cnt = init_m_policy - m_cnt;
            } else {
                if (l_cnt > recompilation_policy_loop)
                    cnt = ((unsigned)-1) - l_cnt + recompilation_policy_loop;
                else
                    cnt = recompilation_policy_loop - l_cnt;
            }
            if (inlined_prof != NULL &&
                (inlined_prof->m_entry < inlined_prof->m_policy &&
                cnt/(inlined_prof->m_policy - inlined_prof->m_entry) > COLD_PATH_RATIO))
                node->set_cold_code();
       }
    }

    if (!O3_itable->accept_this_method(mh)) return false;
    if (//O3_itable->accept_this_method(mh) && // command line controls 
        !method_is_synchronized(mh) && hot_method &&
        bc_size != 0 &&                 // native methods (bc_size == 0) 
        bc_size <= SMALL_METHOD_SIZE && // don't want to inline big method 
        inline_bc_size + bc_size <= MAX_INLINE_SIZE) { // avoid deep call chains
        if (false || instrumenting || global_inline_bc_size <= 70) {
            global_inline_bc_size += bc_size;
            inline_bc_size += bc_size;
            return true;
        }
    }
    if (!method_is_synchronized(mh) && hot_method &&
        bc_size != 0 &&                 // native methods (bc_size == 0) 
        bc_size <= TINY_METHOD_SIZE) { // avoid deep call chains
        if (safe_to_inline(mh, cinst)) {
            global_inline_bc_size += bc_size;
            inline_bc_size += bc_size;
            return true;
        }
    }
    return false;
}

//#ifdef INLINE_NATIVE
Operand_Exp* get_exp_from_opnd(Expressions& exprs, Operand* opnd )
{
	Operand_Exp* exp = NULL ;
	if(opnd->is_vreg()){
		exp = exprs.lookup_reg_exp( ((Reg_Operand*)opnd)->id,opnd->type,1) ;
	}else if(opnd->is_temp_reg()){
		exp = exprs.lookup_temp_reg_exp((Reg_Operand*)opnd) ;
	}else if(opnd->kind == Operand::Immediate){
        unsigned imm;
        if (opnd->is_single_def_temp_reg())
            imm = ((Imm_Operand*)((Temp_Reg*)opnd)->inst()->src(0))->imm();
        else
            imm = ((Imm_Operand*)opnd)->imm();
        exp = exprs.lookup_imm_exp(imm,opnd->type) ;
    }else{
		assert(0) ;
	}

	return 	exp ;
}

//Link list for label_nodes
struct label_list ;
struct label_list{
	char name[16] ;
	Cfg_Node* node ;
	struct label_list* next ;
} ;
typedef struct label_list Label_List ;
static Label_List* label_list_head = NULL ;
static Label_List* label_list_cur = NULL ;
void clear_label_array()
{
	//label_array_len = 0	;
	label_list_head = NULL ;
	label_list_cur = NULL ;
}

void add_label(Mem_Manager& mem, Cfg_Node* node, char* label)
{
	Label_List* new_label = (Label_List*)mem.alloc(sizeof(Label_List)) ;
	new_label->name[0] = '\0' ;
	if(label) strcpy(new_label->name,label) ;
	new_label->node = node; 
	new_label->next = NULL ;
	if(label_list_head==NULL){
		label_list_head = label_list_cur = new_label ;
	}else{
		label_list_cur->next = new_label ;
		label_list_cur = label_list_cur->next ;
	}
}

void add_end_label(Mem_Manager& mem, Cfg_Node* node)
{
	add_label(mem,node, "end") ;
}

Cfg_Node* lookup_node_by_label(Mem_Manager& lmem, Flow_Graph *fg, Cfg_Node* bb, char* label) 
{
	Cfg_Node* ret_node = NULL ;
	if(label){
		for(Label_List* cur = label_list_head; cur!=NULL; cur=cur->next){
			if(strcmp(cur->name,label)==0){
				return cur->node ;
			}
		}
	}
	ret_node = fg->new_cfg_node(bb) ;
    if (bb->eh_out_edge() != NULL)
        ret_node->add_eh_edge(fg->mem_manager, bb->eh_out_edge());

    if(label) add_label(lmem, ret_node, label) ;

	return ret_node ;
}

void add_edge_between(Mem_Manager& mem, Cfg_Node* prev, Cfg_Node* next)
{
	int in_edge_size = prev->in_edge_size() ;
	int i = 0 ;
	for(i = 0 ; i<in_edge_size ; i++){
		if(next == prev->in_edges(i))
			break ;
	}
	if(i>= in_edge_size)//must have not out_edge from cur_node to label_node
		prev->add_edge(mem, next) ;
}

void add_linear_between(Cfg_Node* cur_node, Cfg_Node* next_node)
{
	next_node->linearization_node()->unlink() ;
	next_node->linearization_node()->insert_after(cur_node->linearization_node()) ;
}

Cfg_Node* build_native_node(Mem_Manager& lmem, Flow_Graph *fg, Cfg_Node* cur_node, char* label, Cfg_Node* linear_node)
{
	Mem_Manager& mem = fg->mem_manager ;
	Cfg_Node* label_node = lookup_node_by_label(lmem, fg, linear_node, label) ;
	add_linear_between(linear_node, label_node) ;
	if(cur_node){
		add_edge_between(mem,cur_node,label_node) ;
	}

	return label_node ;

}

bool after_branch(Cfg_Node* node)
{
	Inst* last_i = node->IR_instruction_list() ;
	last_i = last_i->prev() ;

	return (last_i->is_branch()) ;
	
}

enum Prev_Br_Type {br_none, br_condi, br_non_condi} ;

void Inlined_Native::merge_flow_graph(Flow_Graph *fg, 
                                         Expressions& exprs,
                                         Recursion *recursion)
{
	Mem_Manager	lmem(10000) ;//enough?
	///////////////////////////////////////////////////////////////////////////////////////////////
	//Set the be_inlined Inst.
	Inst* be_inlined = call_i ;
	///////////////////////////////////////////////////////////////////////////////////////////////
	//Get the input Operand*. 
	int in_args = 0 , out_args = 1;//, src_args = be_inlined->n_srcs;
	
	Operand** opnd_array = call_i->get_native_args(exprs, in_args) ;
	///////////////////////////////////////////////////////////////////////////////////////////////
	//Set Exp* of the input Operand*.
/*
	Exp** src_exp_array = (Exp**)lmem.alloc(src_args * sizeof(*src_exp_array));
	NativeOpndInfo::NativeOpndNode* src_arg_list = (NativeOpndInfo::NativeOpndNode*)lmem.alloc( sizeof(NativeOpndInfo::NativeOpndNode) * src_args);
	NativeOpndInfo* src_arg_opnd_info = (NativeOpndInfo*)lmem.alloc(sizeof(NativeOpndInfo) * src_args);
	for(int i = 0 ; i<src_args ; i++){
		src_exp_array[i] = get_exp_from_opnd(exprs, be_inlined->src(i)) ;
		src_arg_opnd_info[i].set_info(NativeOpndInfo::src_arg, i+1, 0, 0, NativeOpndInfo::whole);
		NativeOpndInfo::set_opnd_node(&src_arg_list[i], &src_arg_opnd_info[i], be_inlined->src(i), src_exp_array[i]);
	}
*/
    Exp** exp_array= (Exp**)lmem.alloc(in_args * sizeof(*exp_array));
	NativeOpndInfo::NativeOpndNode* in_arg_list = (NativeOpndInfo::NativeOpndNode*)lmem.alloc( sizeof(NativeOpndInfo::NativeOpndNode) * in_args);
	NativeOpndInfo::NativeOpndNode* out_arg_list = (NativeOpndInfo::NativeOpndNode*)lmem.alloc( sizeof(NativeOpndInfo::NativeOpndNode) * out_args);
	NativeOpndInfo* in_arg_opnd_info = (NativeOpndInfo*)lmem.alloc(sizeof(NativeOpndInfo) * in_args);
	NativeOpndInfo* out_arg_opnd_info = (NativeOpndInfo*)lmem.alloc(sizeof(NativeOpndInfo) * out_args);

	int i;
	for( i = 0 ; i<in_args ; i++){
		exp_array[i] = get_exp_from_opnd(exprs, opnd_array[i]) ;
		in_arg_opnd_info[i].set_info(NativeOpndInfo::in_arg, i+1, 0, 0, NativeOpndInfo::whole);
		NativeOpndInfo::set_opnd_node(&in_arg_list[i], &in_arg_opnd_info[i], opnd_array[i], exp_array[i]);
	}

	assert(be_inlined->dst()->is_temp_reg()) ;
	Operand_Exp* out_exp = exprs.lookup_temp_reg_exp((Reg_Operand*)(be_inlined->dst())) ;
	for ( i = 0; i < out_args; i++){
		out_arg_opnd_info[i].set_info(NativeOpndInfo::out_arg, i+1, 0, 0, NativeOpndInfo::whole);
		NativeOpndInfo::set_opnd_node(&out_arg_list[i], &out_arg_opnd_info[i], be_inlined->dst(), out_exp);
	}


	Inst* result = NULL ;
	Inst *nxt = call_i->next();
	char** native_code = call_i->get_native_code() ;
	assert(native_code) ;
	NativeInstParser parser;

	NativeOpndManager nom(exprs.mem, exprs, in_args, out_args, in_arg_list, out_arg_list);

	clear_label_array() ;

	/**************************************************************************************
	 * Add b1 at the beginning
	 **************************************************************************************/
	Cfg_Node *succ_node = fg->split_cfg_node(bb);
	Cfg_Node *b1 = fg->split_cfg_node(bb);
	b1->linearization_node()->unlink();
    b1->linearization_node()->insert_after(bb->linearization_node());
    bb->add_edge(fg->mem_manager, b1);

	add_end_label(lmem, succ_node) ;

	Cfg_Node* cur_node = b1, *nxt_node = NULL ;
	Cfg_Node* target_node = NULL ;
	Prev_Br_Type prev_br = br_none ; 
	NativeInfo info;
	Inst* last_inst = NULL;
	for(i = 0 ; native_code[i][0] ; i++){
		//parse
		parser.parse_inst(native_code[i], &info);

		//cfg control
		if(info.has_label()){//Is a label
			char* label = info.get_label() ;
			Cfg_Node* next_node = lookup_node_by_label(lmem, fg, cur_node, label) ;
			//cur_node = build_native_node(lmem, fg, prev_br==br_non_condi ? NULL : cur_node, label, cur_node) ;
			if(prev_br!=br_non_condi){
				add_edge_between(fg->mem_manager,cur_node, next_node) ;
				if(target_node)
					add_edge_between(fg->mem_manager,cur_node, target_node) ;
			}
			
			add_linear_between(cur_node, next_node) ;
			cur_node = next_node ;
			prev_br = br_none ;
			target_node = NULL ;
		}
		if(prev_br != br_none){ //After a branch 
			Cfg_Node* next_node = lookup_node_by_label(lmem, fg, cur_node, NULL) ;
			//cur_node = build_native_node(lmem, fg, prev_br==br_condi ? cur_node : NULL, 
			//	info.has_label() ? info.get_label() : NULL, cur_node) ;
			if(prev_br!=br_non_condi){
				add_edge_between(fg->mem_manager,cur_node, next_node) ;
				assert(target_node) ;
				add_edge_between(fg->mem_manager,cur_node, target_node) ;
			}

			add_linear_between(cur_node, next_node) ;
			cur_node = next_node ;
			prev_br = br_none ;
			target_node = NULL ;
		}
		if(info.is_br()){ //Is a branch
			char* label = info.get_target() ;
			Cfg_Node* next_node = lookup_node_by_label(lmem, fg, cur_node, label) ;
			if(info.is_direct_br())
				add_edge_between(fg->mem_manager,cur_node, next_node) ;
			else
				target_node = next_node ;
			
			prev_br = info.is_direct_br() ? br_non_condi : br_condi ;
		}

		//gen_inst
		nxt = cur_node->IR_instruction_list() ;
		NativeInstGenerator nig(exprs, info, nom, exprs.mem, nxt, last_inst);
		result = nig.gen_inst();
		last_inst = result;
	}
	/**************************************************************************************
	 * Add link succ_ndoe at the end. Move other inst objects to the succ_node
	 **************************************************************************************/
	cur_node->add_edge(fg->mem_manager,succ_node) ;
    move_insts_after_call_to(exprs,succ_node,recursion);

	be_inlined->unlink(); // remove i from the instruction list
}

void inline_native(Cfg_Node* node, Closure* c)
{
    Inline_Closure *ic = (Inline_Closure*)c;

    Inst *head = node->IR_instruction_list();
    Inst *i;
    for (i = head->next(); i != head; i = i->next()) {
		if(i->is_native_inline() && !i->is_type_inst()){
			assert(i->get_native_code()) ;
			ic->inlined = new (ic->inlined_mem) Inlined_Native(node, i, ic->inlined, i->get_native_code()) ;
			continue ;
		}

    }
}
//#endif

static int n_inlined = 0;
//
// go through IR inst list and inline method calls
//
void inline_bb(Cfg_Node *node, Closure *c) {
    Inline_Closure *ic = (Inline_Closure*)c;
    //
    // If the node is in a loop that is not frequently executed, then we
    // don't want to do inlining.
    //
    if (node->loop_header != NULL &&
        node->loop_header->is_cold()) return;

    Inst *head = node->IR_instruction_list();
    Inst *i;
    for (i = head->next(); i != head; i = i->next()) {
        //
        // find checkcast calls so that we can inserting code to do quick check
        //
//#ifdef INLINE_NATIVE
#if 1
        if (true &&  // turn off checkcast opt by changing true -> false
            i->is_type_inst()  &&
            ((Type_Inst*)i)->kind == Type_Inst::cast &&
            i->is_native_inline()) {
			assert(i->get_native_code()) ;
			ic->inlined = new (ic->inlined_mem) Inlined_Native(node, i, ic->inlined, i->get_native_code()) ;
            continue;
        } 
#endif
#if 0
        if (true &&  // turn off checkcast opt by changing true -> false
            i->is_type_inst()  &&
            ((Type_Inst*)i)->kind == Type_Inst::cast){
            ic->inlined = new (ic->inlined_mem) 
                Inlined_Checkcast(node, (Type_Inst*)i, ic->inlined);
            continue;
        } 
#endif
        if (!i->is_call()) continue;
        if (!i->is_inlinable_call()) continue;
        Method_Handle mh = ((Call_Inst*)i)->get_mhandle();
        //
        // find tail recursion opportunities
        // check if the call and the current method have the same method_handle
        // check if the node is the return block
        //
        if (mh == node->flowgraph->m_handle() && node->flowgraph->epilog() != NULL) {
            Cfg_Node *epilog = node->flowgraph->epilog();
            Eh_Node *eh = epilog->eh_out_edge();
            Cfg_Node *b = node;
            while (b->out_edge_size() == 1 &&   // single out going edge
                   b->out_edges(0) != epilog && // reach epilog
                   b->eh_out_edge() == eh)      // same exception handler
                   b = b->out_edges(0);
            if (b->out_edge_size() == 1 && 
                b->out_edges(0) == epilog && 
                b->eh_out_edge() == eh) {
                ic->recursion = new (ic->inlined_mem) 
                Recursion(node, (Call_Inst*)i, ic->recursion);
                continue;
            }
        }
        unsigned inlined_bc_size = ic->inlined_bc_size;
        //
        // for static, final & private methods, it is always safe to inlined them
        // invokespecial: must name only an instance initialization method <init>,
        // a method in this, a private method, or a method in a superclass of this
        //
        bool safe_inline = safe_to_inline(mh, (Call_Inst*)i);
        //
        // determine if we want to inline this method
        //
//        extern Class_Handle java_lang_obj;
        if (safe_inline ||
            (((Call_Inst*)i)->kind == Call_Inst::virtual_call/* &&
            method_get_class(mh) != java_lang_obj*/)) {
            Profile_Rec *inlined_prof = NULL;
            if (instrumenting) 
                inlined_prof = o1_method_get_profile_info(mh);
            if (inline_policy(ic->fg, node, mh, inlined_bc_size,
                              ic->global_inlined_bc_size, inlined_prof,
                              (Call_Inst*)i)) {
                bool needs_deref = should_deref_before_inline(mh, (Call_Inst*)i);
#ifdef STAT_INDIRECT_CALL
				//
				// Unlink the stat_inst member of i
				//
				if(false && ((Call_Inst*)i)->stat_inst) 
					((Call_Inst*)i)->stat_inst->unlink() ;
#endif
                //
                // add an Inlined_Method into the list
                //
                ic->inlined = new (ic->inlined_mem) 
                Inlined_Method(node, (Call_Inst*)i, ic->inlined, safe_inline, 
                               needs_deref, inlined_bc_size, inlined_prof);
            }
        }
    }
}

//                         | arg0 = t1  |
//   arg0 = t1             | arg1 = t2  |              |            |  pred
//   arg1 = t2    inline   +------------+  eliminate   +------------+
//   call foo     ======>  +------------+  ========>   +------------+
//   t3 = ret.I            | t4 = arg0  |              |  t4 = t1   |  prlg
//                         | t5 = arg1  |              |  t5 = t2   |
//                foo          ...                          ...
//                         | ret.I = t6 |              |  t3 = t6   |  eplg
//                         +------------+              +------------+
//                         +------------+              +------------+
//                         | t3 = ret.I |              |            |  succ
//
void Inlined_Method::eliminate_arg_ret(Cfg_Node *pred, Cfg_Node *prlg, 
                                       Cfg_Node *eplg, Cfg_Node *call_node) {
    Call_Inst *call = (Call_Inst*)call_i;
    //
    // replace arguments
    //
    if (call->n_args() != 0) {
        Inst *args = prlg->IR_instruction_list()->next();
        unsigned ith_arg = 0;
        unsigned i;
        for (i = 0; i < call->n_args(); i++) {
            Inst *asgn= call->get_arg(i);
            Operand *src = asgn->src(0);
            // find the right argument assignment
            while ((!args->is_reg_assignment()  || 
                    !args->src(0)->is_arg() ||
                    ((Arg_Operand*)args->src(0))->num != ith_arg) &&
                   args != prlg->IR_instruction_list()) args = args->next();
            assert(args != prlg->IR_instruction_list());
            ith_arg += num_words_of_type((Java_Type)args->type());
            // replace arg with src
            args->replace_src(0,src);
            // the live range of src across blocks
            if (src->is_temp_reg()) ((Temp_Reg*)src)->set_global_reg_cand();
        }
    }
    //
    // replace return value
    //
    if (eplg != NULL && call->type() != JIT_TYPE_VOID ) {
        Inst *asgn = call->get_ret();
        assert(asgn != NULL);
        Operand *dst = asgn->dst();

        Cfg_Int n = eplg->in_edge_size();
        //
        // count how many return.  If there are more than one, the following
        // transformation violates the property that a temp reg can have only
        // one definition.  Here we annotate the temp reg so that the temp
        // reg is no longer considered as a regular temp reg.
        //
        // find all ret.I = ...
        int n_ret = 0;
        Cfg_Int i;
        for (i = 0; i < n; i++) {
            Cfg_Node *pred = eplg->in_edges(i);
            Inst *ret = pred->IR_instruction_list()->prev(); // last inst
            //
            // athrow or call IndexOutOfBound can immediately return
            //
            if (ret->is_reg_assignment() && ret->dst()->is_ret()) {
                n_ret ++;
                ret->set_dst(dst);
            }
        }
        // the live range of dst across blocks
        if (dst->is_temp_reg() && n_ret > 1) { // more than one return
            ((Temp_Reg*)dst)->set_global_reg_cand();
            ((Temp_Reg*)dst)->set_temp_reg_has_multiple_defs();
        }
    }
}
//
// move inlined call and its push arguments and return assigment to call_node
//
void Inlined::move_call_arg_ret_to(Cfg_Node *call_node) {
    Call_Inst *call = (Call_Inst*)call_i;
    if (call->n_args() != 0) {
        unsigned i;
        for (i = 0; i < call->n_args(); i++) {
            Inst *asgn= call->get_arg(i);
            Operand *src = asgn->src(0);
            // the live range of src across blocks
            if (src->is_temp_reg()) ((Temp_Reg*)src)->set_global_reg_cand();
            asgn->unlink();  // remove arg assignment
            //
            // move arg assignment to call node if we inline a virtual method
            //
            if (call_node != NULL) {
                asgn->insert_before(call_node->IR_instruction_list()); 
                assert(asgn->is_outgoing_arg_assignment());
                if (src->is_temp_reg())
                    ((Temp_Reg*)src)->set_temp_reg_has_multiple_defs();
            }
        }
    }
    call->unlink();
    if (call_node != NULL) { // virtual method
        call->insert_before(call_node->IR_instruction_list());
        Operand *target = call->src(0);
        if (target != NULL && target->is_single_def_temp_reg()){
            call->replace_src(0, ((Temp_Reg *)target)->inst()->src(0));
        }
    }
    //
    // replace return value
    //
    if (call->type() != JIT_TYPE_VOID ) {
        Inst *asgn = call->get_ret();
        assert(asgn != NULL);
        Operand *dst = asgn->dst();

        asgn->unlink(); // remove return inst
        //
        // move return inst to call_node if we inline a virtual method
        //
        if (call_node != NULL) { // virtual method
            asgn->insert_before(call_node->IR_instruction_list());
            if (dst->is_temp_reg())
                ((Temp_Reg*)dst)->set_temp_reg_has_multiple_defs();
        }
    }
#ifdef PRINTABLE_O3
    if (call_node)
        call_node->set_bytecodes(call->get_arg(0)->bc_index,
            call->bc_index - call->get_arg(0)->bc_index);
#endif
}

//
// we have made the decision to inline call_i.
// Here we build the flow graph and IR for the inlined call
//
void Inlined_Method::build_IR(Inline_Closure *ic) {
    Method_Handle mh = ((Call_Inst*)call_i)->get_mhandle();

#ifdef TRACE_O3
    static int non_overridden = 0;
    static int overridden = 0;
    cout << n_inlined << "\n----- Inline ";
    if (method_is_overridden(mh)) {
        cout << "overridden " << overridden++ << "\t";
    } else {
        cout << "non-overridden " << non_overridden++ << "\t";
    }
    cout << class_get_name(method_get_class(mh)) << "."
             << method_get_name(mh) << endl;
    n_inlined++;
#endif // TRACE_O3

    RegID_Map *reg_map = &ic->exprs.reg_map;
    //
    // save reg id map
    //
    unsigned start_reg_id, max_locals, max_stack;
    reg_map->save_context(start_reg_id, max_locals, max_stack);
    //
    // set reg id map before inlining a method
    //
    reg_map->set_context(reg_map->curr_tmp_reg_id(),
                         method_get_max_locals(mh),
                         method_get_max_stack(mh));
    //
    // build flow graph and IR for the method
    //
    JIT_Result result = 
    build_IR_of_method(ic->cmpl_handle, 
                       method_get_class(mh), 
                       mh,
                       ic->gc_requires_write_barriers,
                       ic->mem,
                       ic->exprs,
                       merged_fg,
                       bb->eh_out_edge(),
                       bb->get_enclosing_subr(),
                       bb->flowgraph,
#ifdef PRINTABLE_O3
                       call_i->bc_index,
#else // PRINTABLE_O3
                       0,
#endif // PRINTABLE_O3
                       method_get_byte_code_addr(mh),
                       method_get_byte_code_size(mh),
                       method_get_max_stack(mh),
                       method_get_max_locals(mh),
                       inlined_bc_size,
                       ic->global_inlined_bc_size);
    if (result != JIT_SUCCESS)
        merged_fg = NULL;
    else
        merged_fg->set_prof_rec(prof_rec); // record profile info
    //
    // restore reg id map
    //
    reg_map->restore_context(start_reg_id, max_locals, max_stack);

}
//
// merge the flow graph of inlined method 
// 
void Inlined_Method::merge_flow_graph(Flow_Graph *fg, 
                                      Expressions&   exprs,
                                      Recursion *recursion) {
    //
    // we cannot build IR for the callee, i.e., resolution error
    //
    if (merged_fg == NULL)
        return;

    bool need_classinit =
        (!class_is_initialized(merged_fg->c_handle()) &&
        (!method_is_static(merged_fg->m_handle()) ||
         strcmp(method_get_name(merged_fg->m_handle()), "<init>")));
    Flow_Graph *search = fg->calling_fg;
    search = NULL;
    while (need_classinit && search != NULL)
    {
        need_classinit = (merged_fg->c_handle() != search->c_handle());
        search = search->calling_fg;
    }

    fg->has_fp = (fg->has_fp || merged_fg->has_fp);
    fg->has_virtual_inline = (fg->has_virtual_inline || merged_fg->has_virtual_inline);
    Cfg_Node *split_node = bb;
    Call_Inst *call = (Call_Inst*)call_i;

    // Creates a successor CFG node with an empty IR instruction list.
    Cfg_Node *succ_node = fg->split_cfg_node(split_node);
    // Copy the linearized nodes from the old flow graph to the new.
    while (merged_fg->linear_node_ordering.next() != &merged_fg->linear_node_ordering)
    {
        Cfg_Node_List *node = merged_fg->linear_node_ordering.next();
        node->unlink();
        node->insert_before(succ_node->linearization_node());
    }
    //
    // if we inline a virtual method, we need to insert code to check if branching 
    // to the inlined method is valid, e.g.,
    //
    //                           +---------------+
    //                           |      ...      | split_node
    //                           | if [b] == xxx |
    //                           +---------------+
    //                            /             \
    //                           /               \
    //                  +------------+         +------------+
    //        call_node |  push ar0  |         |            | merged_fg
    //                  |    ...     |         |   inlined  |
    //                  |  call foo  |         |   foo      |
    //                  +------------+         +------------+
    //                           \               /
    //                            \             /
    //                           +---------------+
    //                           |      ...      | succ_node
    //                           |               |
    //                           +---------------+
    Cfg_Node *call_node = NULL;
    if (!safe_inline) { // split_node doesn't have any successors at this point
        fg->has_virtual_inline = true;
        call_node = fg->split_cfg_node(split_node);
        call_node->set_cold_non_inlined();
        call_node->add_edge(fg->mem_manager, succ_node);
        //
        // determine linearization ordering
        //
        call_node->linearization_node()->unlink();
        call_node->linearization_node()->insert_before(&fg->linear_node_ordering);
        call_node->set_cold_code();
        //
        // generate vtable address
        //
        exprs.set_live_lcse(NULL);
        void *addr = class_get_vtable(
                     method_get_class(call->get_mhandle()));
        Inst *vaddr = exprs.lookup_imm((unsigned)addr,JIT_TYPE_ADDR,call);
        //
        // get vtable
        //
        assert(call->src(0)->is_single_def_temp_reg());
        Operand *fld = ((Temp_Reg*)call->src(0))->inst()->src(0);
        assert(fld->kind == Operand::Field);
        Operand *vtab = ((Field_Operand*)fld)->base();
        assert(vtab->is_single_def_temp_reg() && ((Temp_Reg*)vtab)->inst() != NULL);
        //
        // insert "if [b] == xxx" to check vtable address
        //
        Inst *cmp_i = gen_cmp_inst(exprs.mem, exprs, call, 
                                   ((Temp_Reg*)vtab)->inst(), vaddr, JIT_TYPE_ADDR, false);
        // Annotate cmp_i to indicate that it is a vtable compare, and which
        // GCTrack_Operand is being compared.
        ((Compare_Inst *)cmp_i)->set_object_compared_with_vtable(((Temp_Reg*)vtab)->inst()->src(0));
        ((Compare_Inst *)cmp_i)->set_method_inlined(merged_fg->m_handle());
        gen_branch(exprs.mem,exprs,call,false, Exp::Bne, cmp_i);
    }
    // Graft in the new flow graph.
    // Assume both flow graphs use the same mem_manager.
    // first add_edge is fallthrough edge
    split_node->add_edge(fg->mem_manager, merged_fg->prolog()); 
    if (call_node) split_node->add_edge(fg->mem_manager, call_node);
    if (merged_fg->epilog() != NULL) 
        merged_fg->epilog()->add_edge(fg->mem_manager, succ_node);
    fg->make_traversal_number_safe(merged_fg);

    //
    // all insts are still in split_node.
    // Here we divide IR_instruction_list from call inst
    //
    move_insts_after_call_to(exprs,succ_node,recursion);

    // Remove the "return" instruction in the inlined method's epilog.
    if (merged_fg->epilog() != NULL)
        merged_fg->epilog()->IR_instruction_list()->next()->unlink();
    // Merge the inlined fg's exception handlers into this fg.
    fg->merge_eh_info(merged_fg);
    //
    // move call, arg and return instructions to call_node
    //
    move_call_arg_ret_to(call_node);
    //
    // eliminate argument and ret
    // 
    eliminate_arg_ret(split_node, merged_fg->prolog(), merged_fg->epilog(), 
                      call_node);
    //
    // Preserve null pointer exception semantics when the vtable
    // isn't already checked.
    //
    if (safe_inline && requires_obj_deref)
    {
        // Do a little pattern matching on the bytecodes.
        // If the first bytecodes are "aload_0; getfield", then don't do the
        // object dereference.
        const unsigned char *bc = merged_fg->bytecodes();
        if (merged_fg->bc_length() >= 4 &&
            bc[0] == 0x2a &&
            bc[1] == 0xb4)
        {
        }
        else
        {
            Call_Inst *call = (Call_Inst *) call_i;
            Inst *arg0 = call->get_arg(0);
            dereference_object(arg0->src(0), arg0->exp, exprs,
                merged_fg->prolog()->out_edges(0)->IR_instruction_list()->next());
        }
    }
    //
    // Make sure the class is initialized before entering the inlined path.
    //
    if (need_classinit)
    {
        Inst *first = merged_fg->prolog()->IR_instruction_list()->next();
        Inst *ci_inst = exprs.lookup_imm((unsigned)merged_fg->c_handle(),JIT_TYPE_ADDR, first);
        Exp *expr = exprs.lookup_inst_exp(Exp::ClassInit, ci_inst->exp, NULL, JIT_TYPE_VOID);
        new(fg->mem_manager) Classinit_Inst(ci_inst->dst(), expr, first);
    }
}

#define GEN_CMP_NULL
//
// inserting quick check code for checkcast
// The checkcast call takes two arguments, object and class_handle.  If the 
// vtable of the object and the vtable associated with the class_handle are 
// the same, the checkcast call will succeed without throwing ClassCastException.
// The routine generates the following code:
//
//                             +---------------+
//                             |      bb       |
//                             +---------------+
//                                     |
//                             +---------------+
//                             |  cmp  obj, 0  | B1
//                             |  beq          |
//                             +---------------+
//                               /             \
//                   +--------------------+    |
//                B2 |  cmp [obj], vtable |    |
//                   |  bne               |    |
//                   +--------------------+    |
//                       /   ________\ ________/
//                      /   /         \
//         +--------------+/    +-----------------+
//     B3  |  t43 = obj   |     |  push c_handle  |
//         +--------------+     |  push obj       | call_node
//                   \          |  call checkcast |
//                    \         |  t43 = eax      |
//                     \        +-----------------+
//                      \            /
//                      +--------------+
//                      |   succ_node  |
//               
void Inlined_Checkcast::merge_flow_graph(Flow_Graph *fg, 
                                         Expressions& exprs,
                                         Recursion *recursion) {
    Operand *obj = call_i->src(1);
    if (!obj->is_single_def_temp_reg() && 
        !obj->is_vreg()) return;

    //fg->has_virtual_inline = true;
    // Creates a successor CFG node with an empty IR instruction list.
    Cfg_Node *succ_node = fg->split_cfg_node(bb);
#ifdef GEN_CMP_NULL
    Cfg_Node *b1 = fg->split_cfg_node(bb);
    Cfg_Node *b2 = fg->split_cfg_node(bb);
    Cfg_Node *b3 = fg->split_cfg_node(bb);
#else
    Cfg_Node *b2 = fg->split_cfg_node(bb);
    if (b2->eh_out_edge() != NULL) // inherit the exception handler of bb
        b2->delete_eh_edge(b2->eh_out_edge());
    Cfg_Node *b3 = fg->split_cfg_node(b2); // no exception handler
#endif
    Cfg_Node *call_node = fg->split_cfg_node(bb);
    //
    // fix linearization ordering
    // Assume that the common path is bb->B1->B2->B3->succ_node
    // call_node is considered as cold code and moved out of line
    // x = split_cfg_node(y) already puts x after y so we don't need to
    // do anything for b1, b2, b3 and succ_node
    //
    call_node->linearization_node()->unlink();
    b3->linearization_node()->unlink();
    b2->linearization_node()->unlink();
#ifdef GEN_CMP_NULL
    b1->linearization_node()->unlink();
    b1->linearization_node()->insert_before(succ_node->linearization_node());
#endif
    b2->linearization_node()->insert_before(succ_node->linearization_node());
    b3->linearization_node()->insert_before(succ_node->linearization_node());
    call_node->linearization_node()->insert_before(&fg->linear_node_ordering);
    call_node->set_cold_code();
    //
    // fix control flow edges
    //
#ifdef GEN_CMP_NULL
    bb->add_edge(fg->mem_manager, b1);
    b1->add_edge(fg->mem_manager, b2); // 1st edge is fall-through edge
    b1->add_edge(fg->mem_manager, b3); // 2nd edge is branch edge
#else
    bb->add_edge(fg->mem_manager, b2);
#endif
    b2->add_edge(fg->mem_manager, b3); // 1st edge is fall-through edge
    b2->add_edge(fg->mem_manager, call_node); // 2nd edge is branch edge
    b3->add_edge(fg->mem_manager, succ_node); 
    call_node->add_edge(fg->mem_manager, succ_node); 
    //
    // generate       cmp obj, 0    for B1
    //                beq
    Inst *obj_i, *cmp;
    exprs.set_live_lcse(NULL);
#ifdef GEN_CMP_NULL
    Inst *inst_head = b1->IR_instruction_list();
#else
    Inst *inst_head = b2->IR_instruction_list();
#endif
    if (obj->is_single_def_temp_reg()) {
        ((Temp_Reg*)obj)->set_global_reg_cand(); // obj live across basic blocks;
        obj_i = ((Temp_Reg*)obj)->inst();
    } else {
        assert(obj->is_vreg());
        //
        // lookup virtual reg
        //
        Operand_Exp *vexp = exprs.lookup_reg_exp(((Reg_Operand*)obj)->id,obj->type,1); 
	    obj_i = exprs.gen_opnd_tuple(inst_head,vexp);
    }
#ifdef GEN_CMP_NULL
    Inst *imm0 = exprs.lookup_imm(0,JIT_TYPE_CLASS,inst_head);
    cmp  = gen_cmp_inst(exprs.mem, exprs, inst_head, 
                              obj_i, imm0, JIT_TYPE_CLASS, false);
    gen_branch(exprs.mem, exprs, inst_head, false, Exp::Beq, cmp);
#endif

	//::
	//Get class_handle , first
	//::
    Operand *cl = call_i->src(0);
    assert(cl->is_single_def_temp_reg() || cl->kind == Operand::Immediate);
    unsigned c;
    if (cl->is_single_def_temp_reg())
        c = ((Imm_Operand*)((Temp_Reg*)cl)->inst()->src(0))->imm();
    else
        c = ((Imm_Operand*)cl)->imm();
    inst_head = b2->IR_instruction_list();
	
	BOOL do_fast_checkcast  = FALSE ;
	int offset = 0 ;
//#ifdef FAST_INSTOF
	int depth = class_get_depth((Class_Handle)c);
	offset = vtable_get_super_array_offset() + (depth - 1) * sizeof(void *);
	do_fast_checkcast = ((!(class_get_flags((Class_Handle)c) & ACC_INTERFACE)) &&
		class_get_name((Class_Handle)c)[0] != '[' && 
		(depth < MAX_FAST_INSTOF_DEPTH));
//#endif

	if (do_fast_checkcast){
		//::
		// Fast checkcast , using "superclasses" list in VTable.
		// generate		txx = [obj]					for B2
		//				cmp [txx + offset], c_handle
		//				bne
		//::
		//
		// txx = [obj]
		//
		Inst *obj_vtable = exprs.lookup_field(obj_i,0,JIT_TYPE_ADDR,inst_head,(FIELD_UID)c);
		//
		//	cmp [txx+offset], c_handle 
		//	bne
		//
		Inst *clss_i = exprs.lookup_imm((unsigned)c,JIT_TYPE_ADDR,inst_head);
		Inst *obj_clss = exprs.lookup_field(obj_vtable,offset,JIT_TYPE_ADDR,inst_head,(FIELD_UID)c);
		cmp = gen_cmp_inst(exprs.mem, exprs, inst_head, 
						   obj_clss, clss_i, JIT_TYPE_ADDR, false);
		gen_branch(exprs.mem, exprs, inst_head, false, Exp::Bne, cmp);
	}else{
		//
		// generate      cmp [obj], vtable    for B2
		//               bne 
		//
		void *vtab = class_get_vtable((Class_Handle)c);
		Inst *vtab_i = exprs.lookup_imm((unsigned)vtab,JIT_TYPE_ADDR,inst_head);
		Inst *obj_vtab = exprs.lookup_field(obj_i,0,JIT_TYPE_ADDR,inst_head,(FIELD_UID)c);
		cmp = gen_cmp_inst(exprs.mem, exprs, inst_head, 
						   obj_vtab, vtab_i, JIT_TYPE_ADDR, false);
		gen_branch(exprs.mem, exprs, inst_head, false, Exp::Bne, cmp);
	}

    //
    // generate      t43 = obj           for B3
    //
    inst_head = b3->IR_instruction_list();
    Operand *dst = call_i->dst();
    assert(dst->is_reg() && (!dst->is_vreg() && !dst->is_physical_reg()));
    ((Temp_Reg*)dst)->set_temp_reg_has_multiple_defs();
    new (exprs.mem) Assign_Inst(dst,obj,obj_i->exp,inst_head);
    //
    // all insts are still in bb.
    // Here we divide IR_instruction_list from call inst
    //
    move_insts_after_call_to(exprs,succ_node,recursion);
    //
    // move checkcast call
    //
    call_i->unlink();
    call_i->insert_before(call_node->IR_instruction_list());
    ((Type_Inst*)call_i)->set_obj_not_null();
#ifdef PRINTABLE_O3
    call_node->set_bytecodes(call_i->bc_index,
                             succ_node->first_bc_idx() - call_i->bc_index);
#endif
#ifndef GEN_CMP_NULL
    //
    // generate exception handler to catch null pointer exception
    // within b2
    //
    Eh_Node *eh_node = fg->create_eh_node();
    Mem_Manager& mm = fg->mem_manager;
    // add eh edge <node,eh_node>
    b2->add_eh_edge(mm, eh_node);
    // get NullPointerException handler
    Class_Handle ch = cached_class_handles[ch_java_lang_NullPointerException];
    // the exception handler is b3
    eh_node->add_edge(mm,b3,ch,0); // catch throwable
    // insert eh_node into fg->handlers();
    eh_node->insert_before(fg->handlers());
#endif
}

static void set_global_if_live_across(Operand *opnd, Bit_Vector *bv)
{
    if (opnd == NULL)
        return;
    if (opnd->kind == Operand::Field || opnd->kind == Operand::Array)
    {
        set_global_if_live_across(opnd->base(), bv);
        set_global_if_live_across(opnd->index(), bv);
        return;
    }
    if (!opnd->is_temp_reg() || ((Reg_Operand *)opnd)->global_reg_alloc_cand())
        return;
    unsigned bvp = opnd->bv_position();
    if (bv->is_set(bvp))
        ((Temp_Reg *)opnd)->set_global_reg_cand();
}

//
// instructions are still in bb's IR_instruction_list
// Move all insts after call to node
//
void Inlined::move_insts_after_call_to(Expressions& exprs, 
                                       Cfg_Node *node, 
                                       Recursion* list) {
    unsigned numvars = exprs.reg_map.curr_tmp_reg_id();
    unsigned sz = numvars/8+1; // bv size;
    Mem_Manager mm(sz);
    Bit_Vector *bv = new(mm) Bit_Vector(numvars, mm, false);
    Inst *i = call_i->next();
    Inst *head = bb->IR_instruction_list();
    Inst *inst;
    for (inst = head->next(); inst != call_i; inst = inst->next()) {
        Operand *dst = inst->dst();
        if (dst == NULL || !dst->is_reg()) continue;
        bv->set(dst->bv_position());
    }
    while (i != head) {
        Inst *inext = i->next();
        //
        // those not yet processed inlined method may belong to newly created 
        // cfg_node.  We update their bb pointer to point to node
        //
        if (i->is_inlinable_call()) {
            Inlined *m;
            for (m = next; m != NULL; m = m->next) {
                if (m->call_i == i) m->bb = node;
            }
            Recursion *r;
            for (r = list; r != NULL; r = r->next) {
                if (r->call_i == i) r->bb = node;
            }
        }
        i->unlink(); // remove from split_node->IR_instruction_list
        i->insert_before(node->IR_instruction_list()); // append to node
        //
        // check if a temp reg is live across bb (set_global_reg_cand())
        //
        Operand *dst = i->dst();
        if (dst != NULL && !dst->is_reg())
            set_global_if_live_across(dst,bv);
        unsigned s;
        for (s=0; s<i->n_srcs; s++)
            set_global_if_live_across(i->src(s), bv);

        i = inext;
    }
#ifdef PRINTABLE_O3
    // update first_bc_index and bc_length for dot file and dump_jit
    unsigned first_bc_idx = call_i->bc_index;
    unsigned bc_len = first_bc_idx - bb->first_bc_idx();
    node->set_bytecodes(first_bc_idx,bb->bc_length()-bc_len);
    bb->set_bytecodes(bb->first_bc_idx(),bc_len);
#endif
}

void dereference_object(Operand *opnd, Exp *exp, Expressions &exprs, Inst *inst_head)
{
    O3_Jit_Type ty = opnd->type;
    Operand *treg = opnd;
    if (!treg->is_reg())
    {
        // Find a new temp reg.
        unsigned tregno = exprs.reg_map.get_tmp_reg_id(0);
        Operand *treg = exprs.lookup_reg_exp(tregno, ty, 1)->opnd;
        // Assign the object to the new temp reg.
        new(exprs.mem) Assign_Inst(treg, opnd, exp, inst_head);
    }
    new(exprs.mem) Deref_Inst(treg, exp, inst_head);
}

void inline_methods(Compile_Handle cmpl_handle,
                    Mem_Manager&   perm_mem,
                    Expressions&   exprs, 
                    Flow_Graph     *fg,
                    unsigned       accumulated_inlined_bc_size,
                    unsigned&      global_inlined_bc_size,
                    bool           gc_requires_write_barriers) {
    if (accumulated_inlined_bc_size >= MAX_INLINE_SIZE) return;
    //
    // Mem Manager for building inlined method list
    //
    Mem_Manager tmp_mm(sizeof(Inlined_Method)*10);
    Inlined *inlined = NULL;
    Recursion *recursion = NULL;
    Inline_Closure c(fg, cmpl_handle,perm_mem,exprs,tmp_mm,inlined,recursion,
                     accumulated_inlined_bc_size, global_inlined_bc_size,
                     gc_requires_write_barriers);
    fg->apply(inline_bb,&c,true); // only traverse bb within fg

//#ifdef INLINE_NATIVE
	fg->apply(inline_native, &c, true) ;
//#endif

    Inlined *im;
    for (im = inlined; im != NULL; im = im->next) {
        if (im->bb->is_cold()) continue;
        im->build_IR(&c);
        im->merge_flow_graph(fg,exprs,recursion);
    }
    //
    // Do tail recursion transformation
    //
    Tail_Recursion tr(cmpl_handle, fg, perm_mem, exprs, gc_requires_write_barriers);
    tr.transform(recursion);
}
