// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/cse.cpp,v 1.2 2001/08/13 09:59:51 xhshi Exp $
//

#include "defines.h"
#include <iostream.h>
#include "code_emitter.h"
#include "stack.h"
#include "operand.h"
#include "cse.h"
#include "lazy_code_selector.h"
//
// combine the sub expressions that have been popped (in _popped_srcs) with 
// curr_exp (i.e., for the current bytecode).  We combine those expressions 
// only when they are contiguous.  If dst is a scratch register, then we
// also record the resulting expression that ends up in the register.
//
void CSE::combine(const unsigned char *first_bc,
				  CSE_Exp *curr_exp,
				  Operand *dst,
				  unsigned local_regs,  // bit vector of local regs
				  unsigned is_long) {
	//
	// reset dst reg in case that we fail to combine expressions
	//
	X86_Reg_No dst_no = n_reg;
	// dst is a local reg
	if (dst->is_reg() && dst->hold_local_reg(local_regs)) {
		dst_no = ((Reg_Operand*)dst)->opnd.reg_no();
		_reg_cse[dst_no].reset();
	}
	_top++;
	assert(_top <= _size);
	if (_next_src > 3)
		_cse[_top-1]->reset();
	else {
		CSE_Exp *e;
		for (int i = 0; i < _next_src; i++) {
			e = &_popped_srcs[i];
			if (!e->is_combinable(curr_exp)) {
				_cse[_top-1]->reset();
				return;
			} else
				// combine two expressions
				curr_exp->set_exp(e->bc_index(),e->len() + curr_exp->len());
		}
		_cse[_top-1]->set_exp(curr_exp);
		//
		// record that dst contain curr_exp
		// For now, we don't deal with long.
		//
		if (!is_long && dst_no != n_reg) {
			_reg_cse[dst_no].set_exp(curr_exp);
			compute_cse_kill_set(first_bc,dst_no);
		}
	}
}

//
// Since we do lazy code selection, some memory or immediate accesses are folded
// into compute instructions. We emit instructions when folding is immpossible.
// We need to update cse info after emitting.
//
void CSE::update_cse(const unsigned char *first_bc,
					 Operand *opnd, unsigned local_regs, int ith_popped) {
	assert(ith_popped < MAX_POPPED_SRCS);
	// 
	// if opnd is not a local reg or not a CSE cand, then return;
	//
	if (!opnd->is_reg() || !opnd->hold_local_reg(local_regs) ||
		!_popped_srcs[ith_popped].is_CSE_cand()) return;

	X86_Reg_No reg_no = ((Reg_Operand*)opnd)->opnd.reg_no();
	_reg_cse[reg_no].set_exp(&_popped_srcs[ith_popped]);
	compute_cse_kill_set(first_bc,reg_no);
}
// 
// We don't keep kill_set for each CSE_Exp.  Instead, we compute kill_set for
// _reg_cse (only 3 entries).  When updating _reg_cse entries, we then
// compute the kill_set.
//
void CSE::compute_cse_kill_set(const unsigned char *first_bc, X86_Reg_No no) {
	assert(no < _max_regs);
	//
	// if not valid expression, then return;
	//
	unsigned len = _reg_cse[no].len();
	if (!_reg_cse[no].is_CSE_cand() || len > MAX_SZ_CSE) return;
	const unsigned char *bc = first_bc + _reg_cse[no].bc_index();
	const unsigned char *last_bc = bc + len;
	int index;
	while (bc < last_bc) {
		index = -1;
		switch (*bc) {
		case 0x15:	// iload
		case 0x16:	// lload
		case 0x19:	// aload
			index = bc[1];
			break;
		case 0x1a: case 0x1b: case 0x1c: case 0x1d:	// iload_{0,1,2,3}
			index = bc[0] - 0x1a;
			break;
		case 0x2a: case 0x2b: case 0x2c: case 0x2d:	// aload_{0,1,2,3}
			index = bc[0] - 0x2a;
			break;
		case 0x1e: case 0x1f: case 0x20: case 0x21:	// lload_{0,1,2,3}
			index = bc[0] - 0x1e;
			break;
		case 0xb4:	// getfield
			index = (bc[1] << 8) + bc[2];
			break;
		case 0x2e:	// iaload
		case 0x2f:	// laload
		case 0x30:	// faload
		case 0x31:	// daload
		case 0x32:	// aaload
		case 0x33:	// baload
  		case 0x34:	// caload
 		case 0x35:	// saload
			_reg_array_ty[no] |= (1 << (*bc - 0x2e));
			break;
		}
		if (index >=0 && index < MAX_SZ_CSE_KILL) {
			// set bit --- grab lower 5 bits
			_reg_kill_set[no][index >> 5] |= (1 << (index & 0x001f));
		} else if (index >= MAX_SZ_CSE_KILL) {
			_reg_cse[no].reset();
			return;
		}
		assert(exp_len[*bc] != -1);
		bc += exp_len[*bc] + 1;
	}
}

void CSE::kill_reg_cse(int curr_bc_index,int var_no) {
	for (int i = 0; i < _max_regs; i++)
		// test if index is in the kill set 
		if (_reg_cse[i].is_CSE_cand() &&
			(_reg_kill_set[i][var_no >> 5] & (1 << (var_no & 0x001f)))) {
			_reg_cse[i].reset();
		}
}

void CSE::kill_reg_array_cse(int curr_bc_index,int array_ty) {
	for (int i = 0; i < _max_regs; i++)
		// test if index is in the kill set 
		if (_reg_cse[i].is_CSE_cand() &&
			(_reg_array_ty[i] & (1 << array_ty))) {
			_reg_cse[i].reset();
		}
}

X86_Reg_No CSE::find_cse(const unsigned char *first_bc,
					     const unsigned char *last_bc,
					     const unsigned char *curr_bc) {
	//
	// Since we want to match the biggest cse, we decide the ordering in which
	// cse expressions are matched.
	//
	char checked[n_reg];
	((int*)checked)[0] = ((int*)checked)[1] = 0; // init checked[0..7] = 0
	for (int i = 0; i < _max_regs; i++) {
		int cse_cand = -1;
		int len = 0;
		for (int j = 0; j < _max_regs; j++) {
			if (checked[j]) continue;
			else if (_reg_cse[j].is_CSE_cand()) {
				if (_reg_cse[j].len() > len) {
					len = _reg_cse[j].len();
					cse_cand = j;
				}
			} else
				checked[j] = 1;
		}
		if (cse_cand != -1) {
			checked[cse_cand] = 1;
			//
			// byte comparison (matching cse)
			//
			if (curr_bc + len < last_bc) {
				const unsigned char *bc = first_bc + _reg_cse[cse_cand].bc_index();
				int j;
				for(j = 0; j < len; j++)
					if (curr_bc[j] != bc[j]) break;

				if (j == len) {
					return (X86_Reg_No) cse_cand;
				}
			}
		} else
			return n_reg;
	}
	return n_reg;
}

void CSE::map_offset(Map_Entry *map, int emitter_off,
					 const unsigned char *first_bc,
					 const unsigned char *cse_bc, int cse_len) {
	const unsigned char *bc = cse_bc;
	const unsigned char *last_bc = bc + cse_len;
	while (bc < last_bc) {
		map[bc - first_bc].offset = emitter_off;
		assert(exp_len[*bc] != -1);
		bc += exp_len[*bc] + 1;
	}
}
