/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.sql.framework.evaluators.database.base;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.velocity.VelocityContext;
import org.netbeans.modules.sql.framework.evaluators.database.AbstractEvaluator;
import org.netbeans.modules.sql.framework.evaluators.database.StatementContext;
import org.netbeans.modules.sql.framework.evaluators.database.TemplateBuilder;
import org.netbeans.modules.sql.framework.model.SQLCaseOperator;
import org.netbeans.modules.sql.framework.model.SQLCondition;
import org.netbeans.modules.sql.framework.model.SQLConstants;
import org.netbeans.modules.sql.framework.model.SQLObject;
import org.netbeans.modules.sql.framework.model.SQLWhen;

import com.sun.sql.framework.exception.BaseException;

/**
 * Velocity-based evaluator that generates SQL Case expressions.
 * 
 * @author Jonathan Giron
 * @version $Revision: 1.2 $
 */
public class BaseCaseEvaluator extends AbstractEvaluator {

    public String evaluate(SQLObject obj, StatementContext context) throws BaseException {
        if (obj == null || getObjectType(obj) != SQLConstants.CASE) {
            throw new BaseException("Bad SQLObject type - case expected.");
        }

        SQLCaseOperator operator = (SQLCaseOperator) obj;
        VelocityContext vContext = new VelocityContext();

        vContext.put("switch", buildSwitchClause(operator, context));
        vContext.put("whenClauses", buildWhenClauses(operator, context));
        vContext.put("default", buildDefaultClause(operator, context));

        String nestedIndent = (String) context.getClientProperty("nestedIndent");
        vContext.put("nestedIndent", ((nestedIndent != null) ? nestedIndent : "    "));
        return TemplateBuilder.generateSql(this.getDB().getTemplateFileName("caseWhen"), vContext);
    }

    /**
     * Builds appropriate switch clause, if applicable, from the given SQLCaseOperator
     * instance. DB-specific extensions should override this method to reflect
     * vendor-specific syntax quirks.
     * 
     * @param operator SQLCaseOperator to be interrogated
     * @param context StatementContext to use in evaluating switch clause
     * @return String representing switch clause, possibly the empty string (&quot;&quot;)
     * @throws BaseException if error occurs during evaluation
     */
    protected String buildSwitchClause(SQLCaseOperator operator, StatementContext context) throws BaseException {
        String switchClause = "";

        SQLObject condition = operator.getSQLObject(SQLCaseOperator.SWITCH);
        boolean containsPredicate = (condition != null && getObjectType(condition) == SQLConstants.PREDICATE);
        if (containsPredicate) {
            switchClause = this.getDB().getEvaluatorFactory().evaluate(condition, context);
        }

        return switchClause;
    }

    /**
     * Builds List of mapped when-then clauses to be inserted into the Velocity template
     * context. DB-specific extensions should override this method to reflect
     * vendor-specific syntax quirks.
     * 
     * @param operator SQLCaseOperator to be interrogated
     * @param context StatementContext to use in evaluating individual when clauses
     * @return List of Maps containing "when" and "then" clauses
     * @throws BaseException if error occurs during evaluation
     */
    protected List buildWhenClauses(SQLCaseOperator operator, StatementContext context) throws BaseException {
        SQLObject condition = operator.getSQLObject(SQLCaseOperator.SWITCH);
        boolean containsPredicate = (condition != null && getObjectType(condition) == SQLConstants.PREDICATE);

        List whenObjects = operator.getWhenList();
        List whenClauses = new ArrayList(whenObjects.size());

        for (int i = 0; i < operator.getWhenCount(); i++) {
            SQLWhen when = (SQLWhen) whenObjects.get(i);
            SQLCondition wcondition = when.getCondition();
            if (wcondition == null || containsPredicate) {
                throw new BaseException("Badly formed case object.");
            }

            Map whenThen = new HashMap(1);
            whenThen.put("when", this.getDB().getEvaluatorFactory().evaluate(wcondition.getRootPredicate(), context));
            whenThen.put("then", this.getDB().getEvaluatorFactory().evaluate(when.getSQLObject(SQLWhen.RETURN), context));

            whenClauses.add(whenThen);
        }

        return whenClauses;
    }

    /**
     * Builds appropriate default clause, if applicable, from the given SQLCaseOperator
     * instance. DB-specific extensions should override this method to reflect
     * vendor-specific syntax quirks.
     * 
     * @param operator SQLCaseOperator to be interrogated to obtain appropriate default
     *        clause
     * @param context StatementContext to use in evaluating default clause
     */
    protected String buildDefaultClause(SQLCaseOperator operator, StatementContext context) throws BaseException {
        String defaultClause = "";

        SQLObject defaultAction = operator.getSQLObject(SQLCaseOperator.DEFAULT);
        if (defaultAction != null) {
            defaultClause = this.getDB().getEvaluatorFactory().evaluate(defaultAction, context);
        }

        return defaultClause;
    }

    /**
     * Resolve Object type if required.
     * 
     * @param object SQLObject
     * @return int value of SQLObject type
     */
    protected final int getObjectType(SQLObject object) {
        int type;
        switch (object.getObjectType()) {
            case SQLConstants.VISIBLE_PREDICATE:
                type = SQLConstants.PREDICATE;
                break;
            case SQLConstants.VISIBLE_LITERAL:
                type = SQLConstants.LITERAL;
                break;
            default:
                type = object.getObjectType();
        }

        return (type);
    }
}

