/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.mashup.db.ui.wizard;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.net.URL;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.NoSuchElementException;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import org.netbeans.modules.mashup.db.common.FlatfileDBConnectionFactory;
import org.netbeans.modules.mashup.db.common.PropertyKeys;
import org.netbeans.modules.mashup.db.model.FlatfileDBTable;
import org.netbeans.modules.mashup.db.model.impl.FlatfileDBTableImpl;
import org.netbeans.modules.mashup.db.ui.FlatfileResulSetPanel;
import org.netbeans.modules.sql.framework.ui.view.ResultSetTablePanel;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.util.NbBundle;

import com.sun.sql.framework.utils.Logger;
import com.sun.sql.framework.utils.StringUtil;

/**
 * @author Ahimanikya Satapathy
 * @version $Revision: 1.2 $
 */
public class PreviewDataPanel extends JPanel implements ActionListener {
    
    private static final String CMD_SHOWDATA = "ShowData"; // NOI18N
    private static final String LOG_CATEGORY = PreviewDataPanel.class.getName();
    
    private FlatfileDBTable currentTable;
    private JButton previewBtn;
    private JTextField recordCount;
    private ResultSetTablePanel recordViewer;
    private FlatfileColumnTableModel tableModel;
    private JLabel parseErrorMessage;
    private JLabel totalRowsLabel;
    
    public PreviewDataPanel(FlatfileDBTable table) {
        currentTable = table;
        
        String previewLabel = NbBundle.getMessage(TableDefinitionPanel.class, "LBL_import_preview_table");
        setBorder(BorderFactory.createTitledBorder(previewLabel));
        setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
        add(createPreviewControls());
        setPreferredSize(new Dimension(205, 120));
        recordViewer = new ResultSetTablePanel();
        add(recordViewer);
    }
    
    /**
     * Invoked when an action occurs.
     *
     * @param e ActionEvent to handle
     */
    public void actionPerformed(ActionEvent e) {
        Object src = e.getSource();
        
        if (src == recordCount) {
            previewBtn.requestFocusInWindow();
            previewBtn.doClick();
        } else if (src == previewBtn) {
            tryResulSet();
        }
    }
    
    public boolean tryResulSet() {
        int ct = 25;
        boolean isValid = true;
        
        try {
            ct = Integer.parseInt(recordCount.getText());
        } catch (NumberFormatException nfe) {
            recordCount.setText(String.valueOf(25));
        }
        
        // Call helper to obtain result set and display.
        if (tableModel != null && !tableModel.getRowEntries().isEmpty()) {
            tableModel.updateColumns(currentTable);
        }
        
        FlatfileDBTable table = (FlatfileDBTable) currentTable.clone();
        
        table.setProperty(PropertyKeys.WIZARDFIELDCOUNT, new Integer(table.getColumnList().size()));
        String fname = table.getProperty(PropertyKeys.FILENAME);
        if(fname == null || fname.equals("")) {
            if(table.getLocalFilePath() != null && !table.getLocalFilePath().equals("") &&
                    table.getFileName() != null && !table.getFileName().equals("")) {
                ((FlatfileDBTableImpl)table).setOrPutProperty(PropertyKeys.FILENAME, 
                        (new File(table.getLocalFilePath(), table.getFileName())).getAbsolutePath());
            }
        }
        String fileName = StringUtil.escapeControlChars(table.getProperty("URL"));
        ((FlatfileDBTableImpl)table).setOrPutProperty(PropertyKeys.FILENAME, fileName);
        if(table.getParserType().equalsIgnoreCase(PropertyKeys.WEB) || 
                table.getParserType().equalsIgnoreCase(PropertyKeys.RSS)) {
            Map properties = ((FlatfileDBTableImpl)table).getProperties();
            properties.remove(PropertyKeys.FILENAME);            
            ((FlatfileDBTableImpl)table).setProperties(properties);
        }
        
        Connection conn = null;
        File previewDir = new File(System.getProperty("java.io.tmpdir"), ".preview");
        File metadataDir = new File(previewDir, ".metadata");
        
        File lockFile = new File(metadataDir, "lockfile.txt");
        FlatfileDBConnectionFactory factory = FlatfileDBConnectionFactory.getInstance();
        Object oldLockFlag = factory.getIgnoreLockProperty();
        try {
            factory.setIgnoreLockProperty("true");
            
            if (lockFile.exists()) {
                lockFile.delete();
            }
            
            if (!metadataDir.exists()) {
                metadataDir.mkdirs();
            }
            
            String url = FlatfileDBConnectionFactory.DEFAULT_FLATFILE_JDBC_URL_PREFIX + "preview:" + metadataDir;
            Logger.print(Logger.DEBUG, LOG_CATEGORY, "Preview URL: " + url);
            
            conn = FlatfileDBConnectionFactory.getInstance().getConnection(url);
            Statement stmt = conn.createStatement();
            
            stmt.execute("DROP TABLE IF EXISTS " + table.getTableName());
            String create = table.getCreateStatementSQL();
            
            Logger.print(Logger.DEBUG, LOG_CATEGORY, this, "Generated create statement: " + create);
            stmt.execute(create);
            
            ResultSet rs = stmt.executeQuery(table.getSelectStatementSQL(ct));
            recordViewer.clearView();
            recordViewer.setResultSet(rs);
            
            // get the count of all rows
            String countSql = "Select count(*) From " + table.getName();
            Logger.print(Logger.DEBUG, FlatfileResulSetPanel.class.getName(), "Select count(*) statement used for total rows: \n" + countSql);
            
            stmt = conn.createStatement();
            ResultSet cntRs = stmt.executeQuery(countSql);
            
            // set the count
            if (cntRs == null) {
                totalRowsLabel.setText("");
            } else {
                if (cntRs.next()) {
                    int count = cntRs.getInt(1);
                    totalRowsLabel.setText(String.valueOf(count));
                }
            }
            
            stmt.execute("DROP TABLE " + table.getTableName());
        } catch (NoSuchElementException nse) {
            String errorMsg = "ERROR: Current sample file may be corrupt, or does not match specified datatypes.";
            try {
                errorMsg = NbBundle.getMessage(TableDefinitionPanel.class, "ERROR_bad_preview");
            } catch (MissingResourceException mre) {
                Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Could not locate resource string for ERROR_bad_preview.", mre);
            }
            showError("Sample file may be corrupt, or does not match specified datatypes", errorMsg, nse);
            isValid = false;
        } catch (SQLException se) {
            String errorMsg = "ERROR: Current sample file may be corrupt, or does not match specified datatypes.";
            String sqlExMsg = "Sample file may be corrupt, or does not match specified datatypes";
            try {
                sqlExMsg = stripExceptionHeaderFromMessage(se);
                errorMsg = NbBundle.getMessage(TableDefinitionPanel.class, "ERROR_bad_preview_sqlex", sqlExMsg);
            } catch (MissingResourceException mre) {
                Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Could not locate resource string for ERROR_bad_preview.", mre);
            }
            showError(sqlExMsg, errorMsg, se);
            isValid = false;
        } catch (Exception t) {
            String errMsg = "Unknown error occurred while obtaining ResultSet. ";
            showError("Unknown error: " + t.getMessage(), errMsg, unwrapThrowable(t));
            isValid = false;
        } finally {
            table.setProperty(PropertyKeys.FILENAME, table.getFileName());
            if (conn != null) {
                try {
                    conn.createStatement().execute("shutdown");
                    conn.close();
                } catch (SQLException ignore) {
                    // ignore
                }
            }
            
            if (lockFile != null && lockFile.exists()) {
                lockFile.delete();
            }
            
            if (metadataDir != null && metadataDir.exists()) {
                metadataDir.deleteOnExit();
            }
            
            if (previewDir != null && previewDir.exists()) {
                previewDir.deleteOnExit();
            }
            
            FlatfileDBConnectionFactory.getInstance().setIgnoreLockProperty(oldLockFlag);
        }
        
        return isValid;
    }
    
    private void showError(String shortErrMsg, String errorMsg, Throwable t) {
        if (parseErrorMessage != null) {
            parseErrorMessage.setText(shortErrMsg);
            parseErrorMessage.revalidate();
            parseErrorMessage.repaint();
        } else {
            DialogDisplayer.getDefault().notify(new NotifyDescriptor.Message(errorMsg, NotifyDescriptor.WARNING_MESSAGE));
        }
        setEnabled(false);
        Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, errorMsg, t);
    }
    
    /**
     * Overrides parent implementation to allow for addition of this instance as a
     * listener for various child components.
     */
    public void addNotify() {
        super.addNotify();
        if (previewBtn != null) {
            previewBtn.removeActionListener(this);
            previewBtn.addActionListener(this);
        }
    }
    
    public void clearData() {
        if (recordViewer != null) {
            recordViewer.clearView();
        }
    }
    
    /**
     * Overrides parent implementation to allow for removal of this instance as a listener
     * for various child components.
     */
    public void removeNotify() {
        if (previewBtn != null) {
            previewBtn.removeActionListener(this);
        }
        super.removeNotify();
    }
    
    public void setEnabled(boolean enabled) {
        if (previewBtn != null) {
            previewBtn.setEnabled(enabled);
        }
        
        if (recordCount != null) {
            recordCount.setEnabled(enabled);
        }
    }
    
    public void setTable(FlatfileDBTable table) {
        currentTable = table;
    }
    
    public void setTableModel(FlatfileColumnTableModel model) {
        tableModel = model;
    }
    
    public boolean showData(JLabel parseErrMsg) {
        if(currentTable != null && !currentTable.getColumnList().isEmpty()) {
            parseErrorMessage = parseErrMsg;
            return tryResulSet();
        }
        return false;
    }
    
    /*
     * Creates preview button and row count text field to control display of parsed output
     * colMetaTable.
     */
    private JPanel createPreviewControls() {
        JPanel controlPanel = new JPanel(new FlowLayout(FlowLayout.LEADING));
        
        // add refresh button
        URL url = getClass().getResource("/org/netbeans/modules/sql/framework/ui/resources/images/refresh16.png");
        previewBtn = new JButton(new ImageIcon(url));
        previewBtn.setMnemonic('S');
        previewBtn.setToolTipText("Show data for this table definition");
        previewBtn.setActionCommand(CMD_SHOWDATA);
        previewBtn.addActionListener(this);
        
        JPanel recordCountPanel = new JPanel();
        recordCountPanel.setBorder(BorderFactory.createEmptyBorder(0, 10, 0, 0));
        
        JLabel lbl = new JLabel("Limit rows:");
        lbl.setDisplayedMnemonic('l');
        recordCountPanel.add(lbl);
        recordCount = new JTextField("25", 5);
        recordCountPanel.add(recordCount);
        lbl.setLabelFor(recordCount);
        recordCount.addActionListener(this);
        
        // add total row count label
        JPanel totalRowsPanel = new JPanel();
        FlowLayout fl = new FlowLayout();
        fl.setAlignment(FlowLayout.LEFT);
        totalRowsPanel.setLayout(fl);
        
        JLabel totalRowsNameLabel = new JLabel("Total rows:");
        totalRowsNameLabel.setBorder(BorderFactory.createEmptyBorder(0, 10, 0, 0));
        totalRowsPanel.add(totalRowsNameLabel);
        
        totalRowsLabel = new JLabel();
        totalRowsPanel.add(totalRowsLabel);
        
        controlPanel.add(previewBtn);
        controlPanel.add(recordCountPanel);
        controlPanel.add(totalRowsPanel);
        
        return controlPanel;
    }
    
    private String stripExceptionHeaderFromMessage(Exception e) {
        String cookedMsg = null;
        String rawSqlMsg = e.getMessage();
        if (!StringUtil.isNullString(rawSqlMsg)) {
            int beginIndex = rawSqlMsg.lastIndexOf(":");
            int endIndex = rawSqlMsg.lastIndexOf(")") - 1;
            if (rawSqlMsg.length() == beginIndex) {
                beginIndex = -1;
            }
            
            if (endIndex < beginIndex || endIndex == -1) {
                endIndex = rawSqlMsg.length();
            }
            cookedMsg = rawSqlMsg.substring(beginIndex + 1, endIndex);
        }
        return cookedMsg;
    }
    
    private Throwable unwrapThrowable(Throwable t) {
        // Drill down to the root cause, if available.
        while (t.getCause() != null) {
            t = t.getCause();
            // Prevent infinite loops.
            if (t.getCause() == t) {
                break;
            }
        }
        return t;
    }
}

