/* 
 *   Creation Date: <1999/12/22 22:47:48 samuel>
 *   Time-stamp: <2001/04/03 02:05:22 samuel>
 *   
 *	<mouse.c>
 *	
 *	Mouse interface (keyboard to be added?)
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include "os_interface.h"
#include "driver_mgr.h"
#include "mac_registers.h"
#include "mouse_sh.h"
#include "debugger.h"
#include "timer.h"
#include "res_manager.h"

static int osip_get_mouse( int sel, int *params );
static int osip_has_hw_cursor( int sel, int *params );
static int osip_get_mouse_dpi( int sel, int *params );

static int mouse_init( void );
static void mouse_cleanup( void );

driver_interface_t osi_mouse_driver = {
    "osi_mouse", mouse_init, mouse_cleanup
};

typedef struct 
{
	int dx, dy;
	int absx, absy;
	
	int status;	/* see mouse_sh.h for bit defs */

	int cur_buts;	/* current button press/release status */

	ullong moved_mstamp;

	/* use "hardware" cursor (X-cursor for instance) */
	int use_hw_curs;
} mouse_t;

static mouse_t m;


/************************************************************************/
/*	Init / Cleanup							*/
/************************************************************************/

static int 
mouse_init( void )
{
	memset( &m, 0, sizeof(mouse_t) );
	
	os_interface_add_proc( OSI_GET_MOUSE, osip_get_mouse );
	os_interface_add_proc( OSI_HAS_HW_CURSOR, osip_has_hw_cursor );
	os_interface_add_proc( OSI_GET_MOUSE_DPI, osip_get_mouse_dpi );

	m.moved_mstamp = get_mticks_();
	return 1;
}

static void 
mouse_cleanup( void )
{
	os_interface_remove_proc( OSI_GET_MOUSE );
	os_interface_remove_proc( OSI_HAS_HW_CURSOR );
	os_interface_remove_proc( OSI_GET_MOUSE_DPI );
}


/************************************************************************/
/*	Linux Interface							*/
/************************************************************************/

void
mouse_move_to( int x, int y )
{
	m.absx = x;
	m.absy = y;
	m.dx = m.dy = 0;
	m.status |= kMouseEvent_MoveTo;

	m.moved_mstamp = get_mticks_();
}

void
mouse_move( int dx, int dy )
{
	m.dx += dx;
	m.dy += dy;

	m.moved_mstamp = get_mticks_(); 
}

ullong 
get_mouse_moved_stamp( void )
{
	return m.moved_mstamp;
}


void
mouse_but_event( int butevent )
{
	m.cur_buts |= (butevent & 7);
	m.cur_buts &= ~((butevent >> 3) & 7 );

	m.moved_mstamp = get_mticks_();
};

/* but bits: ..00000321. 1=pressed */ 
void
mouse_event( int dx, int dy, int buts )
{
	m.dx += dx;
	m.dy += dy;

	m.cur_buts = buts;
	m.moved_mstamp = get_mticks_();
}

void
use_hw_cursor( int flag )
{
	m.use_hw_curs = flag;
}


/************************************************************************/
/*	OSI Interface							*/
/************************************************************************/

static int
osip_get_mouse( int sel, int *ret_params )
{
	int rets;
	osi_mouse_t *ret_pb = (void*)&mregs->gpr[4];	/* return in r4-r7 */
	
	if( m.dx || m.dy )
		m.status |= kMouseEvent_MoveDelta;

	rets = (m.status & ~7) | (m.cur_buts & 7);

	ret_pb->dx = m.dx;
	ret_pb->dy = m.dy;
	ret_pb->absx = m.absx;
	ret_pb->absy = m.absy;

	m.dx = m.dy = 0;
	m.status &= ~(kMouseEvent_MoveDelta | kMouseEvent_MoveTo | kMouseEvent_ButMask );

	ret_pb->use_hw_curs = m.use_hw_curs;

	return rets; /* to r3 */
}

static int
osip_has_hw_cursor( int sel, int *ret_params )
{
	/* This function is called frequently, thus it might be wise
	 * to include this flag with the mouse x-y stuff to minimize
	 * the number of osip calls.
	 */
/*	printm("cursor question (%d)\n", m.use_hw_curs); */
	return m.use_hw_curs;
}

static int
osip_get_mouse_dpi( int sel, int *ret_params )
{
	static int mouse_dpi=-2;
	
	if( mouse_dpi != -2 )
		mouse_dpi = get_numeric_res("mouse_dpi");
	if( mouse_dpi <= 0 )
		mouse_dpi = 160;
	return mouse_dpi;
}
