/* GraphBrowserWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace.compat;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.worksheet.GraphListComponentType;
import org.grinvin.worksheet.WorksheetWindow;
import org.grinvin.worksheet.WorksheetWindowModel;
import org.grinvin.worksheet.WorksheetWindowModel.SubwindowType;
import org.grinvin.io.list.GraphListLoader;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.list.InvariantListLoader;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionLoader;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.invariants.InvariantList;
import org.grinvin.workspace.DefaultWindowWrapper;

import org.jdom.Element;

/**
 * Wrapper for the GraphBrowserWindow
 */
public class GraphBrowserWindowWrapper extends DefaultWindowWrapper<WorksheetWindow> {
    
    /**
     * Load the window, its graph list and its invariaht list.
     */
    @Override
    public WorksheetWindow load(Element element, SectionLoader sloader) throws IOException {
        try {
            
            WorksheetWindowModel worksheetWindowModel = new WorksheetWindowModel();

            GraphList graphList = worksheetWindowModel.getGraphList();
            InvariantList invariantList = worksheetWindowModel.getInvariantList();
            
            // Load the corresponding graph list
            URI graphlistURI = new URI(element.getAttributeValue("graphlistURI"));
            InputStream input = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            GraphListLoader.load_1_0(graphList, input, sloader);
            graphList.setURI(graphlistURI);
            
            // get the graph list name
            InputStream graphlistInput = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            Element graphlist = LoaderSaverHelper.loadElementNonValidating(graphlistInput, "graphlistelements");
            worksheetWindowModel.setName(graphlist.getAttributeValue("graphlistName"));
            
            // load invariant list
            InputStream invariantlist = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            InvariantListLoader.load_1_0(invariantList, invariantlist);
            
            for (SubwindowType type : SubwindowType.values())
                worksheetWindowModel.getSubwindowModel(type).setVisible(false);
            worksheetWindowModel.getSubwindowModel(SubwindowType.GRAPHLIST).setVisible(true);
            worksheetWindowModel.getSubwindowModel(SubwindowType.INVARIANTLIST).setVisible(true);
            
            String component = element.getAttributeValue("component");
            if (component.equals("graphInvariantTable")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.TABLE);
            } else if (component.equals("graphCellList")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.CELL);
            } else if (component.equals("graphList")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.LIST);
            } else {
                throw new IOFormatException("Invalid component type: " + component);
            }
            
            // create and initialize the window
            WorksheetWindow window = new WorksheetWindow(worksheetWindowModel);
            
            // initialize properties of this window and of its guides tool
            loadWindowProperties(element, window);

            return window;
            
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in meta.xml", ex);
        }
    }
    
    public WorksheetWindow create() {
        throw new UnsupportedOperationException("Unsupported method called");
    }
    
    public String getElementName() {
        throw new UnsupportedOperationException("Unsupported method called");
    }
    
}
