/* RenameGraphListElementAction.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.worksheet.actions;

import java.awt.Component;
import java.awt.event.ActionEvent;

import org.grinvin.invariants.Invariant;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.list.graphs.GraphListElementListener;
import org.grinvin.list.actions.GraphRenamePanel;

/**
 * Action which can be used to rename a certain graph list element.  The graph
 * list element must be set before the action can be performed.
 */
public class RenameGraphListElementAction extends AbstractGraphListElementAction implements GraphListElementListener {
    
    //
    private Component parent;
    
    /**
     * Set the parent for the dialog window.
     */
    public void setParent(Component parent) {
        this.parent = parent;
    }
    
    /**
     * Create an action of this type for the given graph list element.
     */
    public RenameGraphListElementAction(GraphListElement[] graphListElements) {
        super(graphListElements, "renameGraphListElement.description");
        this.parent = null;
    }
    
    /**
     * Create an action of this type for the given graph list element.
     */
    public RenameGraphListElementAction(GraphListElement graphListElement) {
        super(graphListElement, "renameGraphListElement.description");
        this.parent = null;
    }
    
    /**
     * Create an action of this type. The graph list elements to which
     * this action applies should be set later using {@link #setGraphListElements}
     */
    public RenameGraphListElementAction() {
        this(new GraphListElement[0]);
    }
    
    //
    protected void setEnabled() {
        setEnabled(graphListElements != null && graphListElements.length == 1 && graphListElements[0].isNameEditable());
    }
    
    //
    public void actionPerformed(ActionEvent ev) {
        if (graphListElements.length == 1 && graphListElements[0].isNameEditable()) {
            new GraphRenamePanel().showDialog(graphListElements[0], parent);
        }
    }
    
    public void setGraphListElements(GraphListElement[] graphListElements) {
        //unregister
        for(GraphListElement element : this.graphListElements) {
            element.removeGraphListElementListener(this);
        }
        //set the new list
        super.setGraphListElements(graphListElements);
        //register
        for(GraphListElement element : this.graphListElements)
            element.addGraphListElementListener(this);
    }
    
    public void graphListElementURIChanged(GraphListElement element) {
        setEnabled();
    }
    
    public void graphListElementNameChanged(GraphListElement element) {
        //ignore
    }
    
    public void graphListElementInvariantComputed(GraphListElement element, Invariant invariant) {
        //ignore
    }
    
    public void graphListElementGraphChanged(GraphListElement element) {
        //ignore
    }

    
}
