package com.clarkware.junitperf;

import junit.framework.Test;
import junit.framework.TestResult;
import junit.extensions.RepeatedTest;

/**
 * The <code>LoadTest</code> is a test decorator that runs
 * a test with a simulated number of concurrent users and
 * iterations.
 * <p>
 * In its simplest form, a <code>LoadTest</code> is constructed 
 * with a test to decorate and the number of concurrent users.
 * <p>
 * For example, to create a load test of 10 concurrent users
 * with each user running <code>ExampleTest</code> once and 
 * all users started simultaneously, use:
 * <blockquote>
 * <pre>
 * Test loadTest = new LoadTest(new ExampleTest(), 10);
 * </pre>
 * </blockquote>
 * <p>
 * The load can be ramped by specifying a pluggable 
 * <code>Timer</code> instance which prescribes the delay
 * between the addition of each concurrent user.  A
 * <code>ConstantTimer</code> has a constant delay, with 
 * a zero value indicating that all users will be started 
 * simultaneously. A <code>RandomTimer</code> has a random 
 * delay with a uniformly distributed variation.
 * <p>
 * For example, to create a load test of 10 concurrent users
 * with each user running <code>ExampleTest</code> once and 
 * with a one second delay between the addition of users, use:
 * <blockquote>
 * <pre>
 * Timer timer = new ConstantTimer(1000);
 * Test loadTest = new LoadTest(new ExampleTest(), 10, timer);
 * </pre>
 * </blockquote>
 * <p>
 * In order to simulate each concurrent user running a test for a 
 * specified number of iterations, a <code>LoadTest</code> can be 
 * constructed to decorate a <code>RepeatedTest</code>. 
 * Alternatively, a <code>LoadTest</code> convenience constructor 
 * specifying the number of iterations is provided which creates a 
 * <code>RepeatedTest</code>. 
 * <p>
 * For example, to create a load test of 10 concurrent users
 * with each user running <code>ExampleTest</code> for 20 iterations, 
 * and with a one second delay between the addition of users, use:
 * <blockquote>
 * <pre>
 * Timer timer = new ConstantTimer(1000);
 * Test repeatedTest = new RepeatedTest(new ExampleTest(), 20);
 * Test loadTest = new LoadTest(repeatedTest, 10, timer);
 * </pre>
 * </blockquote>
 * or, alternatively, use: 
 * <blockquote>
 * <pre>
 * Timer timer = new ConstantTimer(1000);
 * Test loadTest = new LoadTest(new ExampleTest(), 10, 20, timer);
 * </pre>
 * </blockquote> 
 * A <code>LoadTest</code> can be decorated as a <code>TimedTest</code>
 * to test the elapsed time of the load test.  For example, to decorate 
 * the load test constructed above as a timed test with a maximum elapsed 
 * time of 2 seconds, use:
 * <blockquote>
 * <pre>
 * Test timedTest = new TimedTest(loadTest, 2000);
 * </pre>
 * </blockquote>
 * <p>
 * By default, a <code>LoadTest</code> does not enforce test 
 * atomicity (as defined in transaction processing) if its decorated 
 * test spawns threads, either directly or indirectly.  In other words, 
 * if a decorated test spawns a thread and then returns control without 
 * waiting for its spawned thread to complete, then the test is assumed 
 * to be transactionally complete.  
 * <p>
 * If threads are integral to the successful completion of 
 * a decorated test, meaning that the decorated test should not be 
 * treated as complete until all of its threads complete, then  
 * <code>setEnforceTestAtomicity(true)</code> should be invoked to 
 * enforce test atomicity.  This effectively causes the load test to 
 * wait for the completion of all threads belonging to the same 
 * <code>ThreadGroup</code> as the thread running the decorated test.
 *
 * @author <a href="mailto:mike@clarkware.com">Mike Clark</a>
 * @author <a href="http://www.clarkware.com">Clarkware Consulting, Inc.</a>
 * @author Ervin Varga
 *
 * @see junit.framework.Test
 */

public class LoadTest implements Test {

	private final int _users;
	private final Timer _timer;
	private final ThreadedTest _test;
	private final ThreadedTestGroup _group;
	private final ThreadBarrier _barrier;
	private boolean _enforceTestAtomicity;

	/**
	 * Constructs a <code>LoadTest</code> to decorate 
	 * the specified test using the specified number 
	 * of concurrent users and a delay of 0 ms.
	 *
	 * @param test Test to decorate.
	 * @param users Number of concurrent users.
	 */
	public LoadTest(Test test, int users) {
		this(test, users, new ConstantTimer(0));
	}
	
	/**
	 * Constructs a <code>LoadTest</code> to decorate 
	 * the specified test using the specified number 
	 * of concurrent users, number of iterations per
	 * user, and a delay of 0 ms.
	 *
	 * @param test Test to decorate.
	 * @param users Number of concurrent users.
	 * @param iteration Number of iterations per user.
	 */
	public LoadTest(Test test, int users, int iterations) {
		this(test, users, iterations, new ConstantTimer(0));
	}
		
	/**
	 * Constructs a <code>LoadTest</code> to decorate 
	 * the specified test using the specified number 
	 * of concurrent users, number of iterations per
	 * user, and delay timer.
	 *
	 * @param test Test to decorate.
	 * @param users Number of concurrent users.
	 * @param iteration Number of iterations per user.
	 * @timer Delay timer.
	 */
	public LoadTest(Test test, int users, int iterations, Timer timer) {
		this(new RepeatedTest(test, iterations), users, timer);
	}
	
	/**
	 * Constructs a <code>LoadTest</code> to decorate 
	 * the specified test using the specified number 
	 * of concurrent users and delay timer.
	 *
	 * @param test Test to decorate.
	 * @param users Number of concurrent users.
	 * @timer Delay timer.
	 * @throws IllegalArgumentException If an invalid argument is
	 *         specified, such as a non-positive number of concurrent users.
	 */
	public LoadTest(Test test, int users, Timer timer) {
        
		 if (users < 1) {
            throw new IllegalArgumentException("Number of users must be > 0");
        } else if (timer == null) {
            throw new IllegalArgumentException("Delay timer is null");
        } else if (test == null) {
            throw new IllegalArgumentException("Decorated test is null");
        }
		 
		_users = users;
		_timer = timer;
		setEnforceTestAtomicity(false);
		_barrier = new ThreadBarrier(users);
		_group = new ThreadedTestGroup(this);
		_test = new ThreadedTest(test, _group, _barrier);
	}
	
	/**
	 * Indicates whether test atomicity should be enforced.
	 *
	 * @param isAtomic <code>true</code> to enforce test atomicity;
	 *                 <code>false</code> otherwise.
	 */
	public void setEnforceTestAtomicity(boolean isAtomic) {
		_enforceTestAtomicity = isAtomic;
	}
	
	/**
	 * Returns the number of tests in this test.
	 *
	 * @return Number of tests.
	 */
	public int countTestCases() {
		return _test.countTestCases() * _users;
	}

	/**
	 * Runs the test.
	 *
	 * @param result Test result.
	 */
	public void run(TestResult result) {
	
		_group.setTestResult(result);

		for (int i=0; i < _users; i++) {

			if (result.shouldStop()) {
				_barrier.cancelThreads(_users - i);
				break;
			}

			_test.run(result);

			try {

				Thread.sleep(getDelay());

			} catch(InterruptedException ignored) { }
		}
		
		waitForTestCompletion();

		cleanup();
	}
	
	/**
	 * Waits for test completion.
	 */
	protected void waitForTestCompletion() {
		//
		// TODO: May require a strategy pattern
		//       if other algorithms emerge.
		// 
		if (_enforceTestAtomicity) {
			waitForAllThreadsToComplete();
		} else {
			waitForThreadedTestThreadsToComplete();
		}
	}

	/**
	 * Waits for all threads spawned by 
	 * <code>ThreadedTest</code> instances
	 * to complete.
	 */
	protected void waitForThreadedTestThreadsToComplete() {
		
		while (!_barrier.isReached()) {
			sleep(50);
		}
	}
	
	/**
	 * Waits for all threads in the 
	 * <code>ThreadedTestGroup</code> to complete.
	 */
	protected void waitForAllThreadsToComplete() {
		
		while (_group.activeCount() > 0) {
			sleep(50);
		}
	}
	
	/**
	 * Sleeps.
	 */
	protected void sleep(long time)  {
		try { 
			
			Thread.sleep(time); 
			
		} catch(Exception ignored) { } 
	}
	
	/**
	 * Cleans up thread resources.
	 */
	protected void cleanup() {
		try {
		
			_group.destroy();
		
		} catch (Throwable ignored) { }
	}
	
	/**
	 * Returns the test description.
	 *
	 * @return Description.
	 */
	public String toString() {
		if (_enforceTestAtomicity) {
			return "LoadTest (ATOMIC): " + _test.toString();
		} else {
			return "LoadTest (NON-ATOMIC): " + _test.toString();
		}
	}

	/**
	 * Returns the test delay.
	 *
	 * @return Delay in milliseconds.
	 */
	protected long getDelay() {
		return _timer.getDelay();
	}
}
