//
// Copyright 1998, 1999 CDS Networks, Inc., Medford Oregon
//
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. All advertising materials mentioning features or use of this software
//    must display the following acknowledgement:
//      This product includes software developed by CDS Networks, Inc.
// 4. The name of CDS Networks, Inc.  may not be used to endorse or promote
//    products derived from this software without specific prior
//    written permission.
//
// THIS SOFTWARE IS PROVIDED BY CDS NETWORKS, INC. ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL CDS NETWORKS, INC. BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//


package com.internetcds.jdbc.tds;

import java.sql.*;
import java.math.BigDecimal;
import java.util.Vector;
// import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Calendar;
import java.io.*;

/**
 * <P>A ResultSet provides access to a table of data generated by
 * executing a Statement. The table rows are retrieved in
 * sequence. Within a row its column values can be accessed in any
 * order.
 *
 * <P>A ResultSet maintains a cursor pointing to its current row of
 * data.  Initially the cursor is positioned before the first row.
 * The 'next' method moves the cursor to the next row.
 *
 * <P>The getXXX methods retrieve column values for the current
 * row.  You can retrieve values either using the index number of the
 * column, or by using the name of the column.  In general using the
 * column index will be more efficient.  Columns are numbered from 1.
 *
 * <P>For maximum portability, ResultSet columns within each row should be
 * read in left-to-right order and each column should be read only once.
 *
 * <P>For the getXXX methods, the JDBC driver attempts to convert the
 * underlying data to the specified Java type and returns a suitable
 * Java value.  See the JDBC specification for allowable mappings
 * from SQL types to Java types with the ResultSet.getXXX methods.
 *
 * <P>Column names used as input to getXXX methods are case
 * insensitive.  When performing a getXXX using a column name, if
 * several columns have the same name, then the value of the first
 * matching column will be returned. The column name option is
 * designed to be used when column names are used in the SQL
 * query. For columns that are NOT explicitly named in the query, it
 * is best to use column numbers. If column names were used there is
 * no way for the programmer to guarantee that they actually refer to
 * the intended columns.
 *
 * <P>A ResultSet is automatically closed by the Statement that
 * generated it when that Statement is closed, re-executed, or is used
 * to retrieve the next result from a sequence of multiple results.
 *
 * <P>The number, types and properties of a ResultSet's columns are
 * provided by the ResulSetMetaData object returned by the getMetaData
 * method.
 *
 * @author Craig Spannring
 * @author The FreeTDS project
 * @version  $Id: ResultSet_base.java,v 1.5 2001/01/12 07:35:11 cts Exp $
 *
 * @see Statement#executeQuery
 * @see Statement#getResultSet
 * @see ResultSetMetaData
 @ @see Tds#getRow
 */

public class ResultSet_base 
{
   public static final String cvsVersion = "$Id: ResultSet_base.java,v 1.5 2001/01/12 07:35:11 cts Exp $";


   Tds               tds            = null;
   Statement         stmt           = null;
   Columns           columnsInfo    = null;
   ResultSetMetaData metaData       = null;
   PacketRowResult   currentRow     = null;
   boolean           lastGetWasNull = false;

   boolean           hitEndOfData   = false;
   boolean           isClosed       = false;

   private SQLWarningChain warningChain = null;   // The warnings chain.
   
   public ResultSet_base(Tds tds_, Statement stmt_, Columns columns_)
   {
      tds         = tds_;
      stmt        = stmt_;
      columnsInfo = columns_;

      hitEndOfData = false;
      warningChain = new SQLWarningChain();
   }


   protected void NotImplemented() throws java.sql.SQLException
      {
         throw new SQLException("Not implemented");
      }


   /**
    * After this call getWarnings returns null until a new warning is
    * reported for this ResultSet.
    *
    * @exception SQLException if a database-access error occurs.
    */
   public void clearWarnings() throws SQLException
   {
      warningChain.clearWarnings();
   }

   /**
    * In some cases, it is desirable to immediately release a
    * ResultSet's database and JDBC resources instead of waiting for
    * this to happen when it is automatically closed; the close
    * method provides this immediate release.
    *
    * <P><B>Note:</B> A ResultSet is automatically closed by the
    * Statement that generated it when that Statement is closed,
    * re-executed, or is used to retrieve the next result from a
    * sequence of multiple results. A ResultSet is also automatically
    * closed when it is garbage collected.
    *
    * @exception SQLException if a database-access error occurs.
    */
   public void close() throws SQLException
   {
      Exception   exception = null;
      
      if (isClosed)
      {
         // nop ???
      }
      else
      {
         isClosed = true;
         try
         {
            if (!hitEndOfData)
            {
               tds.discardResultSet(columnsInfo);
               hitEndOfData = true;
            }
            else
            {
               // nop
            }
         }
         catch(com.internetcds.jdbc.tds.TdsException e)
         {
             e.printStackTrace();
            exception = e;
      }
         catch(java.io.IOException e)
         {
             e.printStackTrace();
            exception = e;
         }
         
         currentRow  = null;
         metaData    = null;
         columnsInfo = null;
         stmt        = null;
         
         if (exception != null)
         {
            throw new SQLException(exception.getMessage());
         }
      }
   }


   //----------------------------------------------------------------

   /**
    * Map a Resultset column name to a ResultSet column index.
    *
    * @param columnName the name of the column
    * @return the column index
    * @exception SQLException if a database-access error occurs.
    */
   public int findColumn(String columnName) throws SQLException
   {
      int   i;

      for(i=1; i<=columnsInfo.getColumnCount(); i++)
      {
         if (columnsInfo.getName(i).equalsIgnoreCase(columnName))
         {
            return i;
         }
         // XXX also need to look at the fully qualified name ie. table.column
      }
      throw new SQLException("No such column " + columnName);
   }


   /**
    * A column value can be retrieved as a stream of ASCII characters
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
    * do any necessary conversion from the database format into ASCII.
    *
    * <P><B>Note:</B> All the data in the returned stream must be
    * read prior to getting the value of any other column. The next
    * call to a get method implicitly closes the stream. . Also, a
    * stream may return 0 for available() whether there is data
    * available or not.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return a Java input stream that delivers the database column value
    * as a stream of one byte ASCII characters.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getAsciiStream(int columnIndex) throws SQLException
   {
       String val = getString(columnIndex);
       if (val == null)
           return null;
       try {
           return new ByteArrayInputStream(val.getBytes("ASCII"));
       } catch (UnsupportedEncodingException ue) {
           // plain impossible with encoding ASCII
           return null;
       }
   }


   /**
    * A column value can be retrieved as a stream of ASCII characters
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
    * do any necessary conversion from the database format into ASCII.
    *
    * <P><B>Note:</B> All the data in the returned stream must
    * be read prior to getting the value of any other column. The
    * next call to a get method implicitly closes the stream.
    *
    * @param columnName is the SQL name of the column
    * @return a Java input stream that delivers the database column value
    * as a stream of one byte ASCII characters.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getAsciiStream(String columnName) throws SQLException
   {
      return getAsciiStream(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a
    * java.lang.BigDecimal object.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @param scale the number of digits to the right of the decimal
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public BigDecimal getBigDecimal(int columnIndex, int scale)
      throws SQLException
   {
      Object      tmp    = getObject(columnIndex);
      BigDecimal  result = null;


      if (tmp == null)
      {
         result = null;
      }
      else if (tmp instanceof java.lang.Double)
      {
         result = new BigDecimal(((Double)tmp).doubleValue());
         result = result.setScale(scale, BigDecimal.ROUND_HALF_UP);
      }
      else if (tmp instanceof java.lang.Float)
      {
         result = new BigDecimal(((Float)tmp).doubleValue());
         result = result.setScale(scale, BigDecimal.ROUND_HALF_UP);
      }
      else if (tmp instanceof java.lang.Number)
      {
         // This handles Byte, Short, Integer, and Long
         result = BigDecimal.valueOf(((Number)tmp).longValue(), scale);
      }
      else if (tmp instanceof BigDecimal)
      {
         result = (BigDecimal)tmp;
      }
      else if (tmp instanceof java.lang.String)
      {
         try
         {
            result = new BigDecimal((String)tmp);
         }
         catch (NumberFormatException e)
         {
            throw new SQLException(e.getMessage());
         }
      }
      return result;
   }

   /**
    * Get the value of a column in the current row as a
    * java.lang.BigDecimal object.
    *
    * @param columnName is the SQL name of the column
    * @param scale the number of digits to the right of the decimal
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public BigDecimal getBigDecimal(String columnName, int scale) throws SQLException
   {
      return getBigDecimal(findColumn(columnName), scale);
   }


   /**
    * A column value can be retrieved as a stream of uninterpreted bytes
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARBINARY values.
    *
    * <P><B>Note:</B> All the data in the returned stream must be
    * read prior to getting the value of any other column. The next
    * call to a get method implicitly closes the stream. Also, a
    * stream may return 0 for available() whether there is data
    * available or not.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return a Java input stream that delivers the database column value
    * as a stream of uninterpreted bytes.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getBinaryStream(int columnIndex)
      throws SQLException
   {
       byte[] bytes = getBytes(columnIndex);
       if (bytes != null)
           return new ByteArrayInputStream(bytes);
       return null;
   }


   /**
    * A column value can be retrieved as a stream of uninterpreted bytes
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARBINARY values.
    *
    * <P><B>Note:</B> All the data in the returned stream must
    * be read prior to getting the value of any other column. The
    * next call to a get method implicitly closes the stream.
    *
    * @param columnName is the SQL name of the column
    * @return a Java input stream that delivers the database column value
    * as a stream of uninterpreted bytes.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getBinaryStream(String columnName)
      throws SQLException
   {
      return getBinaryStream(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java boolean.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is false
    * @exception SQLException if a database-access error occurs.
    */
   public boolean getBoolean(int columnIndex) throws SQLException
   {
      Object  obj = getObject(columnIndex);
      boolean result;

      if (obj == null)
      {
         result = false;
      }
      else
      {
         switch(getMetaData().getColumnType(columnIndex))
         {
            case java.sql.Types.TINYINT:
            case java.sql.Types.SMALLINT:
            case java.sql.Types.INTEGER:
            case java.sql.Types.BIGINT:
            case java.sql.Types.REAL:
            case java.sql.Types.FLOAT:
            case java.sql.Types.DOUBLE:
            case java.sql.Types.DECIMAL:
            case java.sql.Types.NUMERIC:
            {
               if (! (obj instanceof java.lang.Number))
               {
                  // Must be out of sync with the implementation of
                  // Tds.getRow() for this to happen.
                  throw new SQLException("Internal error");
               }
               // Would somebody like to tell what a true/false has
               // to do with a double?
               result = ((java.lang.Number)obj).intValue()!=0;
               break;
            }
            case java.sql.Types.BIT:
            {
               if (! (obj instanceof Boolean))
               {
                  // Must be out of sync with the implementation of
                  // Tds.getRow() for this to happen.
                  throw new SQLException("Internal error");
               }
               result = ((Boolean)obj).booleanValue();
               break;
            }
            case java.sql.Types.CHAR:
            case java.sql.Types.VARCHAR:
            case java.sql.Types.LONGVARCHAR:
            {
               // Okay, I'm really confused as to what you mean
               // by a character string being true or false.  What
               // is the boolean value for "Let the wookie win"?
               // But since the spec says I have to convert from
               // character to boolean data...

               if (! (obj instanceof String))
               {
                  // Must be out of sync with the implementation of
                  // Tds.getRow() for this to happen.
                  throw new SQLException("Internal error");
               }
               char  ch = (((String)obj) + "n").charAt(0);

               result = (ch=='Y')||(ch=='y')||(ch=='t')||(ch=='T');
               break;
            }
            default:
            {
               throw new SQLException("Can't convert column " + columnIndex
                                      + " from "
                                      + obj.getClass().getName()
                                      + " to boolean");
            }
         }
      }
      return result;
   } // getBoolean()


   /**
    * Get the value of a column in the current row as a Java boolean.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is false
    * @exception SQLException if a database-access error occurs.
    */
   public boolean getBoolean(String columnName) throws SQLException
   {
      return getBoolean(findColumn(columnName));
   } // getBoolean()


   /**
    * Get the value of a column in the current row as a Java byte.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public byte getByte(int columnIndex) throws SQLException
   {
      return (byte) getLong(columnIndex);
   }


   /**
    * Get the value of a column in the current row as a Java byte.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public byte getByte(String columnName) throws SQLException
   {
      return getByte(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java byte array.
    * The bytes represent the raw values returned by the driver.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public byte[] getBytes(int columnIndex) throws SQLException
   {
      byte   result[];

      try
      {
         Object tmp = currentRow.getElementAt(columnIndex);
         lastGetWasNull = false;
         if (tmp == null)
         {
            lastGetWasNull = true;
            result = null;
         }
         else if (tmp instanceof byte[])
         {
            result = (byte[])tmp;
         }
         else if (tmp instanceof String)
         {
            result = tds.getEncoder().getBytes((String)tmp);
         }
         else
         {
            throw new SQLException("Can't convert column " + columnIndex
                                   + " from "
                                   + tmp.getClass().getName()
                                   + " to byte[]");
         }
      }
      catch (TdsException e)
      {
         e.printStackTrace();
         throw new SQLException(e.getMessage());
      }
      return result;
   }


   /**
    * Get the value of a column in the current row as a Java byte array.
    * The bytes represent the raw values returned by the driver.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public byte[] getBytes(String columnName) throws SQLException
   {
      return getBytes(findColumn(columnName));
   }


   /**
    * Get the name of the SQL cursor used by this ResultSet.
    *
    * <P>In SQL, a result table is retrieved through a cursor that is
    * named. The current row of a result can be updated or deleted
    * using a positioned update/delete statement that references the
    * cursor name.
    *
    * <P>JDBC supports this SQL feature by providing the name of the
    * SQL cursor used by a ResultSet. The current row of a ResultSet
    * is also the current row of this SQL cursor.
    *
    * <P><B>Note:</B> If positioned update is not supported a
    * SQLException is thrown
    *
    * @return the ResultSet's SQL cursor name
    * @exception SQLException if a database-access error occurs.
    */
   public String getCursorName() throws SQLException
   {
      throw new SQLException("Not implemented (getCursorName)");
   }


   /**
    * Get the value of a column in the current row as a java.sql.Date object.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Date getDate(int columnIndex) throws SQLException
   {
      java.sql.Date        result = null;
      java.sql.Timestamp   tmp    = getTimestamp(columnIndex);

      if (tmp != null)
      {
         result = new java.sql.Date(tmp.getTime());
      }
      return result;
   }


   /**
    * Get the value of a column in the current row as a java.sql.Date object.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Date getDate(String columnName) throws SQLException
   {
      return getDate(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java double.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public double getDouble(int columnIndex) throws SQLException
   {
      double  result;
      Object  obj    = getObject(columnIndex);

      if (obj == null)
      {
         result = 0.0;
      }
      else
      {
         try
         {
            switch(getMetaData().getColumnType(columnIndex))
            {
               case java.sql.Types.TINYINT:
               case java.sql.Types.SMALLINT:
               case java.sql.Types.INTEGER:
               {
                  result = ((Number)obj).doubleValue();
                  break;
               }
               case java.sql.Types.BIGINT:
               {
                  result = ((Number)obj).doubleValue();
                  break;
               }
               case java.sql.Types.REAL:
               {
                  result = ((Float)obj).doubleValue();
                  break;
               }
               case java.sql.Types.FLOAT:
               case java.sql.Types.DOUBLE:
               {
                  result = ((Number)obj).doubleValue();
                  break;
               }
               case java.sql.Types.CHAR:
               case java.sql.Types.VARCHAR:
               case java.sql.Types.LONGVARCHAR:
               {
                  try
                  {
                     Double d = new Double((String)obj);
                     result   = d.doubleValue();
                  }
                  catch (NumberFormatException e)
                  {
                     throw new SQLException(e.getMessage());
                  }
                  break;
               }
               case java.sql.Types.DECIMAL:
               case java.sql.Types.NUMERIC:
               {
                  result = ((BigDecimal)obj).doubleValue();
                  break;
               }
               case java.sql.Types.BIT:
               {
                  // XXX according to JDBC spec we need to handle these
                  // for now just fall through
               }
               default:
               {
                  throw new SQLException("Internal error. "
                                         + "Don't know how to convert from "
                                         + "java.sql.Types." +
                                         TdsUtil.javaSqlTypeToString(getMetaData().getColumnType(columnIndex))
                                         + " to an Dboule");
               }
            }
         }
         catch(ClassCastException e)
         {
            throw new SQLException("Couldn't convert column " + columnIndex
                                   + " to an long.  "
                                   + e.getMessage());
         }
      }
      return result;
   } /* getDouble()  */


   /**
    * Get the value of a column in the current row as a Java double.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public double getDouble(String columnName) throws SQLException
   {
      return getDouble(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java float.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public float getFloat(int columnIndex) throws SQLException
   {
      return (float)getDouble(columnIndex);
   }


   /**
    * Get the value of a column in the current row as a Java float.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public float getFloat(String columnName) throws SQLException
   {
      return getFloat(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java int.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public int getInt(int columnIndex) throws SQLException
   {
      return (int) getLong(columnIndex);
   }


   /**
    * Get the value of a column in the current row as a Java int.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public int getInt(String columnName) throws SQLException
   {
      return getInt(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java long.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public long getLong(int columnIndex) throws SQLException
   {
      long    result = 0;
      Object  obj    = getObject(columnIndex);

      if (obj == null)
      {
         result = 0;
      }
      else
      {
         try
         {
            switch(getMetaData().getColumnType(columnIndex))
            {
               case java.sql.Types.TINYINT:
               case java.sql.Types.SMALLINT:
               case java.sql.Types.INTEGER:
               {
                  result = ((Number)obj).longValue();
                  break;
               }
               case java.sql.Types.BIGINT:
               {
                  result = ((Number)obj).longValue();
                  break;
               }
               case java.sql.Types.REAL:
               case java.sql.Types.FLOAT:
               case java.sql.Types.DOUBLE:
               {
                  result = ((Number)obj).longValue();
                  break;
               }
               case java.sql.Types.CHAR:
               case java.sql.Types.VARCHAR:
               case java.sql.Types.LONGVARCHAR:
               {
                  try
                  {
                     Long i   = new Long((String)obj);
                     result   = i.longValue();
                  }
                  catch (NumberFormatException e)
                  {
                     throw new SQLException(e.getMessage());
                  }
                  break;
               }
               case java.sql.Types.NUMERIC:
               {
                  result = ((Number)obj).longValue();
                  break;
               }
               case java.sql.Types.DECIMAL:
               {
                  result = ((Number)obj).longValue();
                  break;
               }
               case java.sql.Types.BIT:
               {
                  // XXX according to JDBC spec we need to handle these
                  // for now just fall through
               }
               default:
               {
                  throw new SQLException("Internal error. "
                                         + "Don't know how to convert from "
                                         + "java.sql.Types " +
                                         TdsUtil.javaSqlTypeToString(getMetaData().getColumnType(columnIndex))
                                         + " to an long");
               }
            }
         }
         catch(ClassCastException e)
         {
            throw new SQLException("Couldn't convert column " + columnIndex
                                   + " to an long.  "
                                   + e.getMessage());
         }
      }
      return result;
   } /* getLong()  */


   /**
    * Get the value of a column in the current row as a Java long.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public long getLong(String columnName) throws SQLException
   {
      return getLong(findColumn(columnName));
   }


   /**
    * The number, types and properties of a ResultSet's columns
    * are provided by the getMetaData method.
    *
    * @return the description of a ResultSet's columns
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.ResultSetMetaData getMetaData() throws SQLException
   {
      if (metaData == null)
      {
         metaData = new ResultSetMetaData(columnsInfo);
      }
      return metaData;
   }


   /**
    * <p>Get the value of a column in the current row as a Java object.
    *
    * <p>This method will return the value of the given column as a
    * Java object.  The type of the Java object will be the default
    * Java Object type corresponding to the column's SQL type,
    * following the mapping specified in the JDBC spec.
    *
    * <p>This method may also be used to read datatabase specific abstract
    * data types.
    *
    * JDBC 2.0
    *
    * In the JDBC 2.0 API, the behavior of method
    * <code>getObject</code> is extended to materialize  
    * data of SQL user-defined types.  When the a column contains
    * a structured or distinct value, the behavior of this method is as 
    * if it were a call to: getObject(columnIndex, 
    * this.getStatement().getConnection().getTypeMap()).
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return A java.lang.Object holding the column value.
    * @exception SQLException if a database-access error occurs.
    */
   public Object getObject(int columnIndex) throws SQLException
   {
      // This method is implicitly coupled to the getRow() method in the
      // Tds class.  Every type that getRow() could return must
      // be handled in this method.
      //
      // The object type returned by getRow() must correspond with the
      // jdbc SQL type in the switch statement below.
      //
      // Note-  The JDBC spec (version 1.20) does not define the type
      // of the Object returned for LONGVARCHAR data.

      // XXX-  Needs modifications for JDBC 2.0

      Object   result = null;

      if (currentRow == null)
      {
         throw new SQLException("No current row in the result set.  " +
                                "Did you call ResultSet.next()?");
      }
      
      try
      {
         Object   tmp    = currentRow.getElementAt(columnIndex);

         lastGetWasNull = false;
         if (tmp == null)
         {
            lastGetWasNull = true;

            result = null;
         }
         else
         {
            switch(getMetaData().getColumnType(columnIndex))
            {
               case java.sql.Types.CHAR:
               case java.sql.Types.VARCHAR:
               {
                  if (tmp instanceof String)
                  {
                     result = tmp;
                  }
                  else
                  {
                     throw new SQLException("Was expecting CHAR data.  Got"
                                            + tmp.getClass().getName());
                  }
                  break;
               }
               case java.sql.Types.TINYINT:
               {
                  if (! (tmp instanceof Long))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = new Byte((byte) ((Long)tmp).intValue());
                  break;
               }
               case java.sql.Types.SMALLINT:
               {
                  if (! (tmp instanceof Long))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = new Short((short) ((Long)tmp).intValue());
                  break;
               }
               case java.sql.Types.INTEGER:
               {
                  if (! (tmp instanceof Long))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = new Integer(((Long)tmp).intValue());
                  break;
               }
               case java.sql.Types.BIGINT:
               {
                  if (! (tmp instanceof Long))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = (Long)tmp;
                  break;
               }
               case java.sql.Types.REAL:
               {
                  if (! (tmp instanceof Float))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = (Float)tmp;
                  break;
               }
               case java.sql.Types.FLOAT:
               case java.sql.Types.DOUBLE:
               {
                  if (tmp instanceof Double)
                  {
                     result = (Double)tmp;
                  }
                  else if (tmp instanceof Float)
                  {
                     result = new Double(((Float)tmp).doubleValue());
                  }
                  else
                  {
                     throw new SQLException("Was expecting Double data.  Got"
                                            + tmp.getClass().getName());
                  }

                  break;
               }
               case java.sql.Types.DATE:
               {
                  // XXX How do the time types hold up with timezones?
                  if (! (tmp instanceof Timestamp))
                  {
                     throw new SQLException("Internal error");
                  }

//                   java.util.Calendar  cal = new java.util.GregorianCalendar();
//                   cal.setTime(getTimestamp(columnIndex));
//                   result = cal.getTime();
                  result = new Date(((Timestamp)tmp).getTime());
                  break;
               }
               case java.sql.Types.TIME:
               {
                  if (! (tmp instanceof Timestamp))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = new Time(((Timestamp)tmp).getTime());
                  break;
               }
               case java.sql.Types.TIMESTAMP:
               {
                  if (! (tmp instanceof Timestamp))
                  {
                     throw new SQLException("Internal error");
                  }

                  result = (Timestamp) tmp;
                  break;
               }
               case java.sql.Types.BINARY:
               case java.sql.Types.VARBINARY:
               {
                  result = getBytes(columnIndex);
                  break;
               }
               case java.sql.Types.DECIMAL:
               case java.sql.Types.NUMERIC:
               {
                  if (tmp instanceof BigDecimal)
                  {
                     result = ((BigDecimal)tmp);
                  }
                  else
                  {
                     throw new SQLException("Was expecting NUMERIC data.  Got"
                                            + tmp.getClass().getName());
                  }
                  break;
               }
               case java.sql.Types.LONGVARCHAR:
               {
                  if (tmp instanceof TdsAsciiInputStream)
                  {
                     result = ((TdsAsciiInputStream)tmp).toString();
                  }
                  else if (tmp instanceof java.lang.String)
                  {
                     result = tmp;
                  }
                  else
                  {
                     throw new SQLException("Was expecting LONGVARCHAR data. "
                                            + "Got "
                                            + tmp.getClass().getName());
                  }
                  break;
               }
               case java.sql.Types.LONGVARBINARY:
               {
                  throw new SQLException("Not implemented");
               }
               case java.sql.Types.NULL:
               {
                  throw new SQLException("Not implemented");
               }
               case java.sql.Types.OTHER:
               {
                  throw new SQLException("Not implemented");
               }
               case java.sql.Types.BIT:
               {
                  if (tmp instanceof Boolean)
                  {
                     result = ((Boolean)tmp);
                  }
                  else
                  {
                     throw new SQLException("Was expecting BIT data. "
                                            + "Got"
                                            + tmp.getClass().getName());
                  }
                  break;
               }
               default:
               {
                  String msg = ""
                     + "Unknown datatype "
                     + getMetaData().getColumnType(columnIndex);
                  throw new SQLException(msg);
               }
            }
         }
      }
      catch (com.internetcds.jdbc.tds.TdsException e)
      {
         e.printStackTrace();
         throw new SQLException(e.getMessage());
      }
      return result;
   } // getObject()


   /**
    * <p>Get the value of a column in the current row as a Java object.
    *
    * <p>This method will return the value of the given column as a
    * Java object.  The type of the Java object will be the default
    * Java Object type corresponding to the column's SQL type,
    * following the mapping specified in the JDBC spec.
    *
    * JDBC 2.0
    *
    * 
    * In the JDBC 2.0 API, the behavior of method
    * <code>getObject</code> is extended to materialize  
    * data of SQL user-defined types.  When the a column contains
    * a structured or distinct value, the behavior of this method is as 
    * if it were a call to: getObject(columnIndex, 
    * this.getStatement().getConnection().getTypeMap()).
    *
    * <p>This method may also be used to read datatabase specific abstract
    * data types.
    *
    * @param columnName is the SQL name of the column
    * @return A java.lang.Object holding the column value.
    * @exception SQLException if a database-access error occurs.
    */
   public Object getObject(String columnName) throws SQLException
   {
      return getObject(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a Java short.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public short getShort(int columnIndex) throws SQLException
   {
      return (short) getLong(columnIndex);
   }


   /**
    * Get the value of a column in the current row as a Java short.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is 0
    * @exception SQLException if a database-access error occurs.
    */
   public short getShort(String columnName) throws SQLException
   {
      return getShort(findColumn(columnName));
   }


   //======================================================================
   // Methods for accessing results by column index
   //======================================================================

   /**
    * Get the value of a column in the current row as a Java String.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public String getString(int columnIndex) throws SQLException
   {
      Object  tmp = getObject(columnIndex);

      if (tmp == null)
      {
         return null;
      }
      else if (tmp instanceof byte[])
      {
         return new String((byte[])tmp);
      }
      else
      {
         return tmp.toString();
      }
   }


   //======================================================================
   // Methods for accessing results by column name
   //======================================================================

   /**
    * Get the value of a column in the current row as a Java String.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public String getString(String columnName) throws SQLException
   {
      return getString(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a java.sql.Time object.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Time getTime(int columnIndex) throws SQLException
   {
      java.sql.Time        result = null;
      java.sql.Timestamp   tmp    =  getTimestamp(columnIndex);
      
      if (tmp != null)
      {
         result = new java.sql.Time(tmp.getTime());
      }
      return result;
   }


   /**
    * Get the value of a column in the current row as a java.sql.Time object.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Time getTime(String columnName) throws SQLException
   {
      return getTime(findColumn(columnName));
   }


   /**
    * Get the value of a column in the current row as a java.sql.Timestamp object.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Timestamp getTimestamp(int columnIndex) throws SQLException
   {
      Timestamp result;

      try
      {
         Object   tmp = currentRow.getElementAt(columnIndex);

         lastGetWasNull = false;
         if (tmp == null)
         {
            lastGetWasNull = true;
            result = null;
         }
         else if (tmp instanceof Timestamp)
         {
            result = (Timestamp)tmp;
         }
         else
         {
            throw new SQLException("Can't convert column " + columnIndex
                                   + " from "
                                   + tmp.getClass().getName()
                                   + " to Timestamp");
         }
      }
      catch (TdsException e)
      {
         throw new SQLException(e.getMessage());
      }
      return result;
   }


   /**
    * Get the value of a column in the current row as a java.sql.Timestamp object.
    *
    * @param columnName is the SQL name of the column
    * @return the column value; if the value is SQL NULL, the result is null
    * @exception SQLException if a database-access error occurs.
    */
   public java.sql.Timestamp getTimestamp(String columnName) throws SQLException
   {
      return getTimestamp(findColumn(columnName));
   }


   /**
    * A column value can be retrieved as a stream of Unicode characters
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
    * do any necessary conversion from the database format into Unicode.
    *
    * <P><B>Note:</B> All the data in the returned stream must be
    * read prior to getting the value of any other column. The next
    * call to a get method implicitly closes the stream. . Also, a
    * stream may return 0 for available() whether there is data
    * available or not.
    *
    * @param columnIndex the first column is 1, the second is 2, ...
    * @return a Java input stream that delivers the database column value
    * as a stream of two byte Unicode characters.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getUnicodeStream(int columnIndex) throws SQLException
   {
       String val = getString(columnIndex);
       if (val == null)
           return null;
       try {
           return new ByteArrayInputStream(val.getBytes("UTF8"));
       } catch (UnsupportedEncodingException e) {
           // plain impossible with UTF-8
           return null;
       }
   }

   /**
    * A column value can be retrieved as a stream of Unicode characters
    * and then read in chunks from the stream.  This method is particularly
    * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
    * do any necessary conversion from the database format into Unicode.
    *
    * <P><B>Note:</B> All the data in the returned stream must
    * be read prior to getting the value of any other column. The
    * next call to a get method implicitly closes the stream.
    *
    * @param columnName is the SQL name of the column
    * @return a Java input stream that delivers the database column value
    * as a stream of two byte Unicode characters.  If the value is SQL NULL
    * then the result is null.
    * @exception SQLException if a database-access error occurs.
    */
   public java.io.InputStream getUnicodeStream(String columnName) throws SQLException
   {
      return getUnicodeStream(findColumn(columnName));
   }


   //=====================================================================
   // Advanced features:
   //=====================================================================

   /**
    * <p>The first warning reported by calls on this ResultSet is
    * returned. Subsequent ResultSet warnings will be chained to this
    * SQLWarning.
    *
    * <P>The warning chain is automatically cleared each time a new
    * row is read.
    *
    * <P><B>Note:</B> This warning chain only covers warnings caused
    * by ResultSet methods.  Any warning caused by statement methods
    * (such as reading OUT parameters) will be chained on the
    * Statement object.
    *
    * @return the first SQLWarning or null
    * @exception SQLException if a database-access error occurs.
    */
   public SQLWarning getWarnings() throws SQLException
   {
      return warningChain.getWarnings();
   }


   /**
    * A ResultSet is initially positioned before its first row; the
    * first call to next makes the first row the current row; the
    * second call makes the second row the current row, etc.
    *
    * <P>If an input stream from the previous row is open, it is
    * implicitly closed. The ResultSet's warning chain is cleared
    * when a new row is read.
    *
    * @return true if the new current row is valid; false if there
    * are no more rows
    * @exception SQLException if a database-access error occurs.
    */
   public boolean next() throws SQLException
   {
      boolean       result      = false;
      SQLException  exception   = null;
      boolean       done        = false;
      boolean       wasCanceled = false;


      if (isClosed)
      {
         throw new SQLException("result set is closed");
      }

      try
      {
         clearWarnings();

         Context   context = new Context();
         context.setColumnInfo(columnsInfo);


         // Keep eating garbage and warnings until we reach the next result
         while (!tds.isResultSet()    &&
                !tds.isEndOfResults() &&
                !tds.isResultRow())
         {
            // RMK 2000-06-08: don't choke on RET_STAT package.
            if (tds.isProcId() || tds.peek() == Tds.TDS_RET_STAT_TOKEN)
            {
               tds.processSubPacket();
            }
            else if (tds.isDoneInProc())
            {
               PacketDoneInProcResult tmp =
                  (PacketDoneInProcResult)tds.processSubPacket();
            }
            else if (tds.isTextUpdate())
            {
               PacketResult tmp1 =
                  (PacketResult)tds.processSubPacket();
            }
            else if (tds.isMessagePacket() || tds.isErrorPacket())
            {
               PacketMsgResult  tmp = (PacketMsgResult)tds.processSubPacket();
               exception = warningChain.addOrReturn(tmp);
            }
            else
            {
               throw new SQLException("Protocol confusion.  "
                                      + "Got a 0x"
                                      + Integer.toHexString((tds.peek() & 0xff))
                                      + " packet");
            }
         } // end while
         
         if (exception != null)
         {
            throw exception;
         }
         
         if (tds.isResultRow())
         {
            currentRow = (PacketRowResult)tds.processSubPacket(context);
            result = true;
            done   = true;
         }
         else if (tds.isEndOfResults())
         {
            PacketResult tmp = tds.processSubPacket(context);
            currentRow   = null;
            done         = true;
            hitEndOfData = true;
            wasCanceled = wasCanceled
               || ((PacketEndTokenResult)tmp).wasCanceled();
         }
         else if (!tds.isResultSet())
         {
            throw new SQLException("Protocol confusion.  "
                                   + "Got a 0x"
                                   + Integer.toHexString((tds.peek() & 0xff))
                                   + " packet");
         }


         if (exception != null)
         {
            throw exception;
         }
      }
      catch(java.io.IOException e)
      {
         throw new SQLException(e.getMessage());
      }
      catch(TdsException e)
      {
         e.printStackTrace();
         throw new SQLException(e.getMessage());
      }
      if (wasCanceled)
      {
         throw new SQLException("Query was canceled or timed out.");
      }
      return result;
   }


   /**
    * A column may have the value of SQL NULL; wasNull reports whether
    * the last column read had this special value.
    * Note that you must first call getXXX on a column to try to read
    * its value and then call wasNull() to find if the value was
    * the SQL NULL.
    *
    * @return true if last column read was SQL NULL
    * @exception SQLException if a database-access error occurs.
    */
   public boolean wasNull() throws SQLException
   {
      return lastGetWasNull;
   }
}
