
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.*;
import java.util.*;
import javax.swing.*;

/**
 * Component that display many lines of text against an empty background.
 * The text is left justified and displayed using SansSerif, 12pt.
 * Lines of text
 * can be inserted using the <tt>insertLine()</tt>, <tt>insertLineStrip()</tt>
 * and <tt>insertMessage()</tt> methods and can be removed using
 * <tt>removeAllLines()</tt>.
 * <p>
 * This component is used by the <tt>MessageBox</tt> class amongst other
 * things.
 *
 * @see org.gjt.btools.gui.dialog.MessageBox
 */
public class TextDisplayRegion extends JComponent {
    /**
     * Metrics for the font used to draw text.
     * @serial
     */
    private FontMetrics metrics;

    /**
     * Width in pixels of the longest text line.
     * @serial
     */
    private int maxWidth = 0;

    /**
     * Height in pixels of the entire text display.
     * @serial
     */
    private int textHeight;

    /**
     * Height in pixels of a single text line.
     * @serial
     */
    private int lineHeight;

    /**
     * Vertical padding between text and the edge of the component.
     * @serial
     */
    private int vPad = 0;

    /**
     * Horizontal padding between text and the edge of the component.
     * @serial
     */
    private int hPad = 0;

    /**
     * Actual text to be displayed.
     * @serial
     */
    private Vector allLines;

    /**
     * Create a new text display region.
     */
    public TextDisplayRegion() {
        // Set the font and its metrics.
        Font f = new Font("SansSerif", Font.PLAIN, 12);
        setFont(f);
        metrics = getFontMetrics(f);
        
        // Initialise variables.
        // lineHeight is stored as a separate variable because it will
        // be frequently called upon.
        textHeight = - metrics.getLeading();
        lineHeight = metrics.getHeight();
        allLines = new Vector();
    }

    /**
     * Returns the minimum size for this component to occupy.
     *
     * @return the component's minimum size.
     */
    public Dimension getMinimumSize() {
        // Add padding in each direction to the total size of the text display.
        return new Dimension((2*hPad) + maxWidth,
            (2*vPad) + textHeight);
    }

    /**
     * Returns the preferred size for this component to occupy.
     *
     * @return the component's preferred size.
     */
    public Dimension getPreferredSize() {
        return getMinimumSize();
    }

    /**
     * Removes all lines of text from the display region.
     * <p>
     * If the component is visible or already belongs to a container,
     * completeRefresh() will need to be called after this routine.
     */
    public void removeAllLines() {
        textHeight = - metrics.getLeading();
        lineHeight = metrics.getHeight();
        allLines.removeAllElements();
    }

    /**
     * Inserts a line of text into the display region.
     * This line will be placed after all existing lines.
     * <p>
     * If the component is visible or already belongs to a container,
     * completeRefresh() will need to be called after this routine.
     * <p>
     * The method <tt>insertLineStrip()</tt> performs the same function as
     * <tt>insertLine()</tt>, but first strips all trailing spaces from
     * the line of text.
     *
     * @param line the line of text to be inserted.
     * @see #insertLineStrip
     */
    public void insertLine(String line)
    {
        maxWidth = Math.max(maxWidth, metrics.stringWidth(line));
        textHeight += lineHeight;
        allLines.addElement(line);
    }

    /**
     * Inserts a line of text into the display region, first stripping all
     * trailing spaces.  This line will be placed after all existing lines.
     * <p>
     * If the component is visible or already belongs to a container,
     * completeRefresh() will need to be called after this routine.
     * <p>
     * If the string is known to have no trailing spaces,
     * <tt>insertLine()</tt> should be used instead.
     *
     * @param line the line of text to be inserted.
     * @see #insertLine
     */
    public void insertLineStrip(String line) {
        // Strip spaces from the end of line.
        // We can't use String.trim(), since we only wish to strip
        // from the end.
        int end = line.length();
        
        // end will now become the length of the final stripped string.
        while (end > 0) {
            if (line.charAt(end-1) != ' ')
                break;
            end--;
        }

        insertLine(line.substring(0, end));
    }

    /**
     * Inserts the given message into the text display region, splitting
     * the message across several lines to ensure that no line is too long.
     * The message is inserted below any lines that might already be there.
     * <p>
     * If the component is visible or already belongs to a container,
     * completeRefresh() will need to be called after this routine.
     *
     * @param msg the message to be inserted.
     * @param maxLineLength the maximum number of characters to place on
     * a single line.
     */
    public void insertMessage(String msg, int maxLineLength) {
        // See if there is a newline in the first line's worth
        // of characters.
        int newline = msg.indexOf('\n');
        if (newline >= 0 && newline <= maxLineLength) {
            // Break the line at newline and then
            // insert the rest of the message.
            insertLineStrip(msg.substring(0, newline));
            insertMessage(msg.substring(newline+1), maxLineLength);
            return;
        }
        
        // We need to siphon off at most maxLineLength
        // characters for the first line.
        if (msg.length() > maxLineLength) {
            // The message is too long to fit on a single line.
            // Find somewhere appropriate to break it.
            // Start by looking for the last space within the allowable
            // line length.
            int space = msg.lastIndexOf(' ', maxLineLength);
            if (space == -1) {
                // There are no spaces at which to break the line.
                // Just use the first MaxLineLength characters, and then
                // insert the rest of the message.
                insertLine(msg.substring(0, maxLineLength));
                insertMessage(msg.substring(maxLineLength), maxLineLength);
                    return;
            }
            
            // We have found a space at which to break the line.
            // Insert this first line.
            insertLineStrip(msg.substring(0, space));
            
            // Now move forwards to the next non-space character.
            space++;
            int len = msg.length();
            while (space < len) {
                // Known: msg[space-1] == ' '.
                if (msg.charAt(space) != ' ') {
                    // We have found somewhere to start the next line.
                    // If it's a newline character, skip it, since we're
                    // already going to a new line.
                    if (msg.charAt(space) == '\n')
                        insertMessage(msg.substring(space+1), maxLineLength);
                    else
                        insertMessage(msg.substring(space), maxLineLength);
                    return;
                }
                space++;
            }
            // We have reached the end of msg.
            // Thus there are no more non-space characters in msg.
            // So do nothing.
        }
        else {
            // The message is short enough to fit on a single line.
            insertLineStrip(msg);
        }
    }

    /**
     * Called whenever the TextDisplayRegion is to be painted.
     *
     * @param g the graphics object through which to paint.
     */
    public void paint(Graphics g) {
        // Just draw the individual lines of text.
        
        // currentHeight represents the height at which to draw the next line.
        int currentHeight = hPad + metrics.getAscent();
        Enumeration e = allLines.elements();
    
        while (e.hasMoreElements()) {
            // Draw the next line of text.
            g.drawString((String)e.nextElement(), hPad, currentHeight);
            currentHeight += lineHeight;
        }
    }

    /**
     * Sets the padding in pixels between the text and the edge of the
     * component.  Default values are zero in each direction.
     *
     * @param hPad horizontal padding.
     * @param vPad vertical padding.
     */
    public void setPads(int hPad, int vPad) {
        this.hPad = hPad;
        this.vPad = vPad;
    }

    /**
     * Performs any revalidation and repainting required when the
     * contents of this region change.
     * This routine should be called if the contents change whilst this
     * region is already visible or belongs to a container.
     * <p>
     * The current implementation is simply to call
     * <tt>revalidate()</tt> and <tt>repaint()</tt>.
     */
    public void completeRefresh() {
        revalidate();
        repaint();
    }
}

