//
// File:        Grammar.java
// Package:     gov.llnl.babel.parsers.sidl
// Revision:    @(#) $Id: Grammar.java 4434 2005-03-17 17:05:29Z epperly $
// Description: utility class for deciphering productions in the sidl grammar
//
// Copyright (c) 2000-2004, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.parsers.sidl;

import gov.llnl.babel.parsers.sidl.ParserConstants;

/**
 * <code>Grammar</code> defines integer constants for the major productions
 * in the SIDL grammar.  These production numbers are used by the parser to
 * identify a particular grammar production in the event of a parse error.
 * This class also provides lookup functions that translate grammar tokens
 * and productions into human-friendly strings.
 */
public class Grammar {
   public static final int SPECIFICATION       =  1;
   public static final int REQUIRE             =  2;
   public static final int IMPORT              =  3;
   public static final int PACKAGE             =  4;
   public static final int DEFINITION          =  5;
   public static final int CLASS               =  6;
   public static final int ENUM                =  7;
   public static final int ENUMERATOR          =  8;
   public static final int INTERFACE           =  9;
   public static final int INVARIANTS          = 10;
   public static final int CLASS_METHOD        = 11;
   public static final int METHOD              = 12;
   public static final int ARGUMENT            = 13;
   public static final int TYPE                = 14;
   public static final int ARRAY               = 15;
   public static final int ARRAY_ORDER         = 16;
   public static final int RARRAY	       = 17;
   public static final int REQUIRES            = 18;
   public static final int ENSURES             = 19;
   public static final int ASSERTIONS          = 20;
   public static final int ASSERTION_EXPR      = 21;
   public static final int CONDITIONAL_EXPR    = 22;
   public static final int INCLUSIVE_OR_EXPR   = 23;
   public static final int EXCLUSIVE_OR_EXPR   = 24;
   public static final int AND_EXPR            = 25;
   public static final int EQUALITY_EXPR       = 26;
   public static final int RELATIONAL_EXPR     = 27;
   public static final int SHIFT_EXPR          = 28;
   public static final int ADDITIVE_EXPR       = 29;
   public static final int MULTIPLICATIVE_EXPR = 30;
   public static final int POWER_EXPR          = 31;
   public static final int UNARY_EXPR          = 32;
   public static final int POSTFIX_EXPR        = 33;
   public static final int PRIMARY_EXPR        = 34;
   public static final int RESERVED_EXPR       = 35;
   public static final int LITERAL_EXPR        = 36;
   public static final int ARGUMENT_EXPR_LIST  = 37;
   public static final int SCOPED_IDENTIFIER   = 38;

   public static final int MIN_PRODUCTION      =  1;
   public static final int MAX_PRODUCTION      = 38;

   /**
    * Static utility method <code>getProduction</code> returns a string
    * representing the specified production integer.
    */
   public static String getProduction(int production) {
      String s = null;
      switch (production) {
      case SPECIFICATION:
         s = "Specification ::= (Require)* (Import)* (Package)* EOF";
         break;
      case REQUIRE:
         s = "Require ::= \"require\" ScopedIdentifier \"version\" "
           + "V1.V2...Vn\";\"";
         break;
      case IMPORT:
         s = "Import ::= \"import\" ScopedIdentifier [ \"version\" "
           + "V1.V2...Vn ] \";\"";
         break;
      case PACKAGE:
         s = "Package ::= [ \"final\" ] \"package\" ScopedIdentifier "
           + "[ \"version\" V1.V2...Vn ] "
           + "\"{\" (Definition)* \"}\" [ \";\" ]";
         break;
      case DEFINITION:
         s = "Definition ::= Class | Enum | Interface | Package";
         break;
      case CLASS:
         s = "Class ::= [ \"abstract\" ] \"class\" Identifier "
           + "[ \"extends\" ScopedIdentifier ] "
           + "[ \"implements-all\" ScopedIdentifier "
           + "( \",\" ScopedIdentifier )* ] "
           + "[ \"implements\" ScopedIdentifier "
           + "( \",\" ScopedIdentifier )*]  "
           + "\"{\" [ Invariants ] "
           + "(ClassMethod)* \"}\" [ \";\" ]";
         break;
      case ENUM:
         s = "Enum ::= \"enum\" Identifier "
           + "\"{\" Enumerator ( \",\" Enumerator )* \"}\" [ \";\" ]";
         break;
      case ENUMERATOR:
         s = "Enumerator ::= Identifier [ \"=\" Integer ]";
         break;
      case INTERFACE:
         s = "Interface ::= \"interface\" Identifier "
           + "[ \"extends\" ScopedIdentifier ( \",\" ScopedIdentifier )* ] "
           + "\"{\" [ Invariants ] (Method)* "
           + "\"}\" [ \";\" ]";
         break;
      case INVARIANTS:
         s = "Invariants ::= \"invariant\" Assertions";
         break;
      case CLASS_METHOD:
         s = "ClassMethod ::= [ (\"abstract\" | \"final\" | \"static\") ] "
           + "Method";
         break;
      case METHOD:
         s = "Method ::= (\"void\" | [\"copy\"] Type) Identifier "
           + "[ \"[\" Identifier \"]\" ] "
           + "\"(\" [ Argument (\",\" Argument )* ] \")\" "
           + "[ (\"local\" | \"oneway\") ] "
           + "[ \"throws\" ScopedIdentifier "
           + "( \",\" ScopedIdentifier )* ] "
           + "[ Requires ] [ Ensures ] \";\"";
         break;
      case ARGUMENT:
         s = "Argument ::= [ \"copy\" ] ( \"in\" | \"inout\" | \"out\" ) "
            + "Type Identifier";
         break;
      case TYPE:
         s = "Type ::= \"bool\" | \"char\" | \"dcomplex\" "
           + "| \"double\" | \"fcomplex\" | \"float\" | \"int\" "
           + "| \"long\" | \"opaque\" | \"string\" | Array "
           + "| ScopedIdentifier";
         break;
      case ARRAY:
         s = "Array := \"array<\" Type [ \",\" ( Integer "
           + "[ \",\" ArrayOrder ] | ArrayOrder ) ] \">\"";
         break;
      case ARRAY_ORDER:
         s = "ArrayOrder := ( \"column-major\" | \"row-major\" )";
         break;
      case RARRAY:
	 s = "Rarray := \"rarray<\" Type [ \",\" Integer ]> "
           + " Identifier \"(\" Index ( \",\" Index )* \")\" ";
         break;
      case REQUIRES:
         //s = "Requires ::= \"require\" [ \"else\" ] Assertions";
         s = "Requires ::= \"require\" Assertions";
         break;
      case ENSURES:
         //s = "Ensures ::= \"ensure\" [ \"then\" ] Assertions";
         s = "Ensures ::= \"ensure\" Assertions";
         break;
      case ASSERTIONS:
         s = "Assertions ::= "
           + "(AssertionExpr [ \":\" AssertionExpr ] \";\"] )+ ";
         break;
      case ASSERTION_EXPR:
         s = "AssertionExpr ::= "
           + "ConditionalExpr [ (\"implies\" | \"iff\") ConditionalExpr ]";
         break;
      case CONDITIONAL_EXPR:
         s = "ConditionalExpr ::= InclusiveORExpr";
         break;
      case INCLUSIVE_OR_EXPR:
         s = "InclusiveORExpr ::= ExclusiveORExpr [ \"or\" ExclusiveORExpr ]";
         break;
      case EXCLUSIVE_OR_EXPR:
         s = "ExclusiveORExpr ::= ANDExpr [ \"xor\" ANDExpr ]";
         break;
      case AND_EXPR:
         s = "ANDExpr ::= EqualityExpr [ \"and\" EqualityExpr ]";
         break;
      case EQUALITY_EXPR:
         s = "EqualityExpr ::= "
           + "RelationalExpr [ (\"==\" | \"!=\" | \"<>\") RelationalExpr ]";
         break;
      case RELATIONAL_EXPR:
         s = "RelationalExpr ::= "
           + "ShiftExpr [ (\"<\" | \">\" | \"<=\" | \">=\") ShiftExpr ]";
         break;
      case SHIFT_EXPR:
         s = "ShiftExpr ::= "
           + "AdditiveExpr ((\">>\" | \"<<\") AdditiveExpr)*";
         break;
      case ADDITIVE_EXPR:
         s = "AdditiveExpr ::= "
           + "MultiplicativeExpr ((\"+\" | \"-\") MultiplicativeExpr)*";
         break;
      case MULTIPLICATIVE_EXPR:
         s = "MultiplicativeExpr ::= "
           + "PowerExpr ((\"*\" | \"/\" | \"mod\" | \"rem\") PowerExpr)*";
         break;
      case POWER_EXPR:
         s = "PowerExpr ::= "
           + "UnaryExpr ((\"^\" | \"**\") UnaryExpr)*";
         break;
      case UNARY_EXPR:
         s = "UnaryExpr ::= "
           + "[ (\"~\" | \"not\" | \"is\") ] "
           + "PostfixExpr";
         break;
      case POSTFIX_EXPR:
         s = "PostfixExpr ::= PrimaryExpr "
           + "| Identifier \"(\" [ArgumentExpressionList] \")\" )";
         break;
      case PRIMARY_EXPR:
         s = "PrimaryExpr ::= Reserved | Identifier | Literal "
           + "| \"(\" ConditionalExpr \")\" "
           + "| \"{\" FloatLiteral \",\" FloatLiteral \"}\"";
         break;
      case RESERVED_EXPR:
         s = "Reserved ::= \"null\" | \"pure\" | \"result\" | BooleanLiteral ";
         break;
      case LITERAL_EXPR:
/*
         s = "Literal ::= CharacterLiteral | FloatLiteral | IntegerLiteral "
           + "| StringLiteral";
*/
         s = "Literal ::= FloatLiteral | IntegerLiteral ";
         break;
      case ARGUMENT_EXPR_LIST:
         s = "ArgumentExprList ::= "
           + "ConditionalExpr (\",\" ConditionalExpr)*";
         break;
      case SCOPED_IDENTIFIER:
         s = "ScopedIdentifier ::= Identifier (\".\" Identifier)*";
         break;
      default:
         s = "Internal error (unknown production)";
         break;
      }
      return s;
   }

   /**
    * Static utility method <code>getTokenString</code> returns a string
    * token representation for the specified token number.
    */
   public static String getTokenString(int token) {
      switch (token) {
      case ParserConstants.EOF:
         return "\"END-OF-FILE\"";
      case ParserConstants.T_IDENTIFIER:
         return "\"identifier\"";
      case ParserConstants.T_VERSION_STRING:
         return "\"V1.V2...Vn\"";
      case ParserConstants.T_INTEGER:
         return "\"integer\"";
      case ParserConstants.T_DIGIT:
         return "\"[0-9]\"";
      case ParserConstants.T_LETTER:
         return "\"[a-z,A-Z]\"";
      default:
         return ParserConstants.tokenImage[token];
      }
   }

   /**
    * Print the grammar productions to the standard system output.
    */
   public static void main(String[] args) {
      System.out.println("*** Grammar productions are:");
      System.out.println();
      for (int p = MIN_PRODUCTION; p <= MAX_PRODUCTION; p++) {
         System.out.println(getProduction(p));
         System.out.println();
      }

      System.out.println();
      System.out.println();
      System.out.print("*** Conversions are made below only for the ");
      System.out.println("following:"); 
      System.out.println("DIGIT          = "+ParserConstants.T_DIGIT);
      System.out.println("IDENTIFIER     = "+ParserConstants.T_IDENTIFIER);
      System.out.println("INTEGER        = "+ParserConstants.T_INTEGER);
      System.out.println("LETTER         = "+ParserConstants.T_LETTER);
      System.out.println("VERSION_STRING = "+ParserConstants.T_VERSION_STRING);

      System.out.println();
      for (int p=ParserConstants.EOF; p<=ParserConstants.T_CATCH_ALL; p++) {
         System.out.println("token "+p+" = "+getTokenString(p));
      }
   }
}
