//
// File:        CxxStubHeader.java
// Package:     gov.llnl.babel.backend.cxx
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.rmi2;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.rmi2.Cxx;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Iterator;

/**
 * 
 */
public class CxxStubHeader {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxStubHeader(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    String filename = Cxx.generateFilename( d_ext.getSymbolID(), 
                                            Cxx.FILE_ROLE_STUB, 
                                            Cxx.FILE_TYPE_CXX_HEADER );
    //System.out.println("Create " + filename + "..." );
    
    try { 
      d_writer = Cxx.createHeader( d_ext, Cxx.FILE_ROLE_STUB, "STUBHDRS");
      d_writer.println();
      d_writer.openHeaderGuard( filename );

      writeClassDeclaration();
      
      Cxx.generateDependencyIncludes( d_writer, d_ext, true );

      Cxx.nestPackagesInNamespaces( d_writer, d_ext );
      
      writeClassBegin();
      
      writeUserDefinedMethods();
      
      writeTypedefs();

      writeConstructors();

      writeCastingOperators();

      writeBindingSpecificMethods();

      writeClassEnd(); 
      
      Cxx.unnestPackagesInNamespaces( d_writer, d_ext );
      
      writeArrayDefinition();

      d_writer.closeHeaderGuard();
    } catch ( Exception ex ) { 
      throw new CodeGenerationException("Exception : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  private void writeClassDeclaration() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    
    d_writer.writeCommentLine("declare class before #includes");
    d_writer.writeCommentLine("(this alleviates circular #include guard problems)[BUG#393]");
    Cxx.nestPackagesInNamespaces( d_writer, d_ext );
    d_writer.println("class " + name + ";");
    Cxx.unnestPackagesInNamespaces( d_writer, d_ext );
    d_writer.writeCommentLine("Some compilers need to define array template before the specializations");
    d_writer.generateInclude("sidl_cxx.hh", true );
    d_writer.println("namespace sidl {");
    d_writer.tab();
    d_writer.println("template<>");
    d_writer.println("class array< " +  Cxx.getObjectName(id) + 
                     " >;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

  }

  private void writeClassBegin() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();

    d_writer.writeComment(d_ext, true);    
    d_writer.println("class " + name + " : public ::sidl::StubBase {");
    d_writer.tab();
  }

  private void writeUserDefinedMethods() throws CodeGenerationException { 
    d_writer.beginBoldComment();
    d_writer.println("User Defined Methods");
    d_writer.endBoldComment();

    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    
    Iterator m = null;
    m = d_ext.getStaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      Cxx.generateMethodSignature( d_writer, method, 
                                   "user defined static method", 
                                   Cxx.FILE_ROLE_STUB );
      d_writer.println();
    }

    m = d_ext.getNonstaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      Cxx.generateMethodSignature( d_writer, method, 
                                   "user defined non-static method.", 
                                   Cxx.FILE_ROLE_STUB );
      d_writer.println();
    }
    d_writer.beginBoldComment();
    d_writer.println("End User Defined Methods");
    d_writer.println("(everything else in this file is specific to");
    d_writer.println(" Babel's C++ bindings)");
    d_writer.endBoldComment();
  }

  private void writeTypedefs() { 
    SymbolID id = d_ext.getSymbolID();

    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    d_writer.println("typedef " + IOR.getObjectName( id ) + " ior_t;");
    d_writer.println("typedef " + IOR.getExternalName( id ) + " ext_t;");
    d_writer.println("typedef " + IOR.getSEPVName( id ) + " sepv_t;");
    d_writer.println();
  }

  private void writeConstructors() { 
    SymbolID id = d_ext.getSymbolID();
    String ior_ptr = "ior_t*";
    String name = id.getShortName();

    d_writer.writeCommentLine("default constructor");
    d_writer.println( name + "() : d_self(0), d_weak_reference(false) { }");
    d_writer.println();
    
    if( ! d_ext.isAbstract() ) {
      d_writer.writeCommentLine("static constructor");
      d_writer.println("static " + Cxx.getObjectName(id) + " _create();");
    
      d_writer.println();
    }

    d_writer.writeCommentLine("default destructor");
    d_writer.println("virtual ~" + name + " ();");
    d_writer.println();
    
    d_writer.writeCommentLine("copy constructor");
    d_writer.println( name + " ( const " + name + "& original );");
    d_writer.println();

    d_writer.writeCommentLine("assignment operator");
    d_writer.println( name + "& operator= ( const " + name + "& rhs );");
    d_writer.println();
  }
  
  private void writeCastingOperators() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    String ior_ptr = "ior_t*";

    d_writer.writeCommentLine("conversion from ior to C++ class");
    d_writer.println( name + " ( " + name + "::ior_t* ior );");
    d_writer.println();

    d_writer.writeCommentLine("Alternate constructor: does not call addRef()");
    d_writer.writeCommentLine("(sets d_weak_reference=isWeak)");
    d_writer.writeCommentLine("For internal use by Impls (fixes bug#275)");
    d_writer.println( name + " ( " + name + "::ior_t* ior, bool isWeak );");
    d_writer.println();

    d_writer.writeCommentLine("conversion from a StubBase");
    d_writer.println( name + " ( const ::sidl::StubBase& base );");
    d_writer.println();
  }

  private void writeBindingSpecificMethods() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    String ior_ptr = "ior_t*";

    d_writer.println( ior_ptr + " _get_ior() { return d_self; }");
    d_writer.println();

    d_writer.println("const " + ior_ptr + " _get_ior() const { return d_self; }");
    d_writer.println();

    d_writer.println( "void _set_ior( " + ior_ptr + " ptr ) { d_self = ptr; }");
    d_writer.println();

    d_writer.println("bool _is_nil() const { return (d_self==0); }");
    d_writer.println();

    d_writer.println("bool _not_nil() const { return (d_self!=0); }");
    d_writer.println();

    d_writer.println("bool operator !() const { return (d_self==0); }");
    d_writer.println();
    
    d_writer.backTab();
    d_writer.println("protected:");
    d_writer.tab();
 
    d_writer.println("virtual void* _cast(const char* type) const;");
    d_writer.println();

    d_writer.backTab();
    d_writer.println("private:");
    d_writer.tab();
    d_writer.writeCommentLine("Pointer to sidl's IOR type (one per instance)");
    d_writer.println("ior_t * d_self;");
    d_writer.println();
    d_writer.writeCommentLine("Weak references (used by Impl's only) don't add/deleteRef()");
    d_writer.println("bool d_weak_reference;");
    d_writer.println();
    d_writer.writeCommentLine("Pointer to external (DLL loadable) symbols (shared among instances)");
    d_writer.println("static const ext_t * s_ext;");
    d_writer.println();
    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    d_writer.println("static const ext_t * _get_ext() throw ( ::sidl::NullIORException );");
    d_writer.println();
    if ( d_ext.hasStaticMethod(true) ) { // if has static methods
      d_writer.println("static const sepv_t * _get_sepv() {");
      d_writer.tab();
      d_writer.println("return (*(_get_ext()->getStaticEPV))();");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }
  private void writeClassEnd() { 
    String name = d_ext.getSymbolID().getShortName();
    d_writer.backTab();
    d_writer.println("}; // end class " + name);
  } 

  private void writeArrayDefinition() { 
    SymbolID id = d_ext.getSymbolID();
    String cxx_item_t = Cxx.getObjectName(id);
    String cxx_array_t = "array< " + cxx_item_t + " >";
    String ior_item_t = IOR.getObjectName(id);
    String ior_array_t = IOR.getArrayName(id);
    String array_traits = "array_traits< " +cxx_item_t + " >";

    d_writer.println("namespace sidl {");
    d_writer.tab();
    
    d_writer.writeCommentLine("traits specialization");
    d_writer.println("template<>");
    d_writer.println("struct " + array_traits + " {");
    d_writer.tab();
    d_writer.println("typedef " + cxx_array_t + " cxx_array_t;");
    d_writer.println("typedef " + cxx_item_t + " cxx_item_t;");
    d_writer.println("typedef " + ior_array_t + " ior_array_t;");
    d_writer.println("typedef sidl_interface__array ior_array_internal_t;");
    d_writer.println("typedef " +  ior_item_t + " ior_item_t;");
    d_writer.println("typedef cxx_item_t value_type;");
    d_writer.println("typedef value_type reference;");
    d_writer.println("typedef value_type* pointer;");
    d_writer.println("typedef const value_type const_reference;");
    d_writer.println("typedef const value_type* const_pointer;");
    d_writer.println("typedef array_iter< " + array_traits + " > iterator;");
    d_writer.println("typedef const_array_iter< " + array_traits+ " > const_iterator;");
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println();

    d_writer.writeCommentLine("array specialization");
    d_writer.println("template<>");
    d_writer.println("class " + cxx_array_t +
                     ": public interface_array< " + array_traits + " > {");
    d_writer.println("public:");
    d_writer.tab();
    
    d_writer.println("typedef interface_array< " + array_traits + " > Base;");
    d_writer.println("typedef " + array_traits + "::cxx_array_t          cxx_array_t;");
    d_writer.println("typedef " + array_traits + "::cxx_item_t           cxx_item_t;");
    d_writer.println("typedef " + array_traits + "::ior_array_t          ior_array_t;");
    d_writer.println("typedef " + array_traits + "::ior_array_internal_t ior_array_internal_t;");
    d_writer.println("typedef " + array_traits + "::ior_item_t           ior_item_t;");
    d_writer.println();
    d_writer.beginBlockComment(true);
    d_writer.println("conversion from ior to C++ class");
    d_writer.println("(constructor/casting operator)");
    d_writer.endBlockComment(true);
    d_writer.println("array( " + ior_array_t + 
                     "* src = 0) : Base(src) {}");
    d_writer.println();
 

    d_writer.beginBlockComment(true);
    d_writer.println("copy constructor");
    d_writer.endBlockComment(true);
    d_writer.println("array( const " + cxx_array_t + "&src) {");
    d_writer.tab();
    d_writer.println("d_array = src.d_array;");
    d_writer.println("if (d_array) addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
 
    d_writer.beginBlockComment(true);
    d_writer.println("assignment");
    d_writer.endBlockComment(true);
    d_writer.println(cxx_array_t + "&");
    d_writer.println("operator =( const " + cxx_array_t + "&rhs ) { ");
    d_writer.tab();
    d_writer.println("if (d_array != rhs.d_array) {");
    d_writer.tab();
    d_writer.println("if (d_array) deleteRef();");
    d_writer.println("d_array = rhs.d_array;");
    d_writer.println("if (d_array) addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return *this;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
 
    d_writer.backTab();
    d_writer.println("};");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }
}
