//
// File:        StubSource.java
// Package:     gov.llnl.babel.backend.c
// Revision:    @(#) $Id: StubSource.java 4462 2005-03-23 19:29:24Z leek2 $
// Description: generate C backend stub source to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.rmi.RMI;
import gov.llnl.babel.backend.rmi.RMIStubSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

/**
 * Class <code>StubSource</code> generates an C Backend stub source file
 * to a language writer output stream.  The constructor takes a language
 * writer stream and method <code>generateCode</code> generates intermediate
 * object representation for the specified symbol to the output stream.  The
 * language writer output stream is not closed by this object.
 */
public class StubSource {
  private LanguageWriterForC d_writer;

  private static final String s_self = StubHeader.s_self;
  private static final String s_epv  = StubHeader.s_epv;

  private static final String s_externals      = "_externals";
  private static final String s_externals_func = "_getExternals";


  /**
   * This is a convenience utility function that writes the C client
   * stub source information into the provided language writer output
   * stream.  The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   *
   * @param sym the <code>Symbol</code> whose stub source is being written.
   *
   * @param writer the output writer to which the stub source will
   *               be written. This will not be closed.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Symbol symbol, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateCode(symbol);
  }

  /**
   * This is a convenience utility function specifically for the generation
   * of super "Stub" functions in the Impl files.
   * The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   *
   * @param methods is a collection of super methods to be generated.
   *
   * @param writer the output writer to which the stub source will
   *               be written. This will not be closed.
   *
   * @param cls The class in which these supers are to be generated
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateSupers(Collection methods, Class cls,
                                    LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateSupers(methods, cls);
  }

  public static void generateGetExternals(Class cls, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateGetExternals(cls.getSymbolID());
  }

  /**
   * Create a <code>StubSource</code> object that will write symbol
   * information to the provided output writer stream.
   *
   * @param writer the output writer to which the header will be written.
   *               This will not be closed.
   */
  public StubSource(LanguageWriterForC writer) {
    d_writer = writer;
  }

  /**
   * Write C stub source information for the provided symbol to the
   * language writer output stream provided in the constructor.  This
   * method does not close the language writer output stream and may
   * be called for more than one symbol (although the generated source
   * may not be valid input for the C compiler).  A code generation
   * exception is generated if an error is detected.  No code is
   * generated for enumerated and package symbols.
   *
   * @param sym the <code>Symbol</code> whose stub source is being written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    if (symbol != null) {
      switch (symbol.getSymbolType()) {
        case Symbol.CLASS:
        case Symbol.INTERFACE:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.ENUM:
          generateEnum(symbol);
          break;
        case Symbol.PACKAGE:
          break;
        default:
          throw new CodeGenerationException("Unsupported symbol type.");
      }
    } else {
      throw new CodeGenerationException("Unexpected null Symbol.");
    }
  }

  /**
   * This function is designed to generate stubs to the super functions
   * available in this Impl files.  These stubs are generated IN the Impl
   * file.  This method does not close the language writer output stream
   * A code generation exception is generated if an error is detected.
   * No code is generated for enumerated and package symbols.
   *
   * @param methods A collection of methods to write out.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public void generateSupers(Collection methods, Class cls)
    throws CodeGenerationException
  {
    SymbolID clsID = cls.getSymbolID();
    String self = "self";
    if (methods != null) {
      if (methods.size() > 0) {
        //declaring the IOR for super functions
        d_writer.println("static const struct " 
          + C.getObjectName(cls.getParentClass().getSymbolID()) 
          + "__epv* superEPV = " + C.NULL + ";");

        d_writer.println();
        d_writer.println("void " + C.getObjectName(clsID) + "__superEPV(");
        d_writer.println("struct "
          + C.getObjectName(cls.getParentClass().getSymbolID())
          + "__epv* parentEPV){");
        d_writer.tab();
        d_writer.println("superEPV = parentEPV;");
        d_writer.backTab();
        d_writer.println("}");

        for (Iterator m = methods.iterator(); m.hasNext(); ) {
          Method method = (Method) m.next();
          generateMethodSignature(self, false, clsID, method, true);
          d_writer.println("{");
          d_writer.tab();
          if(method.hasRarray()) {
            generateRarrays(method);
          }
          //Get the super IOR pointer
          //d_writer.println("superEPV = superEPV ? superEPV 
          //                 : _getExternals()->getSuperEPV();");
          if (getReturnString(method.getReturnType()) != "void") {
            d_writer.print("return ");
          }
          d_writer.print("(*superEPV");
          d_writer.print("->");
          d_writer.print(IOR.getVectorEntry(method.getLongMethodName()));
          d_writer.println(")((" 
            + IOR.getObjectName(cls.getParentClass().getSymbolID()) + "*)");
          d_writer.tab();
          C.generateArgumentList(d_writer, self, false, clsID, method, false,
                                 false, false, true, false, false, true);
          d_writer.println(");");
          d_writer.backTab();
          d_writer.backTab();
          d_writer.println("}");
          d_writer.println();
        }
      }
    } else {
      throw new CodeGenerationException("Unexpected null Symbol.");
    }
  }

  /**
   * Generate a return string for the specified SIDL type.  Most
   * of the SIDL return strings are listed in the static structures defined
   * at the start of this class.  Symbol types and array types require
   * special processing.
   */
  private static String getReturnString(Type type)
    throws CodeGenerationException
  {
    return IOR.getReturnString(type, false, false);
  }

  /**
   * Generate the Stub source for a SIDL class or interface.  The source
   * file begins with a banner and include files followed by the stub
   * implementations for the methods.
   *
   * @param ext the <code>Extendable</code> whose stub source is being written.
   */
  private void generateExtendable(Extendable ext) throws CodeGenerationException
  {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id        = ext.getSymbolID();
    String   source    = C.getStubFile(id);
    String   header    = C.getHeaderFile(id);
    String   iorHeader = IOR.getHeaderFile(id);

    d_writer.writeBanner(ext, source, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());

    d_writer.generateInclude(header, false);
    d_writer.generateInclude(iorHeader, false);
    d_writer.generateInclude("sidl_interface_IOR.h", true);
    d_writer.printlnUnformatted("#include <stddef.h>");
    if (!iorHeader.equals("sidl_BaseInterface_IOR.h")) {
      d_writer.generateInclude("sidl_BaseInterface_IOR.h", false);
    }
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.printlnUnformatted("#include \"babel_config.h\"");
      d_writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      d_writer.printlnUnformatted("#include <stdio.h>");
      d_writer.printlnUnformatted("#include <stdlib.h>");
      d_writer.printlnUnformatted("#include <string.h>");
      d_writer.printlnUnformatted("#include \"sidl_Loader.h\"");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.println();

    /*
     * Output standard function stubs for methods in the object.
     */
    if (!ext.isInterface()) {
      generateGetExternals(id);
    }
    if (ext.hasStaticMethod(true)) {
      generateGetStaticEPV(id);
    }
    generateMethodStubs(ext);
    ArrayMethods ar = new ArrayMethods(id, false);
    ar.generateStub(d_writer);
    RMIStubSource.generateCode(ext, d_writer);
  }

  /**
   * Generate the Stub source for a SIDL enum.  The source
   * file begins with a banner and include files followed by the stub
   * implementations for the methods.
   *
   * @param enm the <code>Enumeration</code> whose stub source is being
   * written .
   */
  private void generateEnum(Symbol enm) throws CodeGenerationException {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id        = enm.getSymbolID();
    String   source    = C.getStubFile(id);
    String   header    = C.getHeaderFile(id);

    d_writer.writeBanner(enm, source, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());

    d_writer.generateInclude(header, false);
    d_writer.generateInclude("sidl_int_IOR.h", true);
    d_writer.printlnUnformatted("#include <stddef.h>");
    d_writer.println();

    ArrayMethods ar = new ArrayMethods(id, true);
    ar.generateStub(d_writer);
  }

  /**
   * Generate the specified built-in stub.
   */
  private void generateBuiltinStub(SymbolID id, int stubType, int iorType,
                                   boolean doStatic) 
    throws CodeGenerationException
  {
    d_writer.writeComment(StubHeader.getBuiltinComment(stubType, doStatic), 
                          false);
    StubHeader.generateBuiltinSignature(d_writer, stubType, id, doStatic, "");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println(StubHeader.getDerefFunctionPtr(
      IOR.getBuiltinName(iorType, doStatic), doStatic) + "(");
    if (!doStatic) {
      d_writer.println(s_self + ",");
    }
    d_writer.println(StubHeader.getBuiltinArgList(stubType) + ");");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate method implementations for the specified assertion checking
   * method(s).
   */
  private void generateAssertionStubs(SymbolID id, boolean doStatic) 
    throws CodeGenerationException
  {
    generateBuiltinStub(id, StubHeader.SET_CHECKING, IOR.CHECKS, doStatic);
    generateBuiltinStub(id, StubHeader.DUMP_STATS, IOR.DUMP_STATS, doStatic);
  }

  /**
   * Generate method implementations for the methods in the interface
   * or class.  This method will also generate the constructor and cast
   * functions.
   */
  private void generateMethodStubs(Extendable ext)
    throws CodeGenerationException
  {
    /*
     * Generate the name for this entry point vector as well as a pointer
     * to "self" for this structure.  For classes, self will be the object
     * structure whereas for interfaces it is void*.
     */
    SymbolID id   = ext.getSymbolID();
    //String   self = C.getFullSelfDecl(id);
    String   self = s_self;

    /*
     * Output the normal and remote constructors if this extendable is a class.
     */
    if (!ext.isAbstract()) {
      d_writer.writeComment("Constructor function for the class.", false);
      d_writer.println(C.getSymbolName(id));
      d_writer.println(C.getFullMethodName(id, "_create") + "()");
      d_writer.println("{");
      d_writer.tab();
      d_writer.println("return (*(" + s_externals_func 
        + "()->createObject))();");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();

      d_writer.println("static " + C.getSymbolName(id) + " " 
        + C.getFullMethodName(id,"_remote") + "(const char* url, sidl_BaseInterface *_ex);");

      d_writer.writeComment("RMI constructor function for the class.", false);
      d_writer.println(C.getSymbolName(id));

      d_writer.println(C.getFullMethodName(id, "_createRemote") 
        + "(const char* url, sidl_BaseInterface *_ex)");

      d_writer.println("{");
      d_writer.tab();

      d_writer.println("return " + C.getFullMethodName(id,"_remote") 
        + "(url, _ex);");

      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * Output the assertion and/or interceptor methods as appropriate.
     */
    if (!ext.isInterface() && !ext.isAbstract()) {
      boolean hasStatic = ext.hasStaticMethod(true);
      if (IOR.supportAssertions(ext)) {
        if (hasStatic) {
          generateAssertionStubs(id, true);
        }
        generateAssertionStubs(id, false);
      }
      if (IOR.supportInterceptors(ext)) {
        if (hasStatic) {
          generateBuiltinStub(id, StubHeader.SET_INTERCEPTORS, IOR.INTERCEPTORS,
                              true);
        }
        generateBuiltinStub(id, StubHeader.SET_INTERCEPTORS, IOR.INTERCEPTORS, 
                            false);
      }
    }

    /*
     * Output each of the method implementations from the interface or class.
     */
    Collection methods = ext.getMethods(true);
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      generateMethodSignature(self, ext.isInterface(), id, method, false);
      d_writer.println("{");
      d_writer.tab();
      if(method.hasRarray()) {
        generateRarrays(method);
      }

      if (getReturnString(method.getReturnType()) != "void") {
        d_writer.print("return ");
      }
      d_writer.println(StubHeader.getDerefFunctionPtr(
        method.getLongMethodName(), method.isStatic()) + "(");
      d_writer.tab();
      C.generateArgumentList(d_writer, self, ext.isInterface(), id, method, 
                             false, false, false, true, false, false, true);
      d_writer.println(");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
    
    /* 
     * Output static exec functions
     */
    if(!ext.isInterface()) {
      Class cls = (Class) ext;
      if(cls.hasStaticMethod(true)){
        generateMethodExecs(cls);
      }
    }

    /*
     * Output the cast methods for the class or interface.
     */
    final Method castMethod = 
      IOR.getBuiltinMethod(IOR.CAST, id);
    d_writer.writeComment("Cast method for interface and class type "
      + "conversions.", false);
    d_writer.println(C.getSymbolName(id));
    d_writer.println(C.getFullMethodName(id, "_cast") + "(");
    d_writer.tab();
    d_writer.println("void* obj)");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();
    d_writer.println(C.getSymbolName(id) + " cast = " + C.NULL + ";");
    d_writer.println();
    d_writer.println("if (obj != " + C.NULL + ") {");
    d_writer.tab();
    d_writer.println("sidl_BaseInterface base = (sidl_BaseInterface) obj;");
    d_writer.println("cast = (" + C.getSymbolName(id) + ") (*base->" + s_epv 
      + "->" + IOR.getVectorEntry(castMethod.getLongMethodName()) + ")(");
    d_writer.tab();
    d_writer.println("base->d_object,");
    d_writer.println("\"" + id.getFullName() + "\");");
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
    d_writer.println("return cast;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeComment("String cast method for interface and class "
      + "type conversions.", false);
    d_writer.println("void*");
    d_writer.println(C.getFullMethodName(id, "_cast2") + "(");
    d_writer.tab();
    d_writer.println("void* obj,");
    d_writer.println("const char* type)");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("void* cast = " + C.NULL + ";");
    d_writer.println();
    d_writer.println("if (obj != " + C.NULL + ") {");
    d_writer.tab();
    d_writer.println("sidl_BaseInterface base = (sidl_BaseInterface) obj;");
    d_writer.println("cast = (*base->" + s_epv + "->"
      + IOR.getVectorEntry(castMethod.getLongMethodName()) 
      + ")(base->d_object, type);");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
    d_writer.println("return cast;");
    d_writer.backTab();
    d_writer.println("}");
    
    //Generate the __exec method!
    Method m_exec = C.getExecMethod();
    generateMethodSignature(self, ext.isInterface(), id, m_exec, false);
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("(*self->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->"
      + IOR.getVectorEntry(m_exec.getLongMethodName()) + ")(");
    C.generateArgumentList(d_writer, self, ext.isInterface(), id, m_exec, 
                           false, false, false, true, false, false, true);
    d_writer.println(");");
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    if(!ext.isInterface()) {
      Class cls = (Class) ext;
      if(cls.hasStaticMethod(true)){
        generateMainSExec(cls);
      }
    }
  }

  /**
   * Generate the method's signature.
   *
   * @param self the String representing the method's self argument variable.
   *
   * @param is_interface the boolean indicating whether working with a class
   *                     or an interface
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *   stub source is being written.
   *
   * @param method the <code>Method</code> whose signature is being output.
   *
   * @param is_super is a special parameter that is true ONLY if we are
   *   generating code in Impls for calling super methods.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateMethodSignature(String self, boolean is_interface,
                                       SymbolID id, Method method,
                                       boolean is_super)
    throws CodeGenerationException
  {
    d_writer.writeComment(method, false);
    if (is_super) {
      d_writer.print("static ");
    }
    d_writer.println(getReturnString(method.getReturnType()));
    if (is_super) {
      d_writer.print("super_" + method.getShortMethodName());
    } else {
      d_writer.print(C.getFullMethodName(id, method));
    }
    d_writer.println("(");
    d_writer.tab();
    C.generateArgumentList(d_writer, self, is_interface, id, method, true, 
                           true, false, true, false, true, true);
    d_writer.println(")");
    d_writer.backTab();
  }

  /*
   * Generate the _getExternals function that provides access to the IOR 
   * functions either through static linking or dynamic loading.
   */
  private void generateGetExternals(SymbolID id) {
    final String ext_name = IOR.getExternalName(id);
    d_writer.writeComment("Hold pointer to IOR functions.", false);
    d_writer.println("static const " + ext_name + " *" + s_externals 
      + " = " + C.NULL + ";");
    d_writer.writeComment("Lookup the symbol to get the IOR functions.",
                          false);
    d_writer.println("static const " + ext_name + "* _loadIOR(void)");
    
    d_writer.writeComment("Return pointer to internal IOR functions.", false);
    d_writer.println("{");
    d_writer.tab();
    if (BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.println(s_externals + " = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println(s_externals + " = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("sidl_DLL dll = sidl_DLL__create();");
      d_writer.println("const " + ext_name + "*(*dll_f)(void);");
      d_writer.writeCommentLine("check global namespace for symbol first");
      d_writer.println("if (dll && sidl_DLL_loadLibrary(dll, \"main:\", TRUE"
        + ", FALSE)) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("sidl_DLL_lookupSymbol(");
      d_writer.tab();
      d_writer.println("dll, \"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println(s_externals + " = (dll_f ? (*dll_f)() : " + C.NULL 
        + ");");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (dll) sidl_DLL_deleteRef(dll);");
      
      d_writer.println("if (!" + s_externals + ") {");
      d_writer.tab();
      d_writer.println("dll = sidl_Loader_findLibrary(\"" + id.getFullName() 
        + "\",");
      d_writer.tab();
      d_writer.println("\"ior/impl\", sidl_Scope_SCLSCOPE,");
      d_writer.println("sidl_Resolve_SCLRESOLVE);");
      d_writer.backTab();
      d_writer.println("if (dll) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("sidl_DLL_lookupSymbol(");
      d_writer.tab();
      d_writer.println("dll, \"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println(s_externals + " = (dll_f ? (*dll_f)() : " + C.NULL 
        + ");");
      d_writer.println("sidl_DLL_deleteRef(dll);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (!" + s_externals + ") {");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println( "fputs(\"Babel: unable to load the implementation for "
        + id.getFullName() + "; please set sidl_DLL_PATH\\n\", stderr);");
      d_writer.enableLineBreak();
      d_writer.println("exit(-1);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.println("return " + s_externals + ";");
    d_writer.backTab();
    d_writer.println("}");

    d_writer.println();
    d_writer.printlnUnformatted("#define " + s_externals_func + "() (" 
      + s_externals + " ? " + s_externals + " : _loadIOR())");

    d_writer.println();
  }

  /**
   * Return the pointer that provides access to the static EPV.
   */
  private void generateGetStaticEPV(SymbolID id) {
    String sepv_name = IOR.getSEPVName(id);
    d_writer.writeComment("Hold pointer to static entry point vector",false);
    d_writer.println("static const " + sepv_name + " *_sepv = " + C.NULL + ";");
    d_writer.writeComment("Return pointer to static functions.",false);
    d_writer.printlnUnformatted("#define " + StubHeader.s_sepv_func 
      + " (_sepv ? _sepv : (_sepv = (*(" + s_externals_func 
      + "()->getStaticEPV))()))");
    d_writer.println();
  }

  /**
   *  This generates the temporary variables for bundling up rarrays into
   *  sidl arrays.
   */
  private void generateRarrays(Method m) throws CodeGenerationException {
    Collection args = m.getArgumentList();
    ArrayList rarrays = new ArrayList();

    //Get a list of rarrays going
    for(Iterator a = args.iterator(); a.hasNext();) {
      Argument arg = (Argument)a.next();
      if(arg.getType().isRarray())
        rarrays.add(arg);
    }

    for(Iterator r = rarrays.iterator(); r.hasNext();) {
      Argument rarray = (Argument) r.next();
      String r_name = rarray.getFormalName();
      Type type = rarray.getType();
      int dim = type.getArrayDimension();
      d_writer.println("int32_t " + r_name + "_lower[" + dim + "], " + r_name 
        + "_upper[" + dim + "], " + r_name + "_stride[" + dim + "]; ");
      String sidl_array_name = getReturnString(type);
      d_writer.println(sidl_array_name.substring(0,sidl_array_name.length()-1) 
        + " " + r_name + "_real;");
      d_writer.println(sidl_array_name + r_name + C.RAW_ARRAY_EXT + " = &" 
        + r_name + "_real;");
    }
    
    for(Iterator r = rarrays.iterator(); r.hasNext();) {
      Argument rarray = (Argument)r.next();
      String r_name = rarray.getFormalName();
      Type type = rarray.getType();
      Collection indices = type.getArrayIndices();
      String init_func_name = IOR.getArrayNameForFunctions(
                                  rarray.getType().getArrayType().getType()) 
                                 + "_init";
      int x = 0;
      for(Iterator i = indices.iterator(); i.hasNext();++x) {
        d_writer.println(r_name + "_upper[" + x + "] = " + (String) i.next() 
          + "-1;");
      }
      d_writer.println(init_func_name + "(" + r_name + ", " + r_name 
        + C.RAW_ARRAY_EXT + ", " + type.getArrayDimension() + ", " + r_name 
        + "_lower, " + r_name + "_upper, " + r_name + "_stride);");
    }
  }

  /**
   * @param cls
   */
  private void generateMainSExec(Class cls) {
    SymbolID id = cls.getSymbolID();
    String my_symbolName = IOR.getSymbolName(id);
    d_writer.println("struct " + my_symbolName + "__method {");
    d_writer.tab();
    d_writer.println("const char *d_name;");
    d_writer.println("void (*d_func)(struct sidl_io_Deserializer__object *,");
    d_writer.tab();
    d_writer.println("struct sidl_io_Serializer__object *);");
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println();
    d_writer.println("void");
    d_writer.println(my_symbolName + "__sexec(");
    d_writer.println("        const char* methodName,");
    d_writer.println("        struct sidl_io_Deserializer__object* inArgs,");
    d_writer.println("        struct sidl_io_Serializer__object* outArgs ) { ");
    d_writer.tab();
    ArrayList methods = new ArrayList(cls.getStaticMethods(true));
    Collections.sort(methods, new IOR.CompareMethods());
    d_writer.println("static const struct " + my_symbolName 
      + "__method  s_methods[] = {");
    d_writer.tab();
    for(Iterator i = methods.iterator(); i.hasNext();){
      Method m = (Method)i.next();
      d_writer.println("{ \"" + m.getLongMethodName() + "\", " + my_symbolName 
        + '_' + m.getLongMethodName() + "__sexec }" + (i.hasNext() ? "," : ""));
    }
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println("int i, cmp, l = 0;");
    d_writer.println("int u = sizeof(s_methods)/sizeof(struct " + my_symbolName
      + "__method);");
    d_writer.println("if (methodName) {");
    d_writer.tab();
    d_writer.writeCommentLine("Use binary search to locate method");
    d_writer.println("while (l < u) {");
    d_writer.tab();
    d_writer.println("i = (l + u) >> 1;");
    d_writer.println("if (!(cmp=strcmp(methodName, s_methods[i].d_name))) {");
    d_writer.tab();
    d_writer.println("(s_methods[i].d_func)(inArgs, outArgs);");
    d_writer.println("return;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("else if (cmp < 0) u = i;");
    d_writer.println("else l = i + 1;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.writeCommentLine("TODO: add code for method not found");
    d_writer.backTab();
    d_writer.println("}");
  }

  private void generateMethodExecs(Class cls) throws CodeGenerationException {
    SymbolID id = cls.getSymbolID();
    String my_symbolName = IOR.getSymbolName(id);
    Collection c = cls.getStaticMethods(true);
    String self = C.getObjectName(id) + " self";
    for( Iterator i=c.iterator(); i.hasNext(); ) { 
      Method m = (Method) i.next();
      boolean hasThrows = !m.getThrows().isEmpty();
      int numSerialized;  // number of args + return value + exceptions
      String name = m.getLongMethodName();
      Type returnType = m.getReturnType();
      d_writer.println("void");
      d_writer.println(my_symbolName + "_" + name + "__sexec(");
      d_writer.println("        struct sidl_io_Deserializer__object* inArgs,");
      d_writer.println("        struct sidl_io_Serializer__object* outArgs) {");
      d_writer.tab();
      d_writer.writeCommentLine("stack space for arguments");
      List args = m.getArgumentList();
      numSerialized = args.size();
      for (Iterator a = args.iterator(); a.hasNext(); ) { 
        Argument arg = (Argument) a.next();
        if (arg.getType().getType() == Type.STRING && 
            arg.getMode() == Argument.IN ) { /* no const in this case */
          if(arg.getType().isRarray()) {
            d_writer.println(getReturnString(arg.getType()) + " " 
              + arg.getFormalName() + C.RAW_ARRAY_EXT + ";");
          } else {
            d_writer.println(getReturnString(arg.getType()) + " " 
              + arg.getFormalName() + ";");
          }
        } else { 
          if(arg.getType().isRarray()) {
            d_writer.println(getArgumentWithFormal(arg) + C.RAW_ARRAY_EXT 
              + ";");
          } else {
            d_writer.println(getArgumentWithFormal(arg) + ";");
          }
        }
      }
      if ( returnType.getType() != Type.VOID ) { 
        ++numSerialized;
        d_writer.println( IOR.getReturnString(returnType, false, false) 
          + " _retval;");
      }
      //TODO: Get rid of this when we have object and array serialization
      boolean allObj = true;
      boolean hasReturn = returnType.getType() != Type.VOID;
      boolean hasArgs = numSerialized > 0;
      if(returnType.getType() != Type.VOID && returnType.getType() < Type.CLASS) {
        allObj = false;
      }
      if(allObj) {
        for (Iterator a = args.iterator(); a.hasNext(); ) { 
          Argument arg = (Argument) a.next();
          if(arg.getType().getType() < Type.CLASS) {
            allObj = false;
          }
        }
      }
      if(((hasReturn || hasArgs) && !allObj) || hasThrows) {  //We use ex2 when unserializing as well
        d_writer.println("sidl_BaseInterface _ex   = " + C.NULL + ";");
        d_writer.println("sidl_BaseInterface *_ex2 = &_ex;");
      }
      
      /*
      String excRes = null;
      if (hasThrows) {
        ++numSerialized;
        d_writer.println("sidl_BaseInterface _ex   = " + C.NULL + ";");
        excRes = "&_ex";
      } else {
        excRes = C.NULL;
      }
      if (numSerialized > 0) {
        d_writer.println("sidl_BaseInterface *_ex2 = " + excRes + ";");
      }
      */
      d_writer.println();
      d_writer.writeCommentLine("unpack in and inout argments");
      for (Iterator a = args.iterator(); a.hasNext(); ) { 
        Argument arg = (Argument) a.next();
        if ( arg.getType().isArray() || arg.getType().isSymbol() ||
             ( arg.getType().getType() == Type.OPAQUE ) ) { 
          //FIXME: implement complicated stuff later
          continue;
        }
        if ( arg.getMode() == Argument.IN || arg.getMode() == Argument.INOUT ) {
          d_writer.println();
          d_writer.println(RMI.unpackType("sidl_io_Deserializer", "inArgs",
                                          arg.getType(), arg.getFormalName(), 
                                          "&" + arg.getFormalName()));
        }
      }
      d_writer.println();
	    
      d_writer.writeCommentLine("make the call");
      if ( returnType.getType() != Type.VOID ) { 
        d_writer.print( "_retval = ");
      }
     
      d_writer.print("(_getSEPV()->" + IOR.getVectorEntry(m.getLongMethodName())
        + ")");
      
      d_writer.println("(");
      d_writer.tab();
      C.generateArguments(d_writer, self, m.getArgumentList(), m.isStatic(), 
                          hasThrows ? "_ex2" : "", null, false, false, true);
      d_writer.println(");");
      d_writer.backTab();
      d_writer.println();
	    
      if (hasThrows) {
        d_writer.writeCommentLine("check if exception thrown");
        d_writer.writeCommentLine("FIXME");
        d_writer.println();
      }
	    
      d_writer.writeCommentLine("pack return value");
      if (returnType.isArray() || returnType.isSymbol() ||
          (returnType.getType() == Type.OPAQUE) ) { 
        //FIXME:
        ;
      } else if ( returnType.getType() != Type.VOID ) { 
        d_writer.println(RMI.packType("sidl_io_Serializer","outArgs",returnType,
                                      "_retval", "_retval"));
        d_writer.println();
      }
      d_writer.writeCommentLine("pack out and inout argments");
      for (Iterator a = args.iterator(); a.hasNext(); ) { 
        Argument arg = (Argument) a.next();
        if ( arg.getType().isArray() || arg.getType().isSymbol() ||
             ( arg.getType().getType() == Type.OPAQUE ) ) { 
          //FIXME: implement complicated stuff later
          continue;
        }
        if (  arg.getMode() == Argument.OUT 
           || arg.getMode() == Argument.INOUT ) { 
          d_writer.println(RMI.packType("sidl_io_Serializer","outArgs",
                                        arg.getType(), arg.getFormalName(), "*"
                                        + arg.getFormalName()));
        }
      }
      d_writer.println();
	    
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("");
    }
  }

   /**
    * Generate a Stub argument string with the formal argument name.
    *
    * @param arg    the <code>Argument</code> whose string is being built.
    */
   private static String getArgumentWithFormal(Argument arg)
     throws CodeGenerationException
   {
     return IOR.getArgumentWithFormal(arg, false, false, false);
   }
}
