/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.util;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;

/**
 * A set of utilities to inspect current stack frame.
 *
 * @author <a href="mailto:sylvain@apache.org">Sylvain Wallez</a>
 * @author <a href="mailto:stuart.roebuck@adolos.com">Stuart Roebuck</a>
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version CVS $Revision: 1.4 $ $Date: 2001/12/11 09:53:37 $
 */
public final class StackIntrospector
{
    /**
     * Hack to get the call stack as an array of classes. The
     * SecurityManager class provides it as a protected method, so
     * change it to public through a new method !
     */
    private final static class CallStack
        extends SecurityManager
    {
        /**
         * Returns the current execution stack as an array of classes.
         * The length of the array is the number of methods on the execution
         * stack. The element at index 0 is the class of the currently executing
         * method, the element at index 1 is the class of that method's caller,
         * and so on.
         */
        public Class[] get()
        {
            return getClassContext();
        }
    }

    ///Method to cache CallStack hack as needed
    private static CallStack    c_callStack;

    /**
     * Private constructor to block instantiation.
     *
     */
    private StackIntrospector()
    {
    }

    /**
     * Hack a CallStack object to get access to Classes in callStack.
     *
     * @return the Class array for call stack
     * @exception SecurityException if an existing SecurityManager disallows construction 
     *            of another SecurityManager
     */
    private synchronized static Class[] getCallStackAsClassArray()
        throws SecurityException
    {
        //TODO: Check permission even if not created...
        if( null == c_callStack )
        {
            //Lazily create CallStack accessor as appropriate
            c_callStack = new CallStack();
        }
        
        return c_callStack.get();
    }

    /**
     * Find the caller of the passed in Class.
     * May return null if caller not found on execution stack
     *
     * @param clazz the Class to search for on stack to find caller of
     * @return the Class of object that called parrameter class
     * @exception SecurityException if an existing SecurityManager disallows construction 
     *            of another SecurityManager and thus blocks method results
     */
    public final static Class getCallerClass( final int index )
        throws SecurityException
    {
        final Class[] stack = getCallStackAsClassArray();
        if( index < stack.length ) return stack[ index ]; 
        else return null;
    }

    /**
     * Find the caller of the passed in Class.
     * May return null if caller not found on execution stack
     *
     * @param clazz the Class to search for on stack to find caller of
     * @return the Class of object that called parrameter class
     * @exception SecurityException if an existing SecurityManager disallows construction 
     *            of another SecurityManager and thus blocks method results
     */
    public final static Class getCallerClass( final Class clazz )
        throws SecurityException
    {
        final Class[] stack = getCallStackAsClassArray();

        // Traverse the call stack in reverse order until we find clazz
        for( int i = stack.length - 1; i >= 0; i-- )
        {
            if( clazz.isAssignableFrom( stack[ i ] ) )
            {
                // Found : the caller is the previous stack element
                return stack[ i + 1 ];
            }
        }

        //Unable to locate class in call stack
        return null;
    }

    /**
     * Get the name of the method is at specified index in call stack.
     *
     * @return The method path name in the form
     *         "the.package.MyClass.method(MyClass:121)"
     */
    public final static String getCallerMethod( final int index ) 
    {
        final String[] callStack = getCallStackAsStringArray();

        //increment callStack index as added method call with above 
        int actualIndex = index + 1;
        if( actualIndex < callStack.length ) return callStack[ actualIndex ];
        else return null;
    }

    /**
     * Get the name of the method that called specified class.
     *
     * @return The method path name in the form "the.package.MyClass.method(MyClass:121)"
     *         or null if unable to determine caller.
     */
    public final static String getCallerMethod( final Class clazz ) 
    {
        final String[] callStack = getCallStackAsStringArray();
        final int index = getCallerIndex( clazz.getName(), callStack );

        if( -1 == index ) return null;
        else
        {
            return callStack[ index ];
        }
    }

    /**
     * Return the call stack that called specified Class as an array of Strings.
     * The maximum size of call-stack is specified by count parameter.
     *
     * <p>This can be useful for debugging code to determine where calls to a
     * method are coming from.</p>
     *
     * @param clazz the last class on the stack you are <i>not</i> interested in!
     * @param count the number of stack entries to return.
     *
     * @return An array of method names in the form 
     *         "the.package.MyClass.method(MyClass:121)"
     */
    public final static String[] getCallerStack( final Class clazz, int count )
    {
        final String[] callStack = getCallStackAsStringArray();
        final int start = getCallerIndex( clazz.getName(), callStack );
        if( -1 == start ) return null;

        final int size = Math.min( count, callStack.length - start );

        final String[] result = new String[ size ];
        for( int i = 0; i < size; i++ )
        {
            result[ i ] = callStack[ start + i ]; 
        }

        return result;
    }

    /**
     * Return the current call stack as a String array.
     *
     * <p>This can be useful for debugging code to determine where calls to a
     * method are coming from.</p>
     *
     * @return The array of strings containing methods in the form
     *         "the.package.MyClass.method(MyClass:121)"
     */
    public final static String[] getCallStackAsStringArray()
    {
        //Extract stack into a StringBuffer
        final StringWriter sw = new StringWriter();
        final Throwable throwable = new Throwable();
        throwable.printStackTrace( new PrintWriter( sw, true ) );
        final StringBuffer buffer = sw.getBuffer();

        //Resulting stack
        final ArrayList stack = new ArrayList();

        //Cache vars used in loop
        final StringBuffer line = new StringBuffer();
        final int length = buffer.length();

        //setup state
        boolean found = false;
        int state = 0;

        //parse line
        for( int i = 0; i < length; i++ )
        {
            final char ch = buffer.charAt( i );

            switch( state )
            {
            case 0:
                //Strip the first line from input
                if( '\n' == ch ) state = 1;
                break;

            case 1:
                //strip 't' from 'at'
                if( 't' == ch ) state = 2;
                break;

            case 2:
                //Strip space after 'at'
                line.setLength( 0 );
                state = 3;
                break;

            case 3:
                //accumulate all characters to end of line
                if( '\n' != ch ) line.append( ch );
                else
                {
                    //At this stage you have the line that looks like
                    //com.biz.SomeClass.someMethod(SomeClass.java:22)
                    final String method = line.toString();
                    stack.add( method );

                    //start parsing from start of line again
                    state = 1;
                }
            }
        }

        return (String[])stack.toArray( new String[ 0 ] );
    }

    /**
     * Return the index into specified stack, that
     * has a caller that starts with prefix.
     *
     * @param prefix the prefix.
     * @param callStack the call stack
     * @return index into array
     */
    private static int getCallerIndex( final String prefix, final String[] callStack )
    {
        boolean found = false;
        for( int i = 0; i < callStack.length; i++ )
        {
            ///Determine if line is a match for class
            final boolean match = callStack[ i ].startsWith( prefix );
            if( !match ) continue;

            //If this is the first time we cound class then
            //set found to true and look for caller into class
            if( !found ) found = true;
            else
            {
                //We have now located caller of Clazz
                return i;
            }
        }

        return -1;
    }
}
