/*
 * ViewCamera - VRML viewpoint control
 *
 * created: mpichler, 19970124
 *
 * changed: mpichler, 19970728
 *
 * $Id: ViewCamera.java,v 1.9 1997/08/12 09:12:15 mpichler Exp $
 */


package iicm.vrml.vrwave;

import iicm.ge3d.GE3D;
import iicm.utils3d.Camera;
import iicm.utils3d.Quaternion;
import iicm.utils3d.Ray;
import iicm.utils3d.Vec3f;
import iicm.vrml.pw.Viewpoint;
import java.io.PrintStream;


/**
 * ViewCamera - VRML viewpoint (Camera)
 * Copyright (c) 1997 IICM
 *
 * @author Michael Pichler
 * @version 1.0, changed: 28 Feb 97
 */


public class ViewCamera extends Camera
{
  // access to Camera position_, orientation_
  private Viewpoint vpoint_;  // currently bound Viewpoint (or null)
  // will be replaced by a stack of viewpoints (TODO)
  private float viewangle_ = 0.785398f;  // set in setCamera

  // TODO: care for clipping planes
  private float nearclip_ = 0.1f;  // hither
  private float farclip_ = 100.0f;  // yon

  // this Camera stores the movement *relative* to the currenlty bound
  // Viewpoint, because the Viewpoint itself may be animated by a
  // Script while the relative (user) movement remains unchanged

  public ViewCamera ()
  {
    position_[2] = 10.0f;  // VRML 2.0 default camera position
  }

  /**
   * clear Viewpoint stack.
   * Does not change the relative view
   * @see #reset
   */

  public void clear ()
  {
    vpoint_ = null;
    // will empty viewpoint list here
    // caller should do a reset () afterwards
  }

  /**
   * reset to initial (relative) view
   */

  public void reset ()
  {
    super.reset ();
    if (vpoint_ == null)  // default camera position
      position_[2] = 10.0f;
    // else no relative camera translation
  }

  // should overload level and untilt to have an *overall* level/untilt view (TODO)

  /** get near clipping plane (hither) */
  public float getNearClip ()  { return nearclip_; }

  /** get far clipping plane (yon) */
  public float getFarClip ()  { return farclip_; }

  public void setNearClip (float hither)  { nearclip_ = hither; }

  public void setFarClip (float yon)  { farclip_ = yon; }


  /**
   * set camera via GE3D
   */

  public void setCamera (float winaspect)
  {
    // set this camera (relative to Viewpoint)

    if (vpoint_ != null)
      viewangle_ = vpoint_.fieldOfView.getValue ();
    else
      viewangle_ = 0.785398f;  // default field of view (pi/4)

    if (winaspect > 1.0f)  // VRML 2.0 spec: viewing angle to be measured in wider dimension
      viewangle_ /= winaspect;
    // TODO: clipping planes

    super.setCamera (viewangle_, winaspect, nearclip_, farclip_);

    // apply inverse transformation of currently bound viewpoint
    if (vpoint_ != null)
    {
      float[] axisangle = vpoint_.orientation.getValue ();
      GE3D.rotatef3f (axisangle, - axisangle[3]);  // inverse rotation

      float[] position_ = vpoint_.position.getValue ();
      GE3D.translatefff (- position_[0], - position_[1], - position_[2]);  // inverse translation
    }
  } // setCamera

  /**
   * set the initial viewpoint.
   * has no effect when there is already a viewpoint bound
   */

  public void initialViewpoint (Viewpoint vp)
  {
    if (vpoint_ != null)
      return;

    vpoint_ = vp;
    position_[2] = 0.0f;  // no relative position
    // TODO: send isBound TRUE event (after build has been completed)
  } // initialViewpoint

  /**
   * transform point into Viewpoint coordinate system.
   * i.e. into the coordinate system the Camera is defined in.
   * @return unchanged point if there is no Viewpoint bound
   */

  protected float[] transformPointVCS (float[/*3*/] point)
  {
    if (vpoint_ == null)
      return point;

    // apply Viewpoint's inverse transformation to point
    // recall that OpenGL applies transformations in reverse order
    float[] pos = new float[3];
    float[] vpos = vpoint_.position.getValue ();
    pos[0] = point[0] - vpos[0];  // inverse translation
    pos[1] = point[1] - vpos[1];
    pos[2] = point[2] - vpos[2];

    float[] axisangle = vpoint_.orientation.getValue ();
    Quaternion rot = new Quaternion (axisangle, - axisangle[3]);
    pos = rot.rotateVector (pos);  // inverse rotation
    // calculating the rotation matrix by hand or invoking GE3D
    // would be about the same complexity as the round-trip over the
    // quaternion

    // System.out.println ("old point: " + Vec3f.print (point) +
    //   ", new point: " + Vec3f.print (pos));
    return pos;

  } // transformPointVCS

  /**
   * transform vector into Viewpoint coordinate system.
   * like transformPointVCS, but consider orientation only.
   */

  protected float[] transformVectorVCS (float[/*3*/] vector)
  {
    if (vpoint_ == null)
      return vector;

    float[] axisangle = vpoint_.orientation.getValue ();
    Quaternion rot = new Quaternion (axisangle, - axisangle[3]);  // inverse rotation
    return rot.rotateVector (vector);  // input vector not changed
  }

  /**
   * rotate horicontally/vertically.
   * center transformed by current Viewpoint
   */

  public void rotateXYcenter (float l2r, float t2b, float[/*3*/] center)
  {
    super.rotateXYcenter (l2r, t2b, transformPointVCS (center));
  }

  /**
   * approach target position by a fraction of k
   */

  public void approachPosition (float[/*3*/] poi, float ktran, float hither)
  {
    super.approachPosition (transformPointVCS (poi), ktran, hither);
  }

  /**
   * approach target normal vector by a faction of k
   */

  public void approachNormal (float[/*3*/] poi, float[/*3*/] normal, float krot)
  {
    super.approachNormal (transformPointVCS (poi), transformVectorVCS (normal), krot);
  }

  /**
   * get a viewing ray (for picking).
   */

  public Ray viewingRay (float fx, float fy, float winaspect)
  {
    // ray in camera coordinate system (parameters of last setCamera)
    Ray ray = super.viewingRay (fx, fy, viewangle_, winaspect, nearclip_, farclip_);

    if (vpoint_ != null)
    {
      // transform ray into viewpoint coordinate system
      // by applying the inverse of the object transformation,
      // i.e. the non inverted camera transformation
      // rotation
      float[] axisangle = vpoint_.orientation.getValue ();
      Quaternion rot = new Quaternion (axisangle, axisangle[3]);
      ray.start_ = new Vec3f (rot.rotateVector (ray.start_.value_));
      ray.direction_ = new Vec3f (rot.rotateVector (ray.direction_.value_));
      // translation (start point only)
      ray.start_.increase (vpoint_.position.getValue ());
      // this is the ray in the viewpoint's world coordinate system;
      // the hit normal does not have to be changed by the orientation
    }

    return ray;
  } // viewingRay

  /**
   * print camera values
   */

  public void printValues (PrintStream os)
  {
    if (vpoint_ != null)
    {
      os.println ("");
      // total position: relative translation rotated by viewpoint orientation + viewpoint translation
      float[] tv = vpoint_.position.getValue ();
      float[] vrotaa = vpoint_.orientation.getValue ();
      Quaternion vrot = new Quaternion (vrotaa, vrotaa[3]);
      float[] ttran = vrot.rotateVector (position_);
      ttran[0] += tv[0];
      ttran[1] += tv[1];
      ttran[2] += tv[2];
      os.println ("position " + ttran[0] + " " + ttran[1] + " " + ttran[2]);

      // total orientation: relative rotation before viewpoint rotation
      Quaternion trot = Quaternion.product (vrot, orientation_);
      float[] traa = trot.getAxisAngle ();
      os.println ("orientation " + traa[0] + " " + traa[1] + " " + traa[2] + " " + traa[3]);
      os.println ("");
    }
    else
      super.printValues (os);
  } // printValues

} // ViewCamera
