/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  February 1996 - August 98                */
/*		      RFC Synthesis                                      */
/*                                                                       */
/*=======================================================================*/

#include "tilt.h"
#include "EST_unix.h"
#include "EST_tilt.h"
#include "EST_Track.h"
#include "EST_error.h"

// find event portions of fz in contour, cut out, and send one by one
// to individual labeller.
int fill_values(EST_Track &fz, float amp, float dur, float
		start_f0, float start_pos, float f_shift, EST_String type)
{
    float t;
    // this ensures rounding errors don't multiply
    int j = (int) rint(start_pos / f_shift); 

/*    cout << "amp: " << amp << endl;
    cout << "dur: " << dur << endl;
    cout << "start_f0: " << start_f0 << endl;
    cout << "start_pos: " << start_pos << endl << endl;
*/
    
    for (t = 0.0; t < (dur + (f_shift /2.0)); t += f_shift, ++j)
    {
	fz.a(j) = unit_curve(type, amp, dur, t) + start_f0;
	fz.set_value(j);
//	cout << j << " : " << fz.t(j) << " a= " << fz.a(j) << endl;
    }

    return 0;
}

void fill_connect(EST_Track &fz, float start_f0, float start_pos,
		 float end_f0, float end_pos,
		 float f_shift)
{
    // this ensures rounding errors don't multiply
    int j = (int) rint(start_pos / f_shift); 

    float m = (end_f0 - start_f0) / (end_pos - start_pos);

    int pos = fz.index(start_pos);


/*    cout << "start_f0: " << start_f0 << endl;
    cout << "start_pos: " << start_pos << endl;
    cout << "end_f0: " << end_f0 << endl;
    cout << "end_pos: " << end_pos << endl;
    cout << "m: " << m << endl;
    cout << "pos: " << pos << endl << endl;
*/

    for (j = pos; j < (fz.index(end_pos) + 1); ++j)
    {
	fz.a(j) = (m * (float) (j -pos) * f_shift) + start_f0;
	fz.set_value(j);
    }
}

#if 0
void start_f0_pos(EST_Item *e, const EST_String &type, float &start_f0, 
		 float &start_pos)
{
    if (type == "RISE")
    {
	start_f0 = e->fF("ev:f0",1);
	start_pos = e->fF("position",1) - e->fF("rfc:rise_dur",1);
    }
    else
    {
	start_f0 = e->fF("ev:f0",1) + e->fF("rfc:rise_amp",1);
	start_pos = e->fF("position", 1);
    }
}
#endif

static float find_start_pos(EST_Item *e, const EST_String &type)
{
    if (type == "RISE")
	return e->fF("position",1) - e->fF("rfc:rise_dur",1);
    else
	return e->fF("position", 1);
}

static float find_start_f0(EST_Item *e, const EST_String &type)
{
    if (type == "RISE")
	return e->fF("ev:f0",1);
    else
	return e->fF("ev:f0",1) + e->fF("rfc:rise_amp",1);
}

float rfc_dur(EST_Item *e)
{
    return e->fF("rfc:rise_dur") + e->fF("rfc:fall_dur");
}

float rfc_amp(EST_Item *e)
{
    return e->fF("rfc:rise_amp") + e->fF("rfc:fall_amp");
}

int rfc_synthesis(EST_Track &fz, EST_Relation &ev, float f_shift, int no_conn)
{
    EST_Item *e,*nn;
    float start_pos=0, start_f0=0;
    EST_String type;
    (void)no_conn;
    
    float last_time = ev.tail()->fF("position") + rfc_dur(ev.tail());
    int n = (int)(2 + (last_time / f_shift));
    fz.resize(n, 1);
    fz.fill(0.0);
    fz.fill_time(f_shift);
    fz.set_contour_type(ct_f0);
    
    // set default to be break (silence)
    for (int i = 0; i < fz.num_frames(); ++i)
	fz.set_break(i);
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	//	cout << "\ntype: " << e->fS("rfc:type") << endl;
	//	cout << "\ntype: " << *e << endl;
	if (e->f("rfc:type",1) == "RISEFALL")
	{
	    start_f0 = find_start_f0(e,"RISE");
	    start_pos = find_start_pos(e,"RISE");
	    fill_values(fz, e->fF("rfc:rise_amp",1), e->fF("rfc:rise_dur",1),
			start_f0, start_pos,
			f_shift, "RISE");
	    
	    fill_values(fz, e->fF("rfc:fall_amp",1), e->fF("rfc:fall_dur",1),
			find_start_f0(e,"FALL"),
			find_start_pos(e,"FALL"),
			f_shift, "FALL");
	    
	}
	else if (e->f("rfc:type",1) == "RISE")
	{
	    start_f0 = find_start_f0(e,"RISE");
	    start_pos = find_start_pos(e,"RISE");
	    fill_values(fz, e->fF("rfc:rise_amp",1), e->fF("rfc:rise_dur",1), 
			start_f0, start_pos,
			f_shift, "RISE");
	}
	else if (e->f("rfc:type",1) =="FALL")
	{
	    fill_values(fz, e->fF("rfc:fall_amp",1), e->fF("rfc:fall_dur",1),
			e->f("ev:f0",1), e->f("position",1), 
			f_shift, "FALL");
	}
	else 
	{
	    EST_Item *nn,*pp;

	    if (e->f("name",1) == "phrase_end")
	    {
		//		cout << "phrase end:\n" << *e << endl;
		if (e->f_present("ev:f0"))
		{
		    pp = prev(e);
		    fill_connect(fz, start_f0 + rfc_amp(pp), start_pos
				 + rfc_dur(pp), e->fF("ev:f0",1),
				 e->fF("position",1), f_shift);
		}
	    }
	    else if (e->f("name",1) == "phrase_start")
	    {
		//		cout << "phrase start:\n" << *e << endl;
		if ((nn = next(e)) == 0)
		    EST_error("phrase start command occurs as last item "
			      "in rfc synthesis\n");
		
		start_f0 = find_start_f0(nn,"RISE");
		start_pos = find_start_pos(nn,"RISE");
		fill_connect(fz, e->fF("ev:f0",1), e->fF("position",1),
			     start_f0,start_pos,
			     f_shift);
		
	    }
	    else if (e->f("name") == "pause")
	    {}
	    else
		EST_error("Unable to synthesis intonation element %s\n", 
			  (const char *)(e->fS("name")));
	    continue;
	}
	if (((nn = next(e)) != 0) && (event_item(*nn)))
	{
	    float f0 = start_f0+rfc_amp(e);
	    float pos = start_pos + rfc_dur(e);
	    float end_f0 = find_start_f0(nn,"RISE");
	    float end_pos = find_start_pos(nn,"RISE");
	    fill_connect(fz, f0, pos, end_f0, end_pos,f_shift);
	}
    }
    return 0;
}

void remove_tilt_features(EST_Relation &ev);
void remove_rfc_features(EST_Relation &ev);

int tilt_synthesis(EST_Track &fz, EST_Relation &ev, float f_shift, int no_conn)
{
    tilt_to_rfc(ev);
    
    //    ev.remove_item_feature("tilt:amp");
    //    ev.remove_item_feature("tilt:dur");
    //    ev.remove_item_feature("tilt:tilt");
    
    //    ev.save("test.rfc");
    //    cout << ev;
    rfc_synthesis(fz, ev, f_shift, no_conn);
    
    remove_rfc_features(ev);
    return 0;
}
