/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black                                    */
/*             Date   :  April 1998                                      */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*  A quick database structure                                           */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <math.h>
#include "festival.h"
#include "EST_FileType.h"
#include "clunits.h"

static void cl_load_catalogue(EST_String &indexfile);

CLDB cldb;

LISP cl_load_db(LISP params)
{
    EST_String indexfile;

    cldb.params = params;
    
    indexfile = EST_String("") +
	get_param_str("db_dir",params,"./")+
	get_param_str("catalogue_dir",params,"./")+
	get_param_str("index_name",params,"catalogue")+
	    ".catalogue";
    
    cl_load_catalogue(indexfile);

    return NIL;
}

static void cl_load_catalogue(EST_String &indexfile)
{
    EST_TokenStream ts;
    EST_EstFileType t;
    EST_Option hinfo;
    EST_String v;
    bool ascii;
    EST_read_status r;
    
    if (((indexfile == "-") ? ts.open(cin) : ts.open(indexfile)) != 0)
    {
	cerr << "CLUNITS: Can't open catalogue file " << indexfile << endl;
	festival_error();
    }

    if (((r = read_est_header(ts, hinfo, ascii, t)) != format_ok) ||
	(t != est_file_index))
    {
	cerr << "CLUNITS: " << indexfile << " is not an indexfile" << endl;
	festival_error();
    }

    EST_Item *ls = 0;
    while(!ts.eof());
    {
	EST_Item *s = new EST_Item;
	s->set_name(ts.get().string());
	s->fset("fileid",ts.get().string());
	s->fset("start",atof(ts.get().string()));
	s->fset("mid",atof(ts.get().string()));
	s->fset("end",atof(ts.get().string()));

	if ((ls != 0) &&
	    (ls->f("fileid") == s->f("fileid")) &&
	    (ls->f("end") == s->f("start")))
	    s->fset("prev_unit",ls->name());
	cldb.index.add(s->name(),s);
    }
}

void CLDB::load_coefs_sig(EST_Item *unit)
{
    // Load in the coefficients and signal for this unit.
    EST_String fileid = unit->f("fileid");
    EST_Item *fileitem = get_fileitem(fileid);

    if (fileitem == 0)
    {   // even the file isn't here
	EST_Track *track = new EST_Track;
	EST_String coef_filename = 
	    EST_String("") +
		get_param_str("db_dir",cldb.params,"./") +
		    get_param_str("pm_coeffs_dir",cldb.params,"pm/") +
			fileid+
			    get_param_str("pm_coeffs_ext",cldb.params,".pm");
	if (track->load(coef_filename) != format_ok)
	{
	    delete track;
	    cerr << "CLUNITS: failed to load coeffs file " << 
		coef_filename << endl;
	    festival_error();
	}
	fileitem = new EST_Item;
	fileitem->fset("coeffs",track,gc_track);
	
	EST_Wave *sig = new EST_Wave;
	EST_String sig_filename = 
	    EST_String("") +
		get_param_str("db_dir",cldb.params,"./") +
		    get_param_str("sig_dir",cldb.params,"wav/") +
			fileid+
			    get_param_str("sig_ext",cldb.params,".wav");
	if (sig->load(coef_filename) != format_ok)
	{
	    delete sig;
	    delete fileitem;
	    cerr << "CLUNITS: failed to load signal file " << 
		sig_filename << endl;
	    festival_error();
	} 
	fileitem->fset("sig",sig,gc_wave);
	fileindex.add(fileid,fileitem);
    }
    EST_Track *coeffs = (EST_Track *)fileitem->f("coefs").ptr();
    EST_Wave *sig = (EST_Wave *)fileitem->f("sig").ptr();
    EST_Track *unit_coeffs = new EST_Track;
    EST_Wave *unit_sig = new EST_Wave;

    int pm_start = unit->f("start");
    int pm_end = unit->f("end");

    coeffs->sub_track(*unit_coeffs, pm_start, pm_end - pm_start + 1);
    unit->fset("coeffs",unit_coeffs,gc_track);
    
    int samp_start = (int)(coeffs->t(Gof((pm_start - 1), 0))
		  * (float)sig->sample_rate());
    int samp_end = (int)(coeffs->t(pm_end + 1)
		     * (float)sig->sample_rate());
    sig->sub_wave(*unit_sig,samp_start, samp_end-samp_start+1);
    unit->fset("sig",unit_sig,gc_wave);

}

CLDB::CLDB()
{
    gc_protect(&params);
}

static void del_item(void *s) { delete (EST_Item *)s; }
CLDB::~CLDB()
{
    index.clear(del_item);
    fileindex.clear(del_item);
    gc_unprotect(&params);
}

