;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                         Copyright (c) 1998                            ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify, distribute this software and its    ;;
;;;  documentation for research, educational and individual use only, is  ;;
;;;  hereby granted without fee, subject to the following conditions:     ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;  This software may not be used for commercial purposes without        ;;
;;;  specific prior written permission from the authors.                  ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Finding features and acoustic distance measture for a set of 
;;;  segments in a database of utterances
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  This is primarily implement for the cluster unit selection method
;;;  but may uses in other unit selection schemes.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  There are five stages to this
;;;     Load in all utterances
;;;     Load in their coefficients
;;;     Collect together the units of the same type
;;;     build distance tables from them
;;;     dump features for them
;;;

(require_module 'clunits)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (do_all)
  (let ((utterances))

    (format t "Loading utterances and sorting types\n")
    (set! utterances (acost:db_utts_load dt_params))
    (set! unittypes (acost:find_same_types utterances))
    (acost:name_units unittypes)

    (format t "Loading coefficients\n")
    (acost:utts_load_coeffs utterances)
    (format t "Building distance tables\n")
    (acost:build_disttabs unittypes dt_params)

    (format t "Dumping features for clustering\n")
    (acost:dump_features unittypes utterances dt_params)

    ;; Build the cluster trees (requires disttabs and features)
    (format t "Building cluster trees\n")
    (acost:find_clusters (mapcar car unittypes) dt_params)
    (acost:collect_trees (mapcar car unittypes) dt_params)
    
    (format t "Saving unit catalogue\n")
    (acost:save_catalogue utterances dt_params)
    
  )
)

(define (do_init)
    (set! utterances (acost:db_utts_load dt_params))
    (set! unittypes (acost:find_same_types utterances))
    (acost:name_units unittypes)
    t)

(define (acost:db_utts_load params)
  "(acost:db_utts_load params)
Load in all utterances identified in database."
  (let ((files (car (cdr (assoc 'files params)))))
    (set! acost:all_utts
	  (mapcar
	   (lambda (fname)
	     (set! utt_seg (Utterance Text fname))
	     (utt.load utt_seg 
		       (string-append 
			(get_param 'db_dir params "./")
			(get_param 'utts_dir params "utts/")
			fname
			(get_param 'utts_ext params ".utt")))
	     utt_seg)
	   files))))

(define (acost:utts_load_coeffs utterances)
  "(acost:utts_load_coeffs utterances)
Loading the acoustic coefficients of for each utterance."
  (mapcar 
   (lambda (utt) (acost:utt.load_coeffs utt dt_params))
   utterances)
  t)

(define (acost:find_same_types utterances)
  "(acost:find_same_types utterances)
Find all the stream items of the same type and collect them into
lists of that type."
  (set! acost:unittypes nil)
  (mapcar 
   (lambda (u)
     (mapcar 
      (lambda (s) 
	(let ((p (assoc (item.name s) acost:unittypes)))
	  (if p
	      (set-cdr! p (cons s (cdr p)))
	      (set! acost:unittypes
		    (cons
		     (list (item.name s) s) acost:unittypes)))))
      (utt.relation.items u 'Segment)))
   utterances)
  acost:unittypes)

(define (acost:name_units unittypes)
  "(acost:name_units unittypes)
Names each unit with a unique id and number the occurences of each type."
  (let ((idnum 0) (tynum 0))
    (mapcar
     (lambda (s)
       (set! tynum 0)
       (mapcar
	(lambda (si)
	  (item.set_feat si "unitid" idnum)
	  (set! idnum (+ 1 idnum))
	  (item.set_feat si "occurid" tynum)
	  (set! tynum (+ 1 tynum)))
	(cdr s))
       (format t "units \"%s\" %d\n" (car s) tynum))
     unittypes)
    (format t "total units %d\n" idnum)
    idnum))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generating feature files

(define (acost:dump_features unittypes utterances params)
  "(acost:dump_features unittypes utterances params)
Do multiple passes over the utterances for each unittype and
dump the desired features.  This would be easier if utterances
weren't require for feature functions."
  (mapcar
   (lambda (utype)
     (acost:dump_features_utype 
      (car utype)
      utterances
      params))
   unittypes)
  t)

(define (acost:dump_features_utype utype utterances params)
  "(acost:dump_features_utype utype utterances params)
Dump features for all items of type utype."
  (let ((fd (fopen 
	     (string-append 
	      (get_param 'db_dir params "./")
	      (get_param 'feats_dir params "feats/")
	      utype
	      (get_param 'feats_ext params ".feats"))
	     "w"))
	(feats (car (cdr (assoc 'feats params)))))
    (format t "Dumping features for %s\n" utype)
    (mapcar 
     (lambda (u)
       (mapcar
	(lambda (s)
	  (if (string-equal utype (item.name s))
	      (begin 
		(mapcar 
		 (lambda (f)
		   (format fd "%s " (item.feat s f)))
		 feats)
		(format fd "\n"))))
	(utt.relation.items u 'Segment)))
     utterances)
    (fclose fd)))
	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tree building functions

(defvar wagon-balance-size 0)

(define (acost:find_clusters unittypes dt_params)
"Use wagon to find the best clusters."
  (mapcar
   (lambda (unittype)
     (build_tree unittype dt_params))
   unittypes)
  t)

(define (build_tree unittype dt_params)
"Build tree with Wagon for this unittype."
  (let ((command 
	 (format nil "%s -desc %s -data %s -balance %s -distmatrix %s -stop %s -output %s %s"
		 (get_param 'wagon_progname dt_params "wagon")
		 (string-append
		  (get_param 'db_dir dt_params "./")
		  (get_param 'wagon_field_desc dt_params "wagon"))
		 (string-append 
		  (get_param 'db_dir dt_params "./")
		  (get_param 'feats_dir dt_params "feats/")
		  unittype
		  (get_param 'feats_ext dt_params ".feats"))
		 (get_param 'wagon_balance_size dt_params 0)
		 (string-append 
		  (get_param 'db_dir dt_params "./")
		  (get_param 'disttabs_dir dt_params "disttabs/")
		  unittype
		  (get_param 'disttabs_ext dt_params ".disttab"))
		 (get_param 'wagon_cluster_size dt_params 10)
		 (string-append 
		  (get_param 'db_dir dt_params "./")
		  (get_param 'trees_dir dt_params "trees/")
		  unittype
		  (get_param 'trees_ext dt_params ".tree"))
		 (get_param 'wagon_other_params dt_params "")
		 )))
    (format t "%s\n" command)
    (system command)))

(define (acost:collect_trees unittypes params)
"Collect the trees into one file as an assoc list"
  (let ((fd (fopen 
	     (string-append 
	      (get_param 'db_dir params "./")
	      (get_param 'trees_dir params "trees/")
	      (get_param 'index_name params "all.")
	      (get_param 'trees_ext params ".tree"))
	      "wb"))
	(tree_pref
	     (string-append 
	      (get_param 'db_dir params "./")
	      (get_param 'trees_dir params "trees/")))
	(cluster_prune_limit (get_param 'cluster_prune_limit params 0)))
    (format fd ";; Autogenerated list of selection trees\n")
    (mapcar
     (lambda (fp)
       (format fd ";; %l %l\n" (car fp) (car (cdr fp))))
     params)
    (format fd "(set! clunits_selection_trees '(\n")
    (mapcar
     (lambda (unit)
       (set! tree (car (load (string-append tree_pref unit ".tree") t)))
       (if (> cluster_prune_limit 0)
	   (set! tree (cluster_tree_prune tree cluster_prune_limit)))
       (pprintf (list unit tree) fd))
     unittypes)
    (format fd "))\n")
    (fclose fd)))

(define (cluster_tree_prune_in_line prune_limit)
"(cluster_tree_prune_in_line)
Prune number of units in each cluster in each tree *by* prune_limit,
if negative, or *to* prune_limit, if positive."
  (set! sucs_select_trees 
        (mapcar
	 (lambda (t)
	     (cluster_tree_prune t prune_limit))
	 sucs_select_trees)))

(define (cluster_tree_prune tree prune_limit)
"(cluster_tree_prune TREE PRUNE_LIMIT)
Reduce the number of elements in the (CART) tree leaves to PRUNE_LIMIT
removing the ones further from the cluster centre.  Maybe later this should
have guards on minimum number of units that must remain in the tree and
a per unit type limit."
  (cond
   ((cdr tree)  ;; a question
    (list
     (car tree)
     (cluster_tree_prune (car (cdr tree)) prune_limit)
     (cluster_tree_prune (car (cdr (cdr tree))) prune_limit)))
   (t           ;; tree leave
    (list 
     (list
      (remove_n_worst 
       (car (car tree))
       (if (< prune_limit 0)
	   (* -1 prune_limit)
	   (- (length (car (car tree))) prune_limit)))
      (car (cdr (car tree))))))))

(define (remove_n_worst lll togo)
"(remove_n_worst lll togo)
Remove togo worst items from lll."
  (cond
   ((< togo 0)
    lll)
   ((equal? 0 togo)
    lll)
   (t
    (remove_n_worst
     (remove (worst_unit (cdr lll) (car lll)) lll)
     (- togo 1)))))

(define (worst_unit lll worst_so_far)
"(worst_unit lll worst_so_far)
Returns unit with worst score in list."
  (cond
   ((null lll)
    worst_so_far)
   ((< (car (cdr worst_so_far)) (car (cdr (car lll))))
    (worst_unit (cdr lll) (car lll)))
   (t
    (worst_unit (cdr lll) worst_so_far))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Save the unit catalogue for use in the run-time index

(define (acost:save_catalogue utterances dt_params)
  "(acost:save_catalogue utterances dt_params)
Save the catalogue with named units with times."
 (let ((fd (fopen 
	    (string-append 
	     (get_param 'db_dir dt_params "./")
	     (get_param 'catalogue_dir dt_params "trees/")
	     (get_param 'index_name dt_params "catalogue.")
	     ".catalogue")
	      "wb"))
       (num_units 0)
       )
   (format fd "EST_File index\n")
   (format fd "DataType ascii\n")
   (format fd "NumEntries %d\n"
	   (apply 
	    + (mapcar (lambda (u) 
			(length (utt.relation.items u 'Segment))) utterances)))
   (format fd "IndexName %s\n" (get_param 'index_name dt_params "cluser"))
   (format fd "EST_Header_End\n")
   (mapcar
    (lambda (u)
      (mapcar
       (lambda (s)
	 (format fd "%s_%s %s %f %f %f\n"
		 (item.name s)
		 (item.feat s 'occurid)
		 (utt.feat u 'fileid)
		 (item.feat s 'segment_start)
		 (item.feat s 'segment_mid)
		 (item.feat s 'segment_end)))
       (utt.relation.items u 'Segment)))
    utterances)
   (fclose fd)))

(provide 'clunits)
