 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/

#ifndef __RAWUNITDATABASE_H__
#define __RAWUNITDATABASE_H__

#include "UnitDatabase.h"

/** Databases which consist of indivisual files in the filesystem,
  * with textual cataloges and indecies and so on.
  *
  * Like any other UnitDatabase, this kind is associated with a
  * specific file. However, the file does not contain the information
  * related to the units, rather it just contains a description of
  * where to get that information.
  * 
  * @see RawUnitCatalogue
  * @author Richard Caley <rjc@cstr.ed.ac.uk>
  * @version $Id: RawUnitDatabase.h,v 1.1 1998/08/12 09:35:52 awb Exp $
  */

class RawUnitDatabase : public UnitDatabase {

public:
    /// Structure used to record information about the data available for each unit.
    struct available_data {
	/// The type of information (eg `residual')
	ContentType contents;
	/// Where to look.
	EST_Pathname directory;
	/// File extension.
	EST_String extension;
	/// Information for waveforms.
	struct UnitDatabase::wave_information_s wave_info;
	/// Information for coefficiant sets.
	struct UnitDatabase::coefficient_information_s coef_info;
	/// Syntactic convinience.
	int is_waveform(void) { return UnitDatabase::is_waveform(contents);} 
    };

private:

    /// `Magic number' on files describing databases this implementation understands.
    static const char s_magic[];
    /// Size of `Magic number'.
    static const int s_magic_size;

    /// Filename of catalogue of units.
    EST_Pathname p_catalogue_filename;

    /// Sample rate for all units.
    int p_sample_rate;

    /// How many kinds of information for each unit.
    int p_num_data;
    /// Description of each type of unit data.
    struct available_data *p_data;

    /// Extract description of data from format in file.
    int process_data_included(ValueType value);
    /// Extract indexes from description in file.
    int process_indexes(ValueType value);

public:
    /// Default constructor.
    RawUnitDatabase(void);
    /// Destructor.
    virtual ~RawUnitDatabase(void);

    /**@name Subtyping */
    //@{
    /// Recognise a RawUnitDatabase on a stream.
    static bool recognise(FILE *stream);
    /// Create a RawUnitDatabase.
    static UnitDatabase *create(void);
    /// Get the database information from the stream.
    virtual EST_read_status fill_from(FILE *stream);
    /// Write the database out.
    virtual EST_write_status write_to(FILE *stream);
    
    /// Name of this subtype.
    virtual EST_String subtype_name(void) const {return "Raw";}
    //@}


    /**@name Synthesis information. */
    //@{
    /// Names of the waveforms available for each unit.
    virtual EST_StrVector wave_names(void) const;
    /// Names of the coefficient sets available for each unit.
    virtual EST_StrVector coefficient_names(void) const;

    /// Return information about an available waveform.
    virtual struct wave_information_s 
		*wave_information(ContentType content);
    /// Return information about an available coefficient set.
    virtual struct coefficient_information_s 
		*coefficient_information(ContentType content);

    /// Get a waveform for this unit.
    virtual EST_Wave *get_wave(ContentType content, Unit &unit);
    /// Get a coefficient set for this waveform.
    virtual EST_Track *get_coefficients(ContentType content,  
					EST_ContourType format, Unit &unit);
    
    /// Tell me how this unit divides into Segments.
    virtual EST_TVector<float> *get_segments(Unit &unit);
    //@}

    /**@name Properties */
    //@{
    /// Names of all available properties.
    virtual void property_names(EST_TList<EST_String> &list) const;
    /// Set a single property
    virtual int set_property(EST_String property, ValueType value);
    /// Get the value for a property.
    virtual ValueType property(EST_String property) const;
    //@}

    /// Human readable description of the database with some detail.
    virtual void print_description(FILE *stream);
};

#endif
