 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1996,1997                         */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/

#ifndef __FRAMESUNIT_H__
#define __FRAMESUNIT_H__

#include "Unit.h"

/** A Unit type for units which have content which consists of frames
  * each representing a pitch period containing LPCs, residual,
  * original signal or whatever.
  * 
  * This class provides a summary of the utterance structure called
  * `Chunks'. An utternace strurture can be mapped into a list of chunks,
  * each of which represents one item in the segment stream. Within a
  * chunk are a number of bits, each of which is a section of a FramesUnit.
  * 
  * The Chunks structure is much easier to pass over than the utterance
  * it summarises, makeing it useful for writing modules in an
  * understandable way.
  * 
  * @see Chunking
  * @see Frames_modifications
  * @author Richard Caley <rjc@cstr.ed.ac.uk>
  * @version $Id: FramesUnit.h,v 1.1 1998/08/12 09:35:52 awb Exp $
  */

class FramesUnit : public Unit {
public:

  /**@name Chunks */
  //@{
  /// How many parts might a segment be represented by.
#    define MAX_UNITS_IN_CHUNK (3)
  /// A structure used to hold the (pieces of) units which make up a given chunk of utterance (a Segment, but could be any span).

  struct Chunk  {
    /// How many fragments.
    int n;
    //  public:
    /// A frament of a segment.
    struct bit_s
    {
      /// The unit involved.
      FramesUnit *unit;
      /// Where to start.
      int start_frame;
      /// Where to end.
      int end_frame;
    } bit[3];
    /// Array of fragments.
    //    struct bit_s bit[3];
  };
  //@}

private:

  /**@name Information 
    * The various types of information associated with the unit are
    * cached here.
    */
  //@{
  /// Original signal.
  EST_Wave *p_signal;
  /// LPC residual.
  EST_Wave *p_residual;
  /// Analysis as supplied by Database
  EST_Track *p_analysis;
  /// Analysis as LPCs.
  EST_Track *p_lpc;
  /// Analysis as reflection coefficiants.
  EST_Track *p_rfc;
  /// Pitchmarks.
  EST_Track *p_pitchmarks;
  //@}

  /// How the unit divides into Segments
  EST_TVector<int> *p_segment_ends;

  /**@name Modifications 
    * If this is a modified version of a Unit from a database, the
    * modifications are stored as small tracks.
    * @see Frames_modifications
    */
  //@{
  /// Number of modifications.
  int p_n_mods;
  /// Modification information (prosody etc)
  EST_Track *p_modifications[MAX_UNITS_IN_CHUNK];
  //@}

public:
  /// Default constructor.
  FramesUnit(void);
  /// Destructor.
  virtual ~FramesUnit(void);


  /**@name Subtyping */
  //@{
  /// Recognise it in a file
  static bool recognise(FILE *stream) { (void)stream; return FALSE;};
  /// Create a FramesUnit
  static Unit *create(void);
  /// Name of this type of unit
  virtual EST_String subtype_name(void) const {return "Frames";};
  //@}

  /**@name Properties*/
  //@{
  /// Names of all available properties.
  virtual void property_names(EST_TList<EST_String> &list) const;
  /// Get single property value.
  virtual ValueType property(EST_String name) const;
  //@}

  /**@name Segmentation */
  //@{
  /// Length in Frames.
  virtual int length(void);
  /// How it breaks down into Segments.
  virtual EST_TVector<int> *segment_ends(void);
  //@}

  /**@name Information */
  //@{
  /// Return the original waveform.
  EST_Wave *signal(void);
  /// Return the pitchmarks
  EST_Track *pitchmarks(void);
  /// Return the analysis in the unit's preferred form.
  EST_Track *analysis(void);
  /// Return the LPC residual.
  EST_Wave *residual(void);
  /// Return the analysis as LPC coefficiants
  EST_Track *lpc(void);
  /// Return the analysis as reflection coefficiants
  EST_Track *rfc(void);
  //@}

  /**@name Modifications 
    * Modified versions of units are created by adding modification
    * tracks. The concatenation of these tracks forms the unit to
    * be synesised.
    * @see Frames_modifications
    */
  //@{
  /// Add amodification too the end of the list.
  int add_modifications(EST_Track *mods);
  /// How many sections of modifications
  int n_modifications(void) const;
  /// The modifications. These combined make up the unit as it should be synthesised.
  EST_Track *modifications(int i) const;
  //@}

  /// Sample rate of the signal or residual.
  int sample_rate(void);

  /**@name Chunks
    * Utility functions which produces a description of how units and joins
    * map to Segments. This just wraps up the loops needed to walk over the
    * utterance structure since these are quite, er, eldritch.
    * 
    * Returns a static buffer, the end of the list of chunks can
    * be recognised by looking for `n' being 0.
    * @see Chunking
    */
//@{

  /// Create chunks from utterance.
  static Chunk *chunk_utterance(EST_Utterance *utterance,
				      const EST_String unit_stream_name,
				      const EST_String join_stream_name);

  /// Create chunks from utterance.
  static Chunk *chunk_utterance(EST_Utterance *utterance,
				      EST_Relation *unit_rel,
				      EST_Relation *join_rel);

  /// Print chunks to stream (for debugging)
  static void dump_chunks(ostream &stream, const Chunk *);
  //@}

};

#endif
