 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/

/**@name SchemeObject.h
  * Common code for objects available from scheme.
  * <p>
  * To add new Scheme types you need only subtype SchemeObject
  * and add your type's id to the festival and siod headers.
  * @see SchemeObject
  * @see SchemeModuleDescription, an example
  * @see SubtypedObject, objects which can have subtypes
  * @author Richard Caley <rjc@cstr.ed.ac.uk>
  * @version $Id: SchemeObject.h,v 1.1 1998/08/12 09:35:52 awb Exp $
  */
//@{

#if !defined(__SCHEMEOBJECT_H__)
#define __SCHEMEOBJECT_H__

/** Undefine to use these types outside SIOD. This is not quite all it
  * should be yet since some of the methods take property lists as
  * parameters and we use the lisp reader once, but the thought is
  * here and it should only take a bounded amount of work to free the 
  * types. 
  */
#define INCLUDE_LISP (1)

/** Don't inline methods.
  * There is a bug in (some versions of?) gcc when not optimising and
  * dealing with inline member functions which are used inside others
  * inline functions whose address is taken. Or at least that's where
  * it bit me (rjc).  
  */

#if defined(__GNUC__) && !defined(__OPTIMIZE__)
#   define DONT_INLINE_EVERYTHING (1)
#endif

#if defined(DONT_INLINE_EVERYTHING)
#   define INLINE_IF_OK(X) // empty
#else
#   define INLINE_IF_OK(X) X
#endif

#if defined(INCLUDE_LISP)

#   include <stdio.h>
#   include "siod.h"
#   include "festival.h"

/**@name Object property values
  * Properties of objects are passed around inside a theoretically
  * opaque type. In fact this is currently a SIOD list structure.
  */
//@{
    /// Type of properties of objects.
    typedef LISP ValueType;

    /// Convert value to const char *
#   define value_as_c_string(V) get_c_string(V)
    /// Convert value to EST_String
#   define value_as_string(V) EST_String(get_c_string(V))
    /// Convert value to int
#   define value_as_int(V) (TYPEP(V,tc_flonum)?(int)FLONM(V):0)
    /// Convert value to float
#   define value_as_float(V) (TYPEP(V,tc_flonum)?FLONM(V):0.0)
    /// Convert value to SIOD list structure
#   define value_as_list(V) (V)
    /// Convert EST_String to value
#   define string_as_value(S) string_cell((const char *)(S), (S).length())
    /// Convert const char * to value 
#   define c_string_as_value(S) string_cell((S), ((S)?(strlen(S)):0))
    /// Convert int to value 
#   define int_as_value(N)  flocons(N)
    /// Convert floatto value 
#   define float_as_value(N)  flocons(N)
    /// Convert NULL pointer to value 
#   define null_as_value (LISP)NULL
    /// Convert SIOD list structure to value 
#   define list_as_value(L) (L)
//@}
#endif

/** Base class which defines the mechanisms necessary for any C++
  * type which is to be used from Scheme.
  * @see SchemeObject.h
  */
class SchemeObject {
protected:
  /// A place to put a description of any error which occurs while processing the object.
  static EST_String error_string;

protected:
  /// Flag for mark-sweep garbage collection in Scheme.
  int p_gc_mark;
  /// Reference count for garbage collection in C++ structures.
  unsigned int p_reference_count;
  /// Name of this object.
  EST_String p_name;

public: 
  /// Default constructor.
  SchemeObject(void);
  /// Copy constructor.
  SchemeObject(const SchemeObject &a);
  /// Destructor
  virtual ~SchemeObject(void);

  /// Return the name of the type.
  virtual EST_String type_name(void) const {return "SchemeObject";};

  /// Object name.
  EST_String name(void) const INLINE_IF_OK({return p_name;});
  /// Set object name.
  void set_name(EST_String name) INLINE_IF_OK({p_name = name;});

  /**@name Properties
    * An object's properties are set and accessed through the
    * ValueType type. A list of all of the properties of an object can
    * be retrieved to allow save and load mechanisms to be defined.
    * @see ValueType
    */
  //@{
  /// Set object properties from alist.
  int set_properties(LISP properties);

  /// Return the name of all the properties this object understands.
  virtual void property_names(EST_TList<EST_String> &list) const;
  /// Return value of the given property. 
  virtual ValueType property(EST_String name) const;
  /// Set a property value.
  virtual int set_property(EST_String name, ValueType value);
  //@}

  /**@name Garbage collection
    * Objects can be accessable from C++ as well as from Scheme.
    * For this reason we implement both mark-sweep and reference counted
    * garbage collection.
    * <p>
    * Some objects are part of larger structures (for instance units in
    * databases), in such cases oneobject is nominated as `master' and
    * carries the mark/counter for the entire structure.
    */
  //@{
  /// Mark as accessable.
  virtual void gc_mark(void) INLINE_IF_OK({(gc_master_pointer()->p_gc_mark)++;});
  /// Unmark.
  virtual void gc_clear_mark(void) INLINE_IF_OK({(gc_master_pointer()->p_gc_mark)=0;});
  /// Check if marked as accessable.
  virtual int  gc_marked(void) INLINE_IF_OK({return (gc_master_pointer()->p_gc_mark);});

  /// Increment reference count.
  virtual void gc_ref(void) INLINE_IF_OK({(gc_master_pointer()->p_reference_count)++;});
  /// Decrement reference count.
  virtual void gc_unref(void) INLINE_IF_OK({(gc_master_pointer()->p_reference_count)--;});
  /// Check for non-zero reference count.
  virtual int  gc_referenced(void) INLINE_IF_OK({return (gc_master_pointer()->p_reference_count)>0;});

  /// Return master object.
  virtual SchemeObject *gc_master_pointer(void) INLINE_IF_OK({return this;});

  /// To safely free an object which may be seen by lisp, we put it in an unreferenced cell and let the GC take care of it.
  static void decrease_reference_count(void *it);
  //@}

#if defined(INCLUDE_LISP)

  /**@name Scheme interface
    * These are the functions which are imported into Scheme. The 
    * importation (allong with any other necessary setup) is done by
    * lisp_declare.
    */
  //@{

protected:
  /// The type tag used in Scheme cells copntaining this type of object.
  static int s_lisp_type;

public:
  /// Mark as accessable.
  static LISP lisp_gc_mark(LISP);
  /// Clear mark.
  static void lisp_gc_clear_mark(LISP);
  /// Free the object.
  static void lisp_gc_free(LISP);

  /// Set a single property.
  static LISP lisp_set_property(LISP object, LISP property, LISP value);
  /// Set properties from an alist.
  static LISP lisp_set_properties(LISP object, LISP properties);
  /// Get list of property names.
  static LISP lisp_property_names(LISP object);
  /// Get value of a property. 
  static LISP lisp_get_property(LISP object, LISP property);

  /// Print out object
  static void lisp_print(LISP, FILE *);
  /// Print object to string.
  static void lisp_print_string(LISP, char *);

  /// Initialisation function.
  static void lisp_declare(void);
  //@}
#endif
};

SchemeObject *get_c_schemeobject(LISP x);

#endif
//@}
