/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Functions for accessing parts of the utterance through "features"     */
/* features are a cononical addressing method for relatively accessing   */
/* information in an utterance from a given item in it.                  */
/*                                                                       */
/* A feature name is is a dotted separated string of names ended by a    */
/* a final feature.  The names before the final feature are navigational */
/* consisting of a few builtin forms (i.e. next and prev) or names of    */
/* relations that are to be followed,                                    */
/*                                                                       */
/* For example:                                                          */
/*   "name"          name of the current item                            */
/*   "n.name"        name of the next item (in the current relation      */
/*   "n.n.name"      is the name of the next next item                   */
/*   "R:SylStructure.parent.name"   name of the parent in the            */
/*                   SylStructure relation                               */
/* for example from an item in the Segment relation                      */
/*   "p.R:SylStructure.parent.syl_break"                                 */
/*                   is the syllable break level of the syllable related */
/*                   the previous segment.                               */
/*   "R:SylStucture.parent.R:Syllable.p.syl_break"                       */
/*                   is the syllable break level of the syllable related */
/*                   this item's previous Syllable item.                 */
/*                                                                       */
/* The following features are defined for all stream items               */
/*     name                                                              */
/* Other feature are defined through the C++ function festival_def_ff    */
/* Note duration, is no longer predefined                                */
/*                                                                       */
/* To extre features are defined here ph_* (i.. any feature prefixed by  */
/* ph_) and lisp_*.  ph_ function check the phoneme set anduse the       */
/* remainder of the name as a phone feature return its value for the     */
/* current item (which will normally be a Segment stream item).  e.g.    */
/* ph_vc or ph_vheight                                                   */
/* The lisp_* feature will call a lisp function (the name following      */
/* the lisp_) with two arguments the utterance and the stream item.      */
/* this allows arbitrary new features without recompilation              */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include "EST_unix.h"
#include <string.h>
#include "festival.h"
#include "festivalP.h"

static LISP ff_pref_assoc(const char *name, LISP alist);

static LISP ff_docstrings = NULL;
static LISP ff_pref_list = NULL;

LISP lisp_val(const EST_Val &pv)
{
    if (pv.type() == val_unset)
    {
	cerr << "EST_Val unset, can't build lisp value" << endl;
	festival_error();
	return NIL;
    }
    else if (pv.type() == val_int)
	return flocons(pv.Int());
    else if (pv.type() == val_float)
	return flocons(pv.Float());
    else if (pv.type() == val_string)
	return strintern(pv.string_only());
    else
    {
	cerr << "EST_PVal has unknown type, can't build lisp value" << endl;
	festival_error();
	return NIL;
    }
}

void festival_def_nff(const EST_String &name,const EST_String &sname, 
		     EST_Item_featfunc func,char *doc)
{
    //  define the given feature function with documentation

    register_featfunc(name,func);
    if (ff_docstrings == NIL)
	gc_protect(&ff_docstrings);
    EST_String id = sname + "." + name;
    ff_docstrings = cons(cons(rintern(id),cstrcons(doc)),ff_docstrings);
    siod_set_lval("ff_docstrings",ff_docstrings);

    return;
}

static int tc_pref_ffunc;

static FT_ff_pref_func get_c_pref_ffunc(LISP x)
{
    if (TYPEP(x,tc_pref_ffunc))
	return (FT_ff_pref_func)USERVAL(x);
    else
	err("wta to get_c_pref_ffunc",x);
    return NULL;  // err doesn't return but compilers don't know that
}

static LISP siod_make_pref_ffunc(FT_ff_pref_func f)
{
    if (f==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_pref_ffunc,f);
}

void festival_def_ff_pref(const EST_String &pref,const EST_String &sname, 
			  FT_ff_pref_func func, char *doc)
{
    // define the given class of feature functions
    // All feature functions names with this prefix will go to this func
    LISP lpair;

    lpair = siod_assoc_str(pref,ff_pref_list);
    
    if (lpair == NIL)
    {
	if (ff_pref_list == NIL)
	{
	    gc_protect(&ff_pref_list);
	    tc_pref_ffunc = siod_register_user_type("Pref_FFunc");
	}
	ff_pref_list = cons(cons(rintern(pref),
				 cons(siod_make_pref_ffunc(func),NIL)),
		       ff_pref_list);
	EST_String id = sname + "." + pref;
	ff_docstrings = cons(cons(rintern(id),cstrcons(doc)),ff_docstrings);
	siod_set_lval("ff_docstrings",ff_docstrings);
    }
    else
    {
	cerr << "ffeature (prefix)" << pref << " duplicate definition" << endl;
	festival_error();
    }

    return;
}

static LISP ff_pref_assoc(const char *name, LISP alist)
{
    // Search list of ff_pref_funcs to see if name has an appropriate
    // prefix 
    LISP l;
    char *prefix;
    
    for (l=alist; CONSP(l); l=CDR(l))
    {
	prefix = get_c_string(CAR(CAR(l)));
	if (strstr(name,prefix) == name)
	    return CAR(l);
    }

    // not found
    return NIL;
}

static EST_String Feature_Separator = ".";
static EST_String Feature_PunctuationSymbols = EST_String::Empty;
static EST_String Feature_PrePunctuationSymbols = EST_String::Empty;
static EST_Val default_feature_value(0);

// Moving this to be static gives me an extra second but of course
// makes it thread unsafe.
static EST_TokenStream ts;

EST_Val ffeature(EST_Item *item,const EST_String &fname)
{
    // Select and apply feature function name to s and return result 
    FT_ff_pref_func pfunc;
    EST_Item_featfunc func;
    LISP lpair;
    EST_Item *s = item;

    if (item == 0)
	return default_feature_value;
    if (strchr(fname,'.') == 0)
    {   // if its a simple name do it quickly, without tokenizing
	if ((func = get_featfunc(fname)) != 0)
	    return (func)(item);
	else if ((lpair = ff_pref_assoc(fname,ff_pref_list)) != NIL)
	{
	    pfunc = get_c_pref_ffunc(CAR(CDR(lpair)));
	    return (pfunc)(item,fname);
	}
	else // it must be a feature name for this item
	    return item->f(fname);
    }
    ts.open_string(fname);
    ts.set_WhiteSpaceChars(Feature_Separator);
    ts.set_PunctuationSymbols(Feature_PunctuationSymbols);
    ts.set_PrePunctuationSymbols(Feature_PrePunctuationSymbols);

    while (!ts.eof())
    {
	const EST_String &Sname = ts.get().string();
	const char *name = Sname;
	if (streq(name,"n"))
	    s=next(s);
	else if (streq(name,"p"))
	    s=prev(s);
	else if (streq(name,"nn"))
	    s=next(next(s));
	else if (streq(name,"pp"))
	    s=prev(prev(s));
	else if (streq(name,"up"))
	    s=up(s);
	else if (streq(name,"down"))
	    s=down(s);
	else if (streq(name,"parent"))
	    s=parent(s);
	else if (streq(name,"daughter1"))
	    s=daughter1(s);
	else if (streq(name,"daughter2"))
	    s=daughter2(s);
	else if (streq(name,"daughtern"))
	    s=daughtern(s);
	else if (streq(name,"last"))
	    s=last(s);
	else if (streq(name,"first"))
	    s=first(s);
	else if ((func = get_featfunc(Sname)) != 0)
	    return (func)(s);
	else if ((lpair = ff_pref_assoc(name,ff_pref_list)) != NIL)
	{
	    pfunc = get_c_pref_ffunc(CAR(CDR(lpair)));
	    return (pfunc)(s,Sname);
	}
	else if (strncmp(name,"R:",2) == 0)  // new relation structure
	    s = s->as_relation(&name[2]);
	else // it must be a feature name for this stream item
	    return s->f(Sname);

	if (s==0)
	    return default_feature_value;
    }

    cerr << "Invalid ffeature name: \"" << fname << "\"" << endl;
    festival_error();

    return default_feature_value;
}

static LISP lisp_item_feature(LISP litem, LISP name)
{
    // return the ffeature name for this stream 
    EST_Item *s = get_c_item(litem);
    EST_String fname = get_c_string(name);

    return lisp_val(ffeature(s,fname));
}

static LISP lisp_item_set_feat(LISP litem, LISP name, LISP value)
{
    // set the feature (locally) on this sitem
    EST_Item *s = get_c_item(litem);
    EST_String fname = get_c_string(name);

    if ((fname.contains(".")) ||
	(fname.contains("R:")))
    {
	cerr << "item.set_feat: cannot set fet name containing " <<
	    "\".\" or \"R:\"" << endl;
	festival_error();
    }
    if (consp(value))
    {
	cerr << "item.set_feat: cannot set non-atomic value" << endl;
	festival_error();
    }
    else if (TYPEP(value,tc_flonum))
	s->fset(fname,get_c_float(value));
    else
	s->fset(fname,get_c_string(value));

    return value;
}

static EST_Val ff_lisp_func(EST_Item *i,const EST_String &name)
{
    // This function is called for features functions starting lisp_
    // It calls the lisp function following that with u and i
    // as arguments, the return value (which must be atomic) is
    // then passed back as a Val.  I'm not sure if this will be 
    // particularly efficient, but it will make development of
    // new features quicker as they can be done in Lisp without
    // changing the C++ code.
    EST_String lfunc_name = name.after("lisp_");
    LISP r,l;

    l = cons(rintern(lfunc_name),
	     cons(siod_make_item(i),NIL));
    r = leval(l,NIL);
    if ((consp(r)) || (r == NIL))
    {
	cerr << "FFeature Lisp function: " << lfunc_name << 
	    " returned non-atomic value" << endl;
	festival_error();
    }
    else if (numberp(r))
	return EST_Val(get_c_float(r));
    
    return EST_Val(get_c_string(r));
}

void festival_features_init(void)
{
    // declare feature specific Lisp functions 

    festival_def_ff_pref("lisp_","any",ff_lisp_func,
    "ANY.lisp_*\n\
  Apply Lisp function named after lisp_.  The function is called with\n\
  an stream item.  It must return an atomic value.\n\
  This method may be inefficient and is primarily desgined to allow\n\
  quick prototyping of new feature functions.");

    init_subr_2("item.feat",lisp_item_feature,
    "(ling_item.feat ITEM FEATNAME)\n\
   Return value of FEATNAME (which may be a simple feature name or a\n\
   pathname of ITEM.");
    init_subr_3("item.set_feat",lisp_item_set_feat,
    "(item.set_feat ITEM FEATNAME VALUE)\n\
   Set FEATNAME to VALUE in ITEM.");

}



