//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This class emulates a 6507 microprocessor, however, the memory
  accesses and cycle counts it generates are only valid at the
  instruction level.  For better compatibility it's neccessary to
  increment cycles counts at the sub-instruction level and issues
  "false" reads for some instructions (such as Indirect,X addressing
  when it crosses a page boundary).

  @author  Bradford W. Mott
  @version $Id: M6507Lo.cxx,v 1.1 1997/05/17 19:04:13 bwmott Exp $
*/

#include <assert.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include "M6507Lo.hxx"

//============================================================================
// Answers true iff the two addresses are not on the same page
//============================================================================
static inline bool notSamePage(uWord addr1, uWord addr2)
{
  return (addr1 ^ addr2) & 0xff00;
}

//============================================================================
// Default constructor
//============================================================================
M6507Low::M6507Low(System& system)
    : M6507(system)
{
}

//============================================================================
// Destructor
//============================================================================
M6507Low::~M6507Low()
{
}

//============================================================================
// Answer the byte at the given address (only lower 8-bits used)
//============================================================================
inline uWord M6507Low::peek(uWord addr)
{
  return mySystem.peek(addr);
}

//============================================================================
// Answer the uByte at the given address that should be part of
// an instruction we're trying to execute
//============================================================================
inline uWord M6507Low::iPeek(uWord addr)
{
#ifdef M6507_FAST_ACCESS
  return mySystem.fastPeek(addr);
#else
  return mySystem.peek(addr);
#endif
}

//============================================================================
// Answer the uWord at the given address
//============================================================================
inline uWord M6507Low::iDPeek(uWord addr)
{
#ifdef M6507_FAST_ACCESS
  return mySystem.fastDPeek(addr);
#else
  uWord value = mySystem.peek(addr++);
  return value | (((uWord)mySystem.peek(addr)) << 8);
#endif
}

//============================================================================
// Write given byte to the given address 
//============================================================================
inline void M6507Low::poke(uWord addr, uByte value)
{
  mySystem.poke(addr,value);
}

//============================================================================
// Execute instructions until at least the given number of
// cycles have passed or execution is halted by someone
//============================================================================
void M6507Low::execute(uLong numberOfCycles)
{
  uLong stop = myCycles + numberOfCycles;
  myHaltExecution = false;

  // Loop until execution is halted or we're finished
  while(!myHaltExecution && (myCycles < stop))
  {
    uWord operandAddress = 0;

#ifdef DEBUG
    debugStream << "PC=" << hex << setw(4) << PC << " ";
#endif

    // Fetch instruction at the program counter
    IR = iPeek(PC++);

    // Update cycles
    myCycles += ourInstructionCycleTable[IR];

#ifdef DEBUG
    debugStream << "IR=" << hex << setw(2) << (int)IR << " ";
    debugStream << "<" << ourAddressingModeTable[IR] << " ";
#endif

    // Call the code to execute the instruction
    switch(IR)
    {
      // The code to simulate the 6507 instructions is 
      // generated by an M4 macro file
      #include "M6507.ins"

      default:
        invalidInstruction();
    } 

#ifdef DEBUG
    debugStream << hex << setw(4) << operandAddress << " ";
    debugStream << setw(4) << ourInstructionMnemonicTable[IR];
    debugStream << "> ";
    debugStream << "A=" << ::hex << setw(2) << (int)A << " ";
    debugStream << "X=" << ::hex << setw(2) << (int)X << " ";
    debugStream << "Y=" << ::hex << setw(2) << (int)Y << " ";
    debugStream << "PS=" << ::hex << setw(2) << (int)PS() << " ";
    debugStream << "SP=" << ::hex << setw(2) << (int)SP << " ";
    debugStream << "Cyc=" << ::dec << cycles();
    debugStream << endl;
#endif
  }
}

