
/*
 * DREADERD/READER.C - reader task
 *
 *	Reader task, main control loop(s).
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void ReaderTask(int fd);
Prototype void NNCommand(Connection *conn);
Prototype void NNCommand2(Connection *conn);
Prototype void NNBadCommandUse(Connection *conn);
Prototype void NNUnknownCommand(Connection *conn);
Prototype void NNTerminate(Connection *conn);
Prototype void NNWriteHello(Connection *conn);
Prototype void NNWaitThread(Connection *conn);
Prototype void NNTPHelp(Connection *conn, char **pptr);
Prototype Connection *InitConnection(ForkDesc *desc, DnsRes *dres);

Prototype KPDB *KDBActive;

void HandleReaderMsg(ForkDesc *desc);
void DeleteConnection(Connection *conn);

KPDB *KDBActive;
int   TFd;		/* thread's return fd to the parent */
int   NumReaders;
int   TerminatePending;

void
ReaderTask(int fd)
{
    time_t dtime = 0;
    time_t ltime = 0;
    int counter = 0;

    TFd = fd;

    /*
     * [re]open RTStatus
     */
    RTStatusOpen(RTStatus, NumReaderForks * DiabloReaderThreads + 1, DiabloReaderThreads);

    /*
     * Setup thread for passed pipe
     */
    ResetThreads();
    AddThread("reader", fd, -1, THREAD_READER, -1);

    FD_SET(fd, &RFds);

    /*
     * Open KPDB database for active file
     */

    if ((KDBActive = XKPDBOpen(O_RDWR, "%s/dactive.kp", NewsHome)) == NULL) {
	logit(LOG_CRIT, "Unable to open dactive.kp");
	exit(1);
    }

    /*
     * Selection core
     */

    CheckServerConfig(time(NULL), 1);

    while (TerminatePending == 0 || NReadServAct || NumReaders) {
	/*
	 *  select core
	 */
	struct timeval tv = { 2, 0 };
	fd_set rfds = RFds;
	fd_set wfds = WFds;
	int i;

	stprintf("reader readers=%02d spoolsrv=%d/%d postsrv=%d/%d",
	    NumReaders,
	    NReadServAct, NReadServers, 
	    NWriteServAct, NWriteServers
	);

	select(MaxFds, &rfds, &wfds, NULL, &tv);

	/*
	 * select is critical, don't make unnecessary system calls.  Only
	 * test the time every 10 selects (20 seconds worst case), and
	 * only check for a new server configuration file every 60 seconds 
	 * after the initial load.  This may rearrange THREAD_SPOOL and
	 * THREAD_POST threads.
	 */

	if (++counter == 10) {
	    time_t t = time(NULL);
	    if (ltime)
		dtime += t - ltime;
	    if (dtime < -5 || dtime >= 60) {
		CheckServerConfig(t, ServerTerminated);
		dtime = 0;
	    }
	    ltime = t;
	    counter = 0;
	}

	for (i = 0; i < MaxFds; ++i) {
	    if (FD_ISSET(i, &rfds) || FD_ISSET(i, &wfds)) {
		ForkDesc *desc;

		if ((desc = FindThread(i, -1)) != NULL) {
		    Connection *conn = desc->d_Data;

		    if (conn) {
			/*
			 * handle output I/O (optimization)
			 */

			MBFlush(&conn->co_TMBuf);
			conn->co_FCounter = 0;
		    }

		    /*
		     * Function dispatch
		     */

		    switch(desc->d_Type) {
		    case THREAD_READER:
			HandleReaderMsg(desc);
			break;
		    case THREAD_NNTP:		/* client	  */
		    case THREAD_SPOOL:		/* spool server	  */
		    case THREAD_POST:		/* posting server */
			conn->co_Func(conn);
			break;
		    default:
			/* panic */
			break;
		    }

		    /*
		     * do not call MBFlush after the function because the
		     * function may be waiting for write data to drain and
		     * we don't want to cause write data to drain here and
		     * then not get a select wakeup later.
		     *
		     * check for connection termination
		     */

		    if (conn) {
			if (conn->co_RMBuf.mh_REof && 
			    conn->co_TMBuf.mh_WEof &&
			    conn->co_TMBuf.mh_MBuf == NULL
			) {
			    DeleteConnection(conn);
			    DelThread(desc);
			}
		    }
		}
	    }
	}
    }
}

void
HandleReaderMsg(ForkDesc *desc)
{
    int r;
    DnsRes  dres;
    struct msghdr msg;
    struct iovec  iov = { (void *)&dres, sizeof(dres) };
    struct {
#if FDPASS_USES_CMSG
	struct cmsghdr cmsg;
#endif
	int fd;
    } cmsg;

    bzero(&msg, sizeof(msg));

    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    msg.msg_control = (caddr_t)&cmsg;
    msg.msg_controllen = sizeof(cmsg);
#if FDPASS_USES_CMSG
    msg.msg_flags = 0;
    cmsg.cmsg.cmsg_len = sizeof(cmsg);
#endif
    cmsg.fd = -1;
    errno = 0;

    printf("RECV MSG\n");

    if ((r = recvmsg(desc->d_Fd, &msg, MSG_EOR|MSG_WAITALL)) == sizeof(dres)) {
	if (cmsg.fd >= 0) {
	    ForkDesc *ndesc;
	    Connection *conn;

	    ndesc = AddThread("client", cmsg.fd, -1, THREAD_NNTP, NumReaders);
	    RTStatusBase(NumReaders, "%s", dres.dr_Host);
	    RTStatusUpdate(NumReaders, "nntp %s%s%s",
		((dres.dr_Flags & DF_FEED) ? "(feed)" : ""),
		((dres.dr_Flags & DF_READ) ? "(read)" : ""),
		((dres.dr_Flags & DF_POST) ? "(post)" : "")
	    );

	    ++NumReaders;
	    if (DebugOpt)
		printf("add thread fd=%d\n", cmsg.fd);
	    FD_SET(ndesc->d_Fd, &WFds);	/* will cause immediate effect */
	    conn = InitConnection(ndesc, &dres);
	    if (conn->co_Auth.dr_Flags & DF_FEED)
		conn->co_Flags |= COF_SERVER;
	    NNWriteHello(conn);
	} else {
	    if (DebugOpt)
		printf("recvmsg(): EOF1\n");
	    DelThread(desc);
	    TerminatePending = 1;
	}
    }
    if (r == 0) {
	if (DebugOpt)
	    printf("recvmsg(): EOF2\n");
	DelThread(desc);
	TerminatePending = 1;
    }
}

Connection *
InitConnection(ForkDesc *desc, DnsRes *dres)
{
    MemPool    *pool = NULL;
    Connection *conn = zalloc(&pool, sizeof(Connection));

    desc->d_Data = conn;

    if (dres)
	conn->co_Auth = *dres;

    conn->co_Desc = desc;
    conn->co_MemPool = pool;
    MBInit(&conn->co_TMBuf, desc->d_Fd, &conn->co_MemPool, &conn->co_BufPool);
    MBInit(&conn->co_RMBuf, desc->d_Fd, &conn->co_MemPool, &conn->co_BufPool);
    MBInit(&conn->co_ArtBuf, -1, &conn->co_MemPool, &conn->co_BufPool);

    return(conn);
}

void
DeleteConnection(Connection *conn)
{
    MemPool *mpool = conn->co_MemPool;
    char c = 0;

    if (conn->co_Desc->d_Type == THREAD_NNTP) {
	RTStatusUpdate(conn->co_Desc->d_Slot, "(closed)");
	--NumReaders;
    }

    write(TFd, &c, 1);

    FreeControl(conn);
    freePool(&conn->co_BufPool);
    freePool(&mpool);		/* includes Connection structure itself */
}

void
NNTerminate(Connection *conn)
{
    conn->co_Func = NNTerminate;
    conn->co_State = "term";
    conn->co_RMBuf.mh_REof = 1;
    conn->co_TMBuf.mh_WEof = 1;
    FD_SET(conn->co_Desc->d_Fd, &WFds);

    /*
     * conn->co_SReq is only non-NULL for a client
     * connection.  Server use of the field will have
     * already NULL'd it out in NNServerTerminate().
     *
     * The problem we have is that the server may be actively
     * using the client connection's MBuf's and be in some 
     * intermediate state.  Therefore, we must change the SReq
     * to point to a sink-NULL client XXX.
     */
    if (conn->co_SReq) {
	if (conn->co_SReq->sr_CConn != conn)
	    fatal("NNTerminate(): server conn had non_NULL co_SReq");
	/* 
	 * Disconnect the co_SReq from the client.  This will cause
	 * the server operation-in-progress to abort, if possible.
	 */
	conn->co_SReq->sr_CConn = NULL;
	conn->co_SReq = NULL;
    }
}

void
NNWriteHello(Connection *conn)
{
    const char *postingOk = "(no posting)";
    const char *noReading = "";
    const char *serverType = "NNTP";

    if (conn->co_Auth.dr_Flags & DF_POST)
	postingOk = "(posting ok)";
    if (conn->co_Flags & COF_SERVER) {
	serverType = "FEED";
    } else if ((conn->co_Auth.dr_Flags & DF_READ) == 0) {
	noReading = "(no reading)";
    }
    if ((conn->co_Auth.dr_Flags & (DF_FEED|DF_READ|DF_POST)) == 0) {
	MBPrintf(
	    &conn->co_TMBuf, 
	    "500 %s Diablo Server, you have no permissions.\r\n",
	    ReportedHostName
	);
	NNTerminate(conn);
	return;
    }

    MBPrintf(
	&conn->co_TMBuf, 
	"200 %s Diablo %s Server ready %s%s.\r\n",
	ReportedHostName,
	serverType,
	postingOk,
	noReading
    );
    NNCommand(conn);
}

/*
 * NNCommand() - general command entry.  Attempt to flush output data
 *		 and then go onto the appropriate command set.
 */

#define CMDF_AUTH	0x00000001
#define CMDF_SERVER	0x00000002
#define CMDF_READER	0x00000004

typedef struct Command {
    const char *cmd_Name;
    int		cmd_Flags;
    void	(*cmd_Func)(Connection *conn, char **pptr);
    const char	*cmd_Help;
} Command;

Command Cmds[] = {
    { 
	"article",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPArticle,
	"[MessageID|Number]"
    },
    { 
	"body",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,	
	NNTPBody,
	"[MessageID|Number]"
    },
    { 
	"date",		
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPDate,
	""
    },
    { 
	"head",	
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPHead,
	"[MessageID|Number]"
    },
    { 
	"help",		
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPHelp,
	""
    },
    { 
	"ihave",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPIHave,
	""
    },
    { 
	"takethis",	
	CMDF_AUTH|CMDF_SERVER|0		 ,
	NNTPTakeThis,
	"MessageID"
    },
    { 
	"check",
	CMDF_AUTH|CMDF_SERVER|0		 ,
	NNTPCheck,
	"MessageID"
    },
    { 
	"mode",
	CMDF_AUTH|CMDF_READER,
	NNTPMode,
	"reader|stream"
    },
    { 
	"mode",
	CMDF_AUTH|CMDF_SERVER,
	NNTPMode ,
	"reader|stream|headfeed"
    },
    { 
	"slave",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPSlave,
	""
    },
    { 
	"quit",
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPQuit,
	""
    },
    { 
	"group",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPGroup ,
	"newsgroup"
    },
    { 
	"last",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPLast,
	""
    },
    { 
	"next",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNext,
	""
    },
    { 
	"list",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPList,
	"[active|active.times|newsgroups|distributions|distrib.pats|moderators|overview.fmt|subscriptions]"
    },
    { 
	"listgroup",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPListGroup,
	"newsgroup"
    },
    { 
	"newgroups",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNewgroups,
	"yymmdd hhmmss [\"GMT\"] [<distributions>]"
    },
    { 
	"newnews",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNewNews,
	"(not implemented)"
    },
    { 
	"post",	
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPPost,
	""
    },
    { 
	"stat",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPStat,
	"[MessageID|Number]"
    },
    { 
	"xgtitle",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXGTitle,
	"[group_pattern]"
    },
    { 
	"xhdr",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXHdr,
	"header [range|MessageID]"
    },
    { 
	"xover",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXOver,
	"[range]"
    },
    { 
	"xpat",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXPat,
	"header range|MessageID pat"
    },
    { 
	"xpath",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXPath,
	"MessageID"
    },
    { 
	"authinfo",
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPAuthInfo,
	"user Name|pass Password"
    }
};

void 
NNCommand(Connection *conn)
{
    MBFlush(&conn->co_TMBuf);
    NNCommand2(conn);
}

void
NNCommand2(Connection *conn)
{
    char *ptr;
    char *cmd;
    char *buf;
    Command *scan;
    int len;

    conn->co_Func = NNCommand2;
    conn->co_State = "waitcmd";

    /*
     * we have to be careful in regards to recursive operation, nor do
     * we want one descriptor to hog the process.  We can't set RFds
     * because the next command may already be entirely loaded into an
     * MBuf so setting RFds may not unblock us.  Instead, we set WFds
     * which basically forces a wakeup at some point in the future.
     */

    if (conn->co_FCounter) {
	FD_SET(conn->co_Desc->d_Fd, &WFds);
	return;
    }
    ++conn->co_FCounter;

    /*
     * get command
     */

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) == 0)
	return;

    /*
     * check EOF
     */

    if (len < 0) {
	NNTerminate(conn);
	return;
    }

    /*
     * strip CR LF
     */

    ptr = buf;

    if (len > 1 && ptr[len-2] == '\r')
	ptr[len-2] = 0;

    if (DebugOpt)
	printf("command: %s\n", ptr);

    /*
     * extract command
     */

    if ((cmd = parseword(&ptr, " \t")) == NULL) {
	NNCommand(conn);
	return;
    }
    {
	int i;

	for (i = 0; cmd[i]; ++i)
	    cmd[i] = tolower((int)(unsigned char)cmd[i]);
    }

    /*
     * Locate and execute command
     */

    for (scan = &Cmds[0]; scan < &Cmds[arysize(Cmds)]; ++scan) {
	if (strcmp(cmd, scan->cmd_Name) == 0 && 
	    (((scan->cmd_Flags & CMDF_SERVER) && (conn->co_Flags & COF_SERVER)) ||
	    ((scan->cmd_Flags & CMDF_READER) && !(conn->co_Flags & COF_SERVER)))
	) {
	    break;
	}
    }
    if (scan < &Cmds[arysize(Cmds)]) {
	if ((scan->cmd_Flags & CMDF_AUTH) &&
	    (conn->co_Auth.dr_Flags & DF_AUTHREQUIRED)
	) {
	    MBPrintf(&conn->co_TMBuf, "480 Authentication required for command\r\n");
	    NNCommand(conn);
	} else {
	    scan->cmd_Func(conn, &ptr);
	}
    } else {
	NNUnknownCommand(conn);
    }
}

void
NNBadCommandUse(Connection *conn)
{
    MBPrintf(&conn->co_TMBuf, "501 Bad command use\r\n");
    NNCommand(conn);
}

void
NNUnknownCommand(Connection *conn)
{
    MBPrintf(&conn->co_TMBuf, "500 What?\r\n");
    NNCommand(conn);
}

void
NNWaitThread(Connection *conn)
{
    conn->co_Func = NNWaitThread;
    conn->co_State = "waitrt";
    FD_CLR(conn->co_Desc->d_Fd, &RFds);
    FD_CLR(conn->co_Desc->d_Fd, &WFds);
}

void
NNTPHelp(Connection *conn, char **pptr)
{
    Command *scan;

    MBPrintf(&conn->co_TMBuf, "100 Legal commands\r\n");
    for (scan = &Cmds[0]; scan < &Cmds[arysize(Cmds)]; ++scan) {
	if (((scan->cmd_Flags & CMDF_SERVER) && (conn->co_Flags & COF_SERVER)) ||
	    ((scan->cmd_Flags & CMDF_READER) && !(conn->co_Flags & COF_SERVER))
	) {
	    MBPrintf(&conn->co_TMBuf, "  %s %s\r\n", scan->cmd_Name, scan->cmd_Help);
	}
    }
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}


