/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include "xdir.h"
#include "gprefs.h"
#include "str.h"
#include "history.h"
#include "view.h"

#define MAXPLINE    100

char *anonymous_ftp_password;
int auto_local_login;
int beep_when_ops_done;
int connection_check_time;
int initial_cache_mode;
int initial_diagnostics;
int initial_dotfiles_mode;
int initial_layout;
int initial_match_case;
int initial_max_search_depth;
int initial_max_viewer_width;
int initial_search_direction;
int initial_search_range;
int initial_tunneling_mode;
int initial_wildcard_mode;
int initial_xfer_mode;
int inquire_on_copy;
int inquire_on_delete;
int inquire_on_move;
int inquire_on_quote;
int max_dir_cache_len;
int max_ftp_retries;
int max_history[NUMKINDS];
int max_history_hosts;
int maxt;
int multiclick_interval;
int overwrite_dirs;
int preserve_vms_ver_nums;
int print_password_in_debug;
int rcopy_mode;
int rdelete_mode;
int sort_caches;
int stop_button_blinking;
int store_unique;
int type_to_select_delay;
int use_last_dir;

extern char *help_anonymous_ftp_password[];
extern char *help_auto_local_login[];
extern char *help_beep_when_ops_done[];
extern char *help_connection_check_time[];
extern char *help_copy_mode[];
extern char *help_delete_mode[];
extern char *help_history_sort_order[];
extern char *help_initial_cache_mode[];
extern char *help_initial_dir_layout[];
extern char *help_initial_dotfiles_mode[];
extern char *help_initial_diagnostics[];
extern char *help_initial_max_viewer_width[];
extern char *help_initial_search_case_match[];
extern char *help_initial_search_depth[];
extern char *help_initial_search_direction[];
extern char *help_initial_search_name_match[];
extern char *help_initial_search_range[];
extern char *help_initial_transfer_mode[];
extern char *help_initial_tunneling_mode[];
extern char *help_initial_wildcard_mode[];
extern char *help_inquire_on_copy[];
extern char *help_inquire_on_delete[];
extern char *help_inquire_on_move[];
extern char *help_inquire_on_quoted_cmd[];
extern char *help_max_dir_cache_len[];
extern char *help_max_dir_name_history[];
extern char *help_max_ftp_retries[];
extern char *help_max_host_name_history[];
extern char *help_max_time_for_ftp_reply[];
extern char *help_max_user_name_history[];
extern char *help_max_wildcard_history[];
extern char *help_multiclick_interval[];
extern char *help_overwrite_dir_on_copy[];
extern char *help_preserve_vms_ver_nums[];
extern char *help_print_password_in_debug[];
extern char *help_max_quoted_cmd_history[];
extern char *help_stop_button_blinking[];
extern char *help_store_unique[];
extern char *help_symbols_on_dir_entries[];
extern char *help_type_to_select_delay[];
extern char *help_use_last_dir[];

static struct choice_st choice_prefs[] = {

	{
		"Beep_When_Ops_Done", &beep_when_ops_done, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Copy_Mode", &rcopy_mode, "Recursive",  2,
		{ { "Recursive", True }, { "Nonrecursive",  False } }
	},

	{
		"Delete_Mode", &rdelete_mode, "Nonrecursive",  2,
		{ { "Recursive", True }, { "Nonrecursive",  False } }
	},

	{
		"History_Sort_Order", &sort_caches, "By Last Use", 2,
		{ { "Alphanumeric", True }, { "By Last Use",  False } }
	},

	{
		"Initial_Cache_Mode", &initial_cache_mode, "Off", 2,
		{ { "On", True }, { "Off",  False } }
	},

	{
		"Initial_Diagnostics", &initial_diagnostics, "Normal", 4,
		{ { "Quiet",  QUIET  }, { "Verbose", VERBOSE },
		  { "Normal", NORMAL }, { "Debug",   DEBUG   } }
	},

	{
		"Initial_Directory_Layout", &initial_layout, "Iconic", 4,
		{ { "Tabular",       TABULAR      }, { "Tree",      TREE       },
		  { "Iconic",        ICONIC       }, { "Full Info", FULL_INFO  } }
	},

	{
		"Initial_Dot_Files_Mode", &initial_dotfiles_mode, "On", 2,
		{ { "On", True }, { "Off",  False } }
	},

	{
		"Initial_Search_Case_Match", &initial_match_case, "Ignore Case", 2,
		{ { "Ignore Case",  False }, { "Match Case", True } }
	},

	{
		"Initial_Search_Direction", &initial_search_direction, "Forward", 2,
		{ { "Forward",  FORWARD }, { "Backward", BACKWARD } }
	},

	{
		"Initial_Search_Range", &initial_search_range, "This Dir", 3,
		{ { "This Dir",  SEARCH_THIS_DIR  }, { "This Host", SEARCH_THIS_HOST },
		  { "All Hosts", SEARCH_ALL_HOSTS } }
	},

    {
        "Initial_Transfer_Mode", &initial_xfer_mode, "Binary", 2,
        { { "ASCII",  ASCII }, { "Binary", BINARY } }
    },

	{
		"Initial_Tunneling_Mode", &initial_tunneling_mode, "On", 2,
		{ { "On", True }, { "Off",  False } }
	},

	{
		"Initial_Wildcard_Mode", &initial_wildcard_mode, "Replace", 2,
		{ { "Replace",  REPLACE }, { "Add", ADD } }
	},

	{
		"Inquire_On_Copy", &inquire_on_copy, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Inquire_On_Delete", &inquire_on_delete, "Yes", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Inquire_On_Move", &inquire_on_move, "Yes", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Inquire_On_Quoted_Cmd", &inquire_on_quote, "Yes", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Overwrite_Dir_On_Copy", &overwrite_dirs, "Yes", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Preserve_Vms_Ver_Nums", &preserve_vms_ver_nums, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Print_Password_In_Debug", &print_password_in_debug, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Stop_Button_Blinking", &stop_button_blinking, "On", 2,
		{ { "On", True }, { "Off",  False } }
	},

	{
		"Store_Unique", &store_unique, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	},

	{
		"Use_Last_Dir_At_Login", &use_last_dir, "No", 2,
		{ { "Yes", True }, { "No",  False } }
	}
};

static int nchoice_prefs = XtNumber(choice_prefs);

static struct text_st text_prefs[] = {

	{ "Anonymous_Ftp_Password", &anonymous_ftp_password, "" }
};

static int ntext_prefs = XtNumber(text_prefs);

static struct number_st number_prefs[] = {

	{ "Connection_Check_Time", &connection_check_time, 5, 0, 30 },

	{ "Initial_Max_Search_Depth", &initial_max_search_depth, 10,  1,
	  MAXSEARCHDEPTH },

	{ "Initial_Max_Viewer_Width", &initial_max_viewer_width, 80,  20, 200 },

	{ "Max_Dir_Cache_Len_(KB)", &max_dir_cache_len, 25, 0, 100 },

	{ "Max_Dir_Name_History", &max_history[DIRECTORY], 15,  1, MAXDIRHISTORY },

	{ "Max_Ftp_Retries", &max_ftp_retries, 3, 0, 10 },

	{ "Max_Host_Name_History", &max_history_hosts, 10, 1, 30 },

	{ "Max_Quoted_Cmd_History", &max_history[QUOTE], 10,  1, 30 },

	{ "Max_Time_For_Ftp_Reply", &maxt, 60, 30, 300 },

	{ "Max_User_Name_History", &max_history[USER], 10,  1, 30 },

	{ "Max_Wildcard_History", &max_history[WILDCARD], 10,  1, 30 },

	{ "Multiclick_Interval", &multiclick_interval, 200,  100, 1000 },

	{ "Type_To_Select_Delay", &type_to_select_delay, 1000,  200, 2000 }
};

static int nnumber_prefs = XtNumber(number_prefs);

struct prefs_struct prefs[] = {

	{ "Anonymous_Ftp_Password", TEXT, help_anonymous_ftp_password },

	{ "Beep_When_Ops_Done", CHOICE, help_beep_when_ops_done },

	{ "Connection_Check_Time", NUMBER, help_connection_check_time },

	{ "Copy_Mode", CHOICE, help_copy_mode },

	{ "Delete_Mode", CHOICE, help_delete_mode },

	{ "History_Sort_Order", CHOICE, help_history_sort_order },

	{ "Initial_Cache_Mode", CHOICE, help_initial_cache_mode },

	{ "Initial_Diagnostics", CHOICE, help_initial_diagnostics },

	{ "Initial_Directory_Layout", CHOICE, help_initial_dir_layout },

	{ "Initial_Dot_Files_Mode", CHOICE, help_initial_dotfiles_mode },

	{ "Initial_Max_Search_Depth", NUMBER, help_initial_search_depth },

	{ "Initial_Max_Viewer_Width", NUMBER, help_initial_max_viewer_width },

	{ "Initial_Search_Case_Match", CHOICE, help_initial_search_case_match },

	{ "Initial_Search_Direction", CHOICE, help_initial_search_direction },

	{ "Initial_Search_Range", CHOICE, help_initial_search_range },

	{ "Initial_Transfer_Mode", CHOICE, help_initial_transfer_mode },

	{ "Initial_Tunneling_Mode", CHOICE, help_initial_tunneling_mode },

	{ "Initial_Wildcard_Mode", CHOICE, help_initial_wildcard_mode},

	{ "Inquire_On_Copy", CHOICE, help_inquire_on_copy },

	{ "Inquire_On_Delete", CHOICE, help_inquire_on_delete },

	{ "Inquire_On_Move", CHOICE, help_inquire_on_move },

	{ "Inquire_On_Quoted_Cmd", CHOICE, help_inquire_on_quoted_cmd },

	{ "Max_Dir_Cache_Len_(KB)", NUMBER, help_max_dir_cache_len },

	{ "Max_Dir_Name_History", NUMBER, help_max_dir_name_history },

	{ "Max_Ftp_Retries", NUMBER, help_max_ftp_retries },

	{ "Max_Host_Name_History", NUMBER, help_max_host_name_history },

	{ "Max_Quoted_Cmd_History", NUMBER, help_max_quoted_cmd_history },

	{ "Max_Time_For_Ftp_Reply", NUMBER, help_max_time_for_ftp_reply },

	{ "Max_User_Name_History", NUMBER, help_max_user_name_history },

	{ "Max_Wildcard_History", NUMBER, help_max_wildcard_history },

	{ "Multiclick_Interval", NUMBER, help_multiclick_interval },

	{ "Overwrite_Dir_On_Copy", CHOICE, help_overwrite_dir_on_copy },

	{ "Preserve_Vms_Ver_Nums", CHOICE, help_preserve_vms_ver_nums },

	{ "Print_Password_In_Debug", CHOICE, help_print_password_in_debug },

	{ "Stop_Button_Blinking", CHOICE, help_stop_button_blinking },

	{ "Store_Unique", CHOICE, help_store_unique },

	{ "Type_To_Select_Delay", NUMBER, help_type_to_select_delay },

	{ "Use_Last_Dir_At_Login", CHOICE, help_use_last_dir }
};

int nprefs = XtNumber(prefs);

static struct pref_category_item_st history_prefs[] = {
	{ "History_Sort_Order" },
	{ "Max_Dir_Name_History" },
	{ "Max_Host_Name_History" },
	{ "Max_User_Name_History" },
	{ "Max_Quoted_Cmd_History" },
	{ "Max_Wildcard_History" },
	{ "Initial_Cache_Mode" },
	{ "Max_Dir_Cache_Len_(KB)" }
};

static struct pref_category_item_st searching_prefs[] = {
	{ "Initial_Max_Search_Depth" },
	{ "Initial_Search_Case_Match" },
	{ "Initial_Search_Range" },
	{ "Initial_Search_Direction" }
};

static struct pref_category_item_st toolbar_prefs[] = {
	{ "Initial_Transfer_Mode" },
	{ "Initial_Directory_Layout" },
	{ "Initial_Tunneling_Mode" },
	{ "Initial_Dot_Files_Mode" },
	{ "Initial_Cache_Mode" }
};

static struct pref_category_item_st reporting_prefs[] = {
	{ "Initial_Diagnostics" },
	{ "Print_Password_In_Debug" },
	{ "Beep_When_Ops_Done" },
	{ "Inquire_On_Copy" },
	{ "Inquire_On_Delete" },
	{ "Inquire_On_Move" },
	{ "Inquire_On_Quoted_Cmd" },
	{ "Stop_Button_Blinking" }
};

static struct pref_category_item_st operations_prefs[] = {
	{ "Delete_Mode" },
	{ "Inquire_On_Delete" },
	{ "Inquire_On_Move" },
	{ "Initial_Wildcard_Mode" }
};

static struct pref_category_item_st miscellaneous_prefs[] = {
	{ "Max_Ftp_Retries" },
	{ "Max_Time_For_Ftp_Reply" },
	{ "Anonymous_Ftp_Password" },
	{ "Use_Last_Dir_At_Login" },
	{ "Type_To_Select_Delay" },
	{ "Initial_Max_Viewer_Width" },
	{ "Multiclick_Interval" },
	{ "Connection_Check_Time" }
};

static struct pref_category_item_st file_xfer_prefs[] = {
	{ "Inquire_On_Copy" },
	{ "Copy_Mode" },
	{ "Store_Unique" },
	{ "Preserve_Vms_Ver_Nums" },
	{ "Initial_Transfer_Mode" },
	{ "Overwrite_Dir_On_Copy" }
};

static struct pref_category_item_st all_prefs[] = {
	{ "Anonymous_Ftp_Password" },
	{ "Beep_When_Ops_Done" },
	{ "Copy_Mode" },
	{ "Delete_Mode" },
	{ "History_Sort_Order" },
	{ "Initial_Cache_Mode" },
	{ "Initial_Diagnostics" },
	{ "Initial_Directory_Layout" },
	{ "Initial_Dot_Files_Mode" },
	{ "Initial_Max_Search_Depth" },
	{ "Initial_Max_Viewer_Width" },
	{ "Initial_Search_Case_Match" },
	{ "Initial_Search_Direction" },
	{ "Initial_Search_Range" },
	{ "Initial_Transfer_Mode" },
	{ "Initial_Tunneling_Mode" },
	{ "Initial_Wildcard_Mode" },
	{ "Inquire_On_Copy" },
	{ "Inquire_On_Delete" },
	{ "Inquire_On_Move" },
	{ "Inquire_On_Quoted_Cmd" },
	{ "Max_Dir_Cache_Len_(KB)" },
	{ "Max_Dir_Name_History" },
	{ "Max_Ftp_Retries" },
	{ "Max_Host_Name_History" },
	{ "Max_Quoted_Cmd_History" },
	{ "Max_Time_For_Ftp_Reply" },
	{ "Max_User_Name_History" },
	{ "Max_Wildcard_History" },
	{ "Multiclick_Interval" },
	{ "Overwrite_Dir_On_Copy" },
	{ "Preserve_Vms_Ver_Nums" },
	{ "Print_Password_In_Debug" },
	{ "Connection_Check_Time" },
	{ "Stop_Button_Blinking" },
	{ "Store_Unique" },
	{ "Type_To_Select_Delay" },
	{ "Use_Last_Dir_At_Login" }
};

struct pref_category_st pref_categories[] = {
	{ "Toolbar Initialization", toolbar_prefs, XtNumber(toolbar_prefs) },
	{ "History & Cache", history_prefs, XtNumber(history_prefs) },
	{ "Searching", searching_prefs, XtNumber(searching_prefs) },
	{ "Reporting & Prompts", reporting_prefs, XtNumber(reporting_prefs) },
	{ "Operations", operations_prefs, XtNumber(operations_prefs) },
	{ "File Transfer", file_xfer_prefs, XtNumber(file_xfer_prefs) },
	{ "Miscellaneous", miscellaneous_prefs, XtNumber(miscellaneous_prefs) },
	{ "All", all_prefs, XtNumber(all_prefs) }
};

int npref_categories = XtNumber(pref_categories);


/*
 * initialize_gprefs - Initialize the general preferences data structures.
 */
initialize_gprefs()
{
	int i;
	int j;
	int k;

	/* Finish initializing prefs table */
	for (i=0; i<nprefs; i++) {
		switch (prefs[i].type) {
		case TEXT:
			for (j=0; j<ntext_prefs; j++)
				if (!strcmp(text_prefs[j].file_label, prefs[i].file_label)) {
					prefs[i].part.text = &text_prefs[j];
					break;
				}
			if (j == ntext_prefs)
				fatal_error("Programming bug in initialize_gprefs()");
			break;
		case CHOICE:
			for (j=0; j<nchoice_prefs; j++)
				if (!strcmp(choice_prefs[j].file_label, prefs[i].file_label)) {
					prefs[i].part.choice = &choice_prefs[j];
					break;
				}
			if (j == nchoice_prefs)
				fatal_error("Programming bug in initialize_gprefs()");
			break;
		case NUMBER:
			for (j=0; j<nnumber_prefs; j++)
				if (!strcmp(number_prefs[j].file_label, prefs[i].file_label)) {
					prefs[i].part.number = &number_prefs[j];
					break;
				}
			if (j == nnumber_prefs)
				fatal_error("Programming bug in initialize_gprefs()");
			break;
		}
	}

	/* Finish initializing the category tables */
	for (i=0; i<npref_categories; i++)
		for (j=0; j<pref_categories[i].nprefs; j++) {
			for (k=0; k<nprefs; k++)
				if (!strcmp(pref_categories[i].prefs[j].file_label,
						prefs[k].file_label)) {
					pref_categories[i].prefs[j].indx = k;
					break;
				}
			if (k == nprefs)
				fatal_error("Programming bug in initialize_gprefs()");
		}
}


/*
 * read_gprefs_from_file - Initializes current general preferences with values
 *                         from .xdirrc (which is pointed to by "fp").  If
 *                         "fp" is NULL, then use default values.  Returns
 *                         True if not all general preferences are in file,
 *                         else False.
 */
read_gprefs_from_file(fp)
FILE *fp;
{
	char prefs_line[MAXPLINE+1];
	char *q;
	int len;
	int i;
	int j;
	int position = 0;
	struct choice_st *c_st;
	struct number_st *n_st;
	int value;
	int new_prefs = False;

	/* Initialize the general preferences data structures. */
	initialize_gprefs();

	/* Set default values and unset preference positions */
	for (i=0; i<nprefs; i++) {
		prefs[i].position = -1;
		switch (prefs[i].type) {
		case TEXT:
			*prefs[i].part.text->value = 
				XtNewString(prefs[i].part.text->def_value);
			break;
		case CHOICE:
			c_st = prefs[i].part.choice;
			for (j=0; j<c_st->nchoices; j++)
				if (strcmp(c_st->def_value, c_st->choices[j].label) == 0) {
					*c_st->value = c_st->choices[j].value;
					break;
				}
			if (j == c_st->nchoices)
				fatal_error("Programming bug in read_gprefs_from_file()");
			break;
		case NUMBER:
			n_st = prefs[i].part.number;
			if (n_st->def_value < n_st->min || n_st->def_value > n_st->max)
				fatal_error("Bug in read_gprefs_from_file()");
			*n_st->value = n_st->def_value;
			break;
		default:
			fatal_error("Bug in read_gprefs_from_file()");
		}
	}

	/* Able to read from preferences file? */
	if (fp == NULL)
		goto positions;

	/* Process preferences */
	while (fgets(prefs_line, MAXPLINE+1, fp) != NULL) {
		len = strlen(prefs_line);
		if (prefs_line[len-1] != '\n')
			continue;
		if ((q=strtok(prefs_line, ":")) == NULL)
			continue;
		for (i=0; i<nprefs; i++)
			if (strcmp(q,prefs[i].file_label) == 0 && prefs[i].position == -1) {
				q += strlen(q)+1;
				while ((*q == ' ') || (*q == '\t'))
					q++;
				if ((q = strtok(q, "\n")) == NULL)
					break;
				switch (prefs[i].type) {
				case TEXT:
					XtFree(*prefs[i].part.text->value);
					*prefs[i].part.text->value = XtNewString(q);
					prefs[i].position = position++;
					break;
				case CHOICE:
					c_st = prefs[i].part.choice;
					for (j=0; j<c_st->nchoices; j++)
						if (strcmp(q, c_st->choices[j].label) == 0) {
							*c_st->value = c_st->choices[j].value;
							prefs[i].position = position++;
							break;
						}
					break;
				case NUMBER:
					n_st = prefs[i].part.number;
					sscanf(q, "%d", &value);
					if (value >= n_st->min && value <= n_st->max) {
						*n_st->value = value;
						prefs[i].position = position++;
					}
					break;
				default:
					fatal_error("Bug in read_gprefs_from_file()");
				}
				break;
			}
	}

positions:

	/* Establish the position of preferences not in file */
	for (i=0; i<nprefs; i++)
		if (prefs[i].position == -1) {
			new_prefs = True;
			prefs[i].position = position++;
		}

	return new_prefs;
}


/*
 * save_gprefs - Write current general preference values to .xdirrc using
 *               file pointer "fp".
 */
save_gprefs(fp)
FILE *fp;
{
	int i;
	int j;
	struct choice_st *c_st;

	for (i=0; i<nprefs; i++) {
		fprintf(fp, "%s: ", prefs[i].file_label);
		switch (prefs[i].type) {
		case TEXT:
			fprintf(fp, "%s\n", *prefs[i].part.text->value);
			break;
		case CHOICE:
			c_st = prefs[i].part.choice;
			for (j=0; j<c_st->nchoices; j++)
				if (*c_st->value == c_st->choices[j].value) {
					fprintf(fp, "%s\n", c_st->choices[j].label);
					break;
				}
            if (j == c_st->nchoices)
                fatal_error("Bug in save_gprefs() - Contact programmer\n");
			break;
		case NUMBER:
			fprintf(fp, "%d\n", *prefs[i].part.number->value);
			break;
		default:
			fatal_error("Bug in save_gprefs() - Contact programmer\n");
		}
	}
}

