#!/usr/bin/perl
# convert astorb.dat on stdin to asteroids.edb on stdout.
#
# actually, only those asteroids that might ever be brighter than dimmag are
# sent to stdout; the others are saved in asteroids_dim.edb.
#
# astorb is a set of elements for 30k+ asteroids maintained by Lowell
# Observatory. See ftp://ftp.lowell.edu/pub/elgb/astorb.html. From the
# Acknowledgments section of same:
#   The research and computing needed to generate astorb.dat were funded
#   principally by NASA grant NAGW-1470, and in part by the Lowell Observatory
#   endowment. astorb.dat may be freely used, copied, and transmitted provided
#   attribution to Dr. Edward Bowell and the aforementioned funding sources is
#   made. 
#
# Elwood Downey
#  2 Jul 1996 first draft
# 10 Oct 1996 update for field 18 change; add notion of astorb_dim.edb
#  1 Sep 1997 change name to asteroids*.edb for OCAAS

$ver = 1.1;

# setup mag and filename
$dimmag = 13;			# set to dimmest mag to be sent to stdout
$dimfn = "asteroids_dim.edb";	# name of file for dim asteroids

# build some common boilerplate
($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime;
$year += 1900;
$mon += 1;
$when = "# Processed $year-$mon-$mday $hour:$min:$sec UTC\n";
$from = "# From ftp://ftp.lowell.edu/pub/elgb/astorb.dat.gz\n";
$what = "# Generated by astorb2edb.pl version $ver by Elwood Downey\n";

# open the dim file
open (DIM, ">$dimfn") || die "Can not create $dimfn\n";

# add boilerplate to each file
print DIM "# Asteroids never brighter than $dimmag.\n";
print DIM $from;
print DIM $what;
print DIM $when;

print "# Asteroids ever brighter than $dimmag.\n";
print $from;
print $what;
print $when;

# process each astorb.dat entry
while (<STDIN>) {
    # build the name
    $nmbr = &s(1,4) + 0;
    $name = &s(6,23);
    $name =~ s/ *$//;

    # gather the params
    $i = &s(147,155) + 0;
    $O = &s(136,145) + 0;
    $o = &s(125,134) + 0;
    $a = &s(168,179) + 0;
    $e = &s(157,166) + 0;
    $M = &s(114,123) + 0;
    $H = &s(41,46) + 0;
    $G = &s(48,51) + 0;
    $TM = &s(109,110) + 0;
    $TD = &s(111,112) + 0;
    $TY = &s(105,108) + 0;

    # decide whether it's ever bright
    $per = $a*(1 - $e);
    $aph = $a*(1 + $e);
    if ($per < 1.1 && $aph > .9) {
	$fd = STDOUT;	# might be in the back yard some day :-)
    } else {
	$maxmag = $H + 5*&log10($per*&absv($per-1));
	$fd = $maxmag > $dimmag ? DIM : STDOUT;
    }

    # print
    print $fd "$nmbr " if ($nmbr != 0);
    print $fd "$name";
    print $fd ",e,$i,$O,$o,$a,0,$e,$M,$TM/$TD/$TY,2000.0,$H,$G\n";
}

# like substr($_,first,last), but one-based.
sub s {
    substr ($_, $_[0]-1, $_[1]-$_[0]+1);
}

# return log base 10
sub log10 {
    .43429*log($_[0]);
}

# return absolute value
sub absv {
    $_[0] < 0 ? -$_[0] : $_[0];
}
