/*
*   Create/Lock/Unlock mailbox files - Matt Ryan   Copyright (C) 1998
*
*     This program is free software, you can redistribute it and/or
*     modify it under the terms of the GNU General Public License
*     as published by the Free Software Foundation; either version
*     2 of the License, or (at your option) any later version.
*
*/

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <paths.h>
#include <pwd.h>
#include <lockfile.h>

typedef enum {False, True} boolean;

#define USAGE "Usage: %s [--create] [--lock] [--unlock]\n", progName

#define MBOX_MODE (S_IFREG|S_IRUSR|S_IWUSR)
#define MBOX_MODE_SPOOL (S_IFREG|S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP)
#define MBOX_SPOOL "/var/spool/mail/"	/* Must have trailing slash */

int main(int argc, char **argv)
{
	boolean create = False, lock = False, unlock = False;
	char progName[MAXPATHLEN + 1], temp[MAXPATHLEN + 1];
	boolean spooldir = True, statfail = False;
	char file[MAXPATHLEN + 1];
	struct passwd *pw;
	struct stat sb;
	mode_t mode;
	uid_t uid;
	gid_t gid;
	int fd;

	(void)strncpy(progName, basename(*argv), MAXPATHLEN);
	if(argc > 1) {
		while(*++argv) {
			if(strcmp(*argv, "--create") == 0) {
				create = True;
			}
			else if(strcmp(*argv, "--lock") == 0) {
				lock = True;
			}
			else if(strcmp(*argv, "--unlock") == 0) {
				unlock = True;
			}
			else if(strcmp(*argv, "--mbox") == 0) {
				if(*++argv) {
					(void)strncpy(temp, *argv, MAXPATHLEN);
				}
				else {
					(void)fprintf(stderr, USAGE); exit(1);
				}
			}
		}

		/* No options specified */
		if((create || lock || unlock) == False) {
			(void)fprintf(stderr, USAGE); exit(0);
		}
	}
	else {
		(void)fprintf(stderr, USAGE); exit(0);
	}

	uid = getuid();
	if((pw = getpwuid(uid)) == NULL) {
		(void)fprintf(stderr,
			"%s: Cannot get username for uid %d\n", progName, uid);
		exit(1);
	}

	(void)strncpy(file, MBOX_SPOOL, MAXPATHLEN);
	(void)strncat(file, pw->pw_name, MAXPATHLEN);

	/* Give away privs if not needed */
	if((strlen(temp) > 0) && strcmp(file, temp)) {
		if(setreuid(uid, uid) == -1) {
			(void)fprintf(stderr,
				"%s: %s\n", progName, strerror(errno));
			exit(1);
		}

		gid = getgid();
		if(setregid(gid, gid) == -1) {
			(void)fprintf(stderr,
				"%s: %s\n", progName, strerror(errno));
			exit(1);
		}
		(void)strncpy(file, temp, MAXPATHLEN);

		spooldir = False;
	}

	if(stat(file, &sb) == -1) {
		if(((errno == ENOENT) && create) == 0) {
			(void)fprintf(stderr,
				"%s: %s\n", progName, strerror(errno));
			exit(1);
		}
		statfail = True;
	}

	if(create) {
		mode = MBOX_MODE;
		if(spooldir) {
			mode = MBOX_MODE_SPOOL;
		}

		if(statfail) {
			(void)umask(0);
			if((fd = creat(file, mode)) == -1) {
				(void)fprintf(stderr,
					"%s: %s\n", progName, strerror(errno));
				exit(1);
			}
			close(fd);
		}
		else {
			/* If regular file, and we own it... */
			if((S_ISREG(sb.st_mode) == 0) && (sb.st_uid == uid)) {
				/* Fix file permissions (if needed) */
				if(sb.st_mode != mode) {
					(void)chmod(file, mode);
				}
			}
		}
	}

	/* Build the lockfile name */
	(void)strncpy(temp, file, MAXPATHLEN);
	(void)strncat(temp, ".lock", MAXPATHLEN);

	if(lock) {
		if(lockfile_create(temp, 5) != L_SUCCESS) {
			(void)fprintf(stderr,
				"%s: Cannot lock %s\n", progName, file);
			exit(1);
		}
	}

	if(unlock) {
		(void)lockfile_remove(temp);
	}

	return(0);
}

